/* lists.c

   Written by Frederic Bois
   22 June 2014

   Copyright (c) 2014 Frederic Bois.

   This code is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   See the GNU General Public License at <http://www.gnu.org/licenses/> 

   -- Revisions -----
     Logfile:  %F%
    Revision:  %I%
        Date:  %G%
     Modtime:  %U%
      Author:  @a
   -- SCCS  ---------

   A bunch of linked list utilities
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "lexerr.h"
#include "lists.h"


/* ----------------------------------------------------------------------------
   FreedList - for doubles

   Frees the memory allocated to a LIST, incrementing through each
   element sequentially.  Allows user to free data with a provided
   callback function, or to specify free'ing of data in this routine.

   If the pfvFreeData function is non-NULL, the function is called
   with each data pointer in the list.  bAndData is ignored.

   If the callback is NULL and the boolean bAndData is set, free() is
   called with each user data pointer before freeing the list element.

   NOTE: ppList is a pointer to a LIST structure which is itself a pointer.
*/
void FreedList (PLISTD *pplist)
{
  PLISTD plist = *pplist; // Get single pointer to work with

  if (!plist)
    return;

  while (plist->Head) {
    plist->Tail = plist->Head; // Save p to current elem
    plist->Head = plist->Head->next; // Next list item
    free (plist->Tail); // Free saved current elem

  } // while

  free (plist); // Free the list header
  *pplist = NULL; // Reset the user's pointer

} /* FreedList */


/* ----------------------------------------------------------------------------
   FreeiList - for integers

   Frees the memory allocated to a LIST, incrementing through each
   element sequentially.  Allows user to free data with a provided
   callback function, or to specify free'ing of data in this routine.

   If the pfvFreeData function is non-NULL, the function is called
   with each data pointer in the list.  bAndData is ignored.

   If the callback is NULL and the boolean bAndData is set, free() is
   called with each user data pointer before freeing the list element.

   NOTE: ppList is a pointer to a LIST structure which is itself a pointer.
*/
void FreeiList (PLISTI *pplist)
{
  PLISTI plist = *pplist; // Get single pointer to work with

  if (!plist)
    return;

  while (plist->Head) {
    plist->Tail = plist->Head; // Save p to current elem
    plist->Head = plist->Head->next; // Next list item
    free (plist->Tail); // Free saved current elem

  } // while

  free (plist); // Free the list header
  *pplist = NULL; // Reset the user's pointer

} /* FreeiList */


/* ----------------------------------------------------------------------------
   InitdList - for doubles

   Initialized a list header structure and returns the pointer.
*/
PLISTD InitdList (void)
{
  // Allocate header record
  PLISTD plist = (PLISTD) malloc (sizeof(LISTD));

  if (plist) {                // If allocation succeeds,
    plist->Head = NULL;       // .. initialize all to 0 
    plist->Tail = NULL;
    plist->OriginalHead = NULL;
    plist->lSize = 0;
  }
  else
    lexerr ("out of memory in InitdList");

  return (plist);

} /* InitdList */


/* ----------------------------------------------------------------------------
   InitiList - for integers

   Initialized a list header structure and returns the pointer.
*/
PLISTI InitiList (void)
{
  // Allocate header record
  PLISTI plist = (PLISTI) malloc (sizeof(LISTI));

  if (plist) {                // If allocation succeeds,
    plist->Head = NULL;       // .. initialize all to 0 
    plist->Tail = NULL;
    plist->OriginalHead = NULL;
    plist->lSize = 0;
  }
  else
    lexerr ("out of memory in InitiList");

  return (plist);

} /* InitiList */


/* ----------------------------------------------------------------------------
   PrintdList

   Print a list of doubles.
*/
void PrintdList (FILE *pFile, PLISTELEMD pfrom, PLISTELEMD pto)
{
  int iTotal = 0;
  PLISTELEMD ple;

  ple = pfrom;
  if (!ple) fprintf (pFile, "empty list.\n\n");
  while (ple != pto) {
    fprintf(pFile, "list node %ld, dVal %g\n", (long) ple, ple->dVal);
    iTotal += 1;
    ple = ple->next; // increment to next elem 
  }
  fprintf (pFile, "N elements = %d\n\n", iTotal);

} /* PrintdList */


/* ----------------------------------------------------------------------------
   PrintiList

   Print a list of integers.
*/
void PrintiList (FILE *pFile, PLISTELEMI pfrom, PLISTELEMI pto)
{
  int iTotal = 0;
  PLISTELEMI ple;

  ple = pfrom;
  if (!ple) fprintf (pFile, "empty list.\n\n");
  while (ple != pto) {
    fprintf(pFile, "list node %ld, ID %d\n", (long) ple, ple->iVal);
    iTotal += 1;
    ple = ple->next; // increment to next elem 
  }
  fprintf (pFile, "N elements = %d\n\n", iTotal);

} /* PrintiList */


/* ----------------------------------------------------------------------------
   QueuedListItem

   Adds a new list item (of type double) to the tail of the list.
*/
void QueuedListItem (PLISTD plist, double dVal)
{
  PLISTELEMD pNewElem;

  if (!plist)
    return;

  if (!(pNewElem = (PLISTELEMD) malloc (sizeof(LISTELEMD))))
    lexerr ("out of memory in QueuedListItem");
  
  pNewElem->dVal = dVal; // init new element
  pNewElem->next = NULL;

  if (plist->Tail)
    plist->Tail->next = pNewElem; // link new element
  else
    plist->Head = pNewElem;       // link first element

  plist->Tail = pNewElem;         // reset tail to new elem
  plist->lSize++;                 // increment size of list

} /* QueuedListItem */


/* ----------------------------------------------------------------------------
   QueueiListItem

   Adds a new list item (of type int) to the tail of the list.
*/
void QueueiListItem (PLISTI plist, int iVal)
{
  PLISTELEMI pNewElem;

  if (!plist)
    return;

  if (!(pNewElem = (PLISTELEMI) malloc (sizeof(LISTELEMI))))
    lexerr ("out of memory in QueueiListItem");
  
  pNewElem->iVal = iVal; // init new element
  pNewElem->next = NULL;

  if (plist->Tail)
    plist->Tail->next = pNewElem; // link new element
  else
    plist->Head = pNewElem;       // link first element

  plist->Tail = pNewElem;         // reset tail to new elem
  plist->lSize++;                 // increment size of list

} /* QueueiListItem */


/* ----------------------------------------------------------------------------
   QueuelListItem

   Adds a new list item (of type long) to the tail of the list.
*/
void QueuelListItem (PLISTL plist, long lVal)
{
  PLISTELEML pNewElem;

  if (!plist)
    return;

  if (!(pNewElem = (PLISTELEML) malloc (sizeof(LISTELEML))))
    lexerr ("out of memory in QueuelListItem");
  
  pNewElem->lVal = lVal; // init new element
  pNewElem->next = NULL;

  if (plist->Tail)
    plist->Tail->next = pNewElem; // link new element
  else
    plist->Head = pNewElem;       // link first element

  plist->Tail = pNewElem;         // reset tail to new elem
  plist->lSize++;                 // increment size of list

} /* QueuelListItem */


/* end */
