/* graph_sampler grammar file 

   Written by Frederic Bois
   22 June 2014

   Copyright (c) 2014 Frederic Bois.

   This code is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
   For more details
   see the GNU General Public License at <http://www.gnu.org/licenses/> 
*/

/* == Prologue ============================================================= */
%{
 // includes
 #include <stdio.h>
 #include <stdlib.h>
 #include <string.h>
 #include <math.h>
 #include "graph_sampler.h"  /* includes global variables to pass to main */
 #include "lexerr.h"
 #include "matrices.h"
 #include "lists.h"

 // prototypes
 int  yylex(void);
 void yyerror(const char *message);

 // local variables
 double     sym[26];   // 26 letters ...
 double*    pmat = 0x0;
 PLISTD     plist = NULL;
 int        count;
 int        dim1, dim2;
 int        i, j;
 PLISTELEMD ple;

%}

/* redefine yylval to extend it */
%union {
  double dval;
  int    ival;
  int    ivarnum;
  double *parray;
}

/* Specific declarations */
%token <ivarnum> alpha_motif_id
%token <ivarnum> autocycle_id
%token <ivarnum> bBN_id
%token <ivarnum> bDBN_id
%token <ivarnum> bDirichlet_id
%token <ivarnum> bPriorConcordance_id
%token <ivarnum> bPriorDegreeNode_id
%token <ivarnum> bPriorMotif_id
%token <ivarnum> bZellner_id
%token <ivarnum> beta_motif_id
%token <ivarnum> bsave_best_graph_id
%token <ivarnum> bsave_the_chain_id
%token <ivarnum> bsave_the_degree_counts_id
%token <ivarnum> bsave_the_edge_probabilies_id
%token <ivarnum> bsave_the_motifs_probabilies_id
%token <ivarnum> gamma_degree_id
%token <ivarnum> lambda_concord_id
%token <ivarnum> nBurnin_id
%token <ivarnum> nData_id
%token <ivarnum> nLevels_id
%token <ivarnum> nNodes_id
%token <ivarnum> nRuns_id
%token <ivarnum> n_saved_adjacency_id
%token <ivarnum> seed_id

%token <ivarnum> Data_id
%token <ivarnum> edge_requirements_id
%token <ivarnum> hyper_pB_id
%token <ivarnum> initial_adjacency_id

/* Non-specific declarations */
%token <ival>    INTEGER
%token <dval>    FLOAT
%token <ivarnum> IDENTIFIER
%token <ivarnum> L_PARENTHESIS
%token <ivarnum> R_PARENTHESIS
%token <ivarnum> L_CBRACE
%token <ivarnum> R_CBRACE
%token <ivarnum> ARRAY
%token <ivarnum> MATRIX
%token <ivarnum> EMPTY
%token <ivarnum> FULL
%token <ivarnum> RANDOM

%type <dval>   expr
%type <parray> list

%left '-' '+'
%left '*' '/'
%nonassoc UMINUS


/* == Grammar rules ======================================================== */
%% 

program:
 program statement ';'
 | program matrix_declaration ';'
 | program array_declaration ';'
 | program ';'
 |
;

statement:
 expr // { printf("%g\n", $1); }

 | IDENTIFIER '=' expr { sym[$1] = $3; }

 | autocycle_id '=' expr {
   if (($3 == 0) || ($3 == 1)) autocycle = $3; 
   else yyerror("autocycle should be 0 or 1"); }

 | bBN_id '=' expr { 
   if (($3 == 0) || ($3 == 1)) bBN = $3; 
   else yyerror("bBN should be 0 or 1"); }

 | bDBN_id '=' expr { 
   if (($3 == 0) || ($3 == 1)) bDBN = $3; 
   else yyerror("bDBN should be 0 or 1"); }

 | bsave_the_chain_id '=' expr {
   if (($3 == 0) || ($3 == 1)) bsave_the_chain = $3; 
   else yyerror("bsave_the_chain should be 0 or 1"); }

 | bsave_best_graph_id '=' expr { 
   if (($3 == 0) || ($3 == 1)) bsave_best_graph = $3; 
   else yyerror("bsave_best_graph should be 0 or 1"); }

 | bsave_the_edge_probabilies_id '=' expr {
   if (($3 == 0) || ($3 == 1)) bsave_the_edge_probabilies = $3; 
   else yyerror("bsave_the_edge_probabilies should be 0 or 1"); }

 | bsave_the_degree_counts_id '=' expr {
   if (($3 == 0) || ($3 == 1)) bsave_the_degree_counts = $3; 
   else yyerror("bsave_the_degree_counts should be 0 or 1"); }

 | bsave_the_motifs_probabilies_id '=' expr {
   if (($3 == 0) || ($3 == 1)) bsave_the_motifs_probabilies = $3; 
   else yyerror("bsave_the_motifs_probabilies should be 0 or 1"); }

 | bDirichlet_id '=' expr {
   if (($3 == 0) || ($3 == 1)) bDirichlet = $3; 
   else yyerror("bDirichlet should be 0 or 1"); }

 | bZellner_id '=' expr {
   if (($3 == 0) || ($3 == 1)) bZellner = $3; 
   else yyerror("bZellner should be 0 or 1"); }

 | n_saved_adjacency_id '=' expr { n_saved_adjacency = $3; }
 | nRuns_id             '=' expr { nRuns             = $3; }
 | nBurnin_id           '=' expr { nBurnin           = $3; }
 | seed_id              '=' expr { seed              = $3; }
 | nNodes_id            '=' expr { 
   if (nNodes)  {lexerr("nNodes cannot be reassigned"); }
   if ($3 == 0) {lexerr("nNodes cannot be zero");       }
   nNodes            = $3; }

 | bPriorConcordance_id '=' expr {
   if (($3 == 0) || ($3 == 1)) bPriorConcordance = $3; 
   else yyerror("bPriorConcordance should be 0 or 1"); }

 | lambda_concord_id '=' expr { lambda_concord = $3; }

 | bPriorDegreeNode_id '=' expr {
   if (($3 == 0) || ($3 == 1)) bPriorDegreeNode = $3; 
   else yyerror("bPriorDegreeNode should be 0 or 1"); }

 | gamma_degree_id '=' expr { gamma_degree = $3; }

 | bPriorMotif_id '=' expr {    
   if (($3 == 0) || ($3 == 1)) bPriorMotif = $3; 
   else yyerror("bPriorMotif should be 0 or 1"); }

 | alpha_motif_id '=' expr { alpha_motif = $3; }
 | beta_motif_id  '=' expr { beta_motif  = $3; }

 | nData_id '=' expr { nData = $3; }

;

matrix_declaration: // syntax: matrix{list}
 initial_adjacency_id '=' MATRIX L_CBRACE list R_CBRACE { 
   if (!nNodes)
    lexerr("nNodes must be set before defining matrices"); 

   if (plist->lSize != (nNodes * nNodes)) // problem
     lexerr("initial adjacency should have nNodes elements");

   if (!current_adj) 
    current_adj = InitiMatrix(nNodes, nNodes);
   else 
    lexerr("initial adjacency redefinition is not allowed");

   ple = plist->Head;
   for (i = 0; i < nNodes; i++) {
     for (j = 0; j < nNodes; j++) {
       if ((ple->dVal != 0) && (ple->dVal != 1))
         lexerr("adjacency matrix elements can only be 0 or 1");
       current_adj[i][j] = (int) ple->dVal;
       ple = ple->next;
     }
   }
 }

 | initial_adjacency_id '=' MATRIX L_CBRACE EMPTY R_CBRACE { // empty graph
   if (!nNodes)
    lexerr("nNodes must be set before defining matrices"); 

   if (!current_adj) 
    current_adj = InitiMatrix(nNodes, nNodes);
   else
    lexerr("initial_adjacency redefinition is not allowed");

   for (i = 0; i < nNodes; i++)
     for (j = 0; j < nNodes; j++)
       current_adj[i][j] = 0;
 }

 | initial_adjacency_id '=' MATRIX L_CBRACE FULL R_CBRACE { // full graph
   if (!nNodes)
     lexerr("nNodes must be set before defining matrices"); 

   if (!current_adj) 
     current_adj = InitiMatrix(nNodes, nNodes);
   else
     lexerr("initial_adjacency redefinition is not allowed");

   for (i = 0; i < nNodes; i++) {
     for (j = 0; j < nNodes; j++) {
       if ((i == j) && (bBN))
         current_adj[i][j] = 0;
       else
         current_adj[i][j] = 1;
     }
   }
 }
 
 | initial_adjacency_id '=' MATRIX L_CBRACE RANDOM R_CBRACE { // random EE graph
   if (!nNodes)
    lexerr("nNodes must be set before defining matrices"); 

   if (!seed)
    lexerr("seed must be set before defining a random matrix"); 
   else SetSeed(seed);

   if (!current_adj) 
    current_adj = InitiMatrix(nNodes, nNodes);
   else
    lexerr("initial_adjacency redefinition is not allowed");

   for (i = 0; i < nNodes; i++) {
     for (j = 0; j < nNodes; j++) {
       if ((i == j) && (bBN))
         current_adj[i][j] = 0;
       else
         current_adj[i][j] = round(Randoms());
     }
   }
   // PrintiMatrix (stdout, nNodes, current_adj); exit(0);
 }
 
 | edge_requirements_id '=' MATRIX L_CBRACE list R_CBRACE { 
   if (!nNodes)
    lexerr("nNodes must be set before defining matrices"); 

   if (plist->lSize !=  (nNodes * nNodes)) // problem
    lexerr("edge_requirements should have nNodes elements");

   if (!edge_requirements) 
    edge_requirements = InitiMatrix(nNodes, nNodes);
   else 
    lexerr("edge_requirements redefinition is not allowed");

   ple = plist->Head;
   for (i = 0; i < nNodes; i++) {
     for (j = 0; j < nNodes; j++) {
       if ((ple->dVal != 0) && (fabs(ple->dVal) != 1))
         lexerr("edge_requirements elements can only be -1, 0 or 1");
       edge_requirements[i][j] = (int) ple->dVal;
       ple = ple->next;
     }
   }
 } // end edge_requirements

 | hyper_pB_id '=' MATRIX L_CBRACE list R_CBRACE { 
   if (!nNodes)
    lexerr("nNodes must be set before defining matrices"); 

   if (plist->lSize != (nNodes * nNodes)) // problem
    lexerr("hyper_pB should have nNodes elements");

   if (!hyper_pB)
    hyper_pB = InitdMatrix(nNodes, nNodes); 
   else 
    lexerr("hyper_pB redefinition is not allowed");

   ple = plist->Head;
   for (i = 0; i < nNodes; i++) {
     for (j = 0; j < nNodes; j++) {
       if ((ple->dVal < 0) || (ple->dVal > 1))
         lexerr("Bernoulli prior probabilities have to be in [0,1]");
       hyper_pB[i][j] = ple->dVal;
       ple = ple->next;
     }
   }
 } // end hyper_pB

 | Data_id '=' MATRIX L_CBRACE list R_CBRACE { 
   if (!nNodes) 
    lexerr("nNodes must be set before defining matrices"); 

   if (!nData)
    lexerr("nData must be set before defining data values"); 

   if (plist->lSize != (nNodes * nData)) // problem
    lexerr("Data should have nNodes * nData elements");

   if (!pData)
    pData = InitdMatrix(nNodes, nData); 
   else 
    lexerr("Data redefinition is not allowed");

   ple = plist->Head;
   for (i = 0; i < nNodes; i++) {
     for (j = 0; j < nData; j++) {
       pData[i][j] = ple->dVal;
       ple = ple->next;
     }
   }
   // printf("Data defined.\n");
 } // end Data matrix
;

array_declaration: // syntax: array{list}
 nLevels_id '=' ARRAY L_CBRACE list R_CBRACE { 
   if (!nNodes)
    lexerr("nNodes must be set before defining arrays"); 

   if (plist->lSize != (nNodes)) // problem
     lexerr("data levels array should have nNodes elements");

   if (!pDataLevels) 
    pDataLevels = InitiVector(nNodes);
   else 
    lexerr("data levels redefinition is not allowed");

   ple = plist->Head;
   for (i = 0; i < nNodes; i++) {
     if ((ple->dVal != (int) ple->dVal) || (ple->dVal < 1))
       lexerr("data levels elements must be strictly positive integers");
     pDataLevels[i] = (int) ple->dVal;
     ple = ple->next;
   }
 }
;

list:
 expr { 
   plist = InitdList();
   QueuedListItem (plist, $1);
 }
 | list ',' expr { 
   QueuedListItem (plist, $3);
 }
 | { yyerror("empty list"); }
;

expr:
 INTEGER         { $$ = (int) $1; }
 | FLOAT         { $$ = $1; }
 | IDENTIFIER    { $$ = sym[$1]; }
 | expr '+' expr { $$ = $1 + $3; }
 | expr '-' expr { $$ = $1 - $3; }
 | expr '*' expr { $$ = $1 * $3; }
 | expr '/' expr { if ($3 == 0) { yyerror("division by zero"); } 
                   else $$ = $1 / (double) $3; }
 | '-' expr %prec UMINUS { $$ = -$2; }
 | L_PARENTHESIS expr R_PARENTHESIS  { $$ = $2; }
;


/* == Epilogue ============================================================= */
%%

void lexerr(const char *message)
{
  fprintf(stderr, "Error: %s - Exiting.\n\n", message); 
  exit(0);
}

/* End */
