#include <matlabint.h>
#include <matlabint_workspace.h>
#include <algorithm>

using namespace matlabint;

  class match_workspace {
    id_type wid;
  public:
    match_workspace(int _wid) { wid = _wid; }
    bool operator()(const getfem_object *o) {
      return o->get_workspace() == wid;
    }
  };

static void
do_stat(id_type wid) {

  const workspace_stack::obj_ct obj = workspace().get_obj_list();
  const workspace_stack::wrk_ct wrk = workspace().get_wrk_list();
  int cnt = 0;

  if (wid == workspace_stack::anonymous_workspace) {
    mexPrintf("Anonymous workspace (objects waiting for deletion)\n");
  } else {
    if (!wrk.index()[wid]) THROW_INTERNAL_ERROR;
    mexPrintf("Workspace %d [%s] -- %d objects\n", 
	      wid, wrk[wid].get_name().c_str(),
	      std::count_if(obj.tas_begin(), obj.tas_end(), match_workspace(wid)));
  }
  for (workspace_stack::obj_ct::const_tas_iterator it = obj.tas_begin(); it != obj.tas_end(); ++it, ++cnt) {
    if (!obj.index()[it.index()]) THROW_INTERNAL_ERROR;
    if (match_workspace(wid)(*it)) {
      mexPrintf(" ID%04d    %10s    %8d bytes   ", (*it)->get_id(), (*it)->type_name().c_str(), (*it)->memsize());
      const std::vector<id_type>& used_by = (*it)->get_used_by();
      if (used_by.size()) {
	mexPrintf("used by ");
	for (size_type i=0; i < used_by.size(); ++i) mexPrintf("ID%04d ", used_by[i]);
      }
      //      if ((*it)->is_anonymous()) mexPrintf("[ano]");
      mexPrintf("\n");
    }
  }
}

static void
do_stats() {
  dal::bit_vector bv = workspace().get_wrk_list().index();
  size_type wid;
  if (std::count_if(workspace().get_obj_list().tas_begin(), 
		    workspace().get_obj_list().tas_end(), 
		    match_workspace(workspace_stack::anonymous_workspace))) {
    do_stat(workspace_stack::anonymous_workspace);
  }
  for (wid << bv; wid != size_type(-1); wid << bv)
    do_stat(wid);
}

/*MLABCOM
  FUNCTION gf_workspace(operation)
    Getfem workspace management function. 

    Getfem uses its own workspaces in Matlab, independant of the
    matlab workspaces. By default, all getfem variables belong to the
    root getfem workspace. A function can create its own workspace by
    invoking gf_workspace("push") at its beginning. When exiting, this
    function MUST invoke gf_workspace("pop") (you can use matlab
    exceptions handling to do this cleanly when the function exits on
    an error).

    USAGE:
    * gf_workspace('push') 
    Creates a new temporary workspace on the workspace stack.

    * gf_workspace('pop' [,i,j,..])  
    Leaves the current workspace, destroying all getfem variables
    belonging to it, except the one listed after 'pop', and the ones
    moved to parent workspace by gf_workspace('keep').

    * gf_workspace('stat') : print infos about variables in current 
    workspace.

    * gf_workspace('stats') : print infos about all getfem variables.
    by gf_new_poly.

    * gf_workspace('keep', i[,j,k..]) 
    
    prevents the listed variables i from being deleted when
    gf_workspace("pop") will be called by moving this variable in the
    parent workspace.

    * gf_workspace('clear')
    Clears the current workspace.

    * gf_workspace('clear all') 
    Clear every workspace, and returns to the main workspace (you
    should not need to use this command).

 MLABCOM*/


void gf_workspace(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 1) {
    mexPrintf("Wrong number of input arguments");
  }
  mexarg_in arg = in.pop();
  std::string cmd = arg.to_string();

  if (check_cmd(cmd, "push", in, out, 0, 1, 0, 0)) {
    std::string s = "unnamed";
    if (!in.remaining() == 0) s = in.pop().to_string();
    workspace().push_workspace(s);
  } else if (check_cmd(cmd, "pop", in, out, 0, 256, 0, 0)) {
    while (in.remaining()) {
      workspace().send_object_to_parent_workspace(in.pop().to_object_id());
    }    
    workspace().pop_workspace();
  } else if (check_cmd(cmd, "stat", in, out, 0, 0, 0, 0)) {
    do_stat(workspace().get_current_workspace());
    mexPrintf("\n");
  } else if (check_cmd(cmd, "stats", in, out, 0, 0, 0, 0)) {
    do_stats();
    mexPrintf("\n");
  } else if (check_cmd(cmd, "keep", in, out, 1, 256, 0, 0)) {
    while (in.remaining()) {
      workspace().send_object_to_parent_workspace(in.pop().to_object_id());
    }
  } else if (check_cmd(cmd, "keep all", in, out, 0, 0, 0, 0)) {
    workspace().send_all_objects_to_parent_workspace();
  } else if (check_cmd(cmd, "clear", in, out, 0, 0, 0, 0)) {
    workspace().clear_workspace();
  } else if (check_cmd(cmd, "clear all", in, out, 0, 0, 0, 0)) {
    while (workspace().get_current_workspace() != workspace().get_base_workspace()) {
      workspace().pop_workspace();
      //      mexPrintf("w <- %d\n", workspace().get_current_workspace());
    }
    workspace().clear_workspace();
  } else bad_cmd(cmd);
}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  catch_errors(nlhs, plhs, nrhs, prhs, gf_workspace);
}
