function h=gf_plot_mesh(M, varargin)
% function h=gf_plot_mesh(M, [,'vertices'][,'convexes'][,'dofs']
%                     [,'cvlst',CVLST] ['boundaries'[BLST]])
%   General mesh plotting function.
%  
%   H=gf_plot_mesh(M) displays a mesh.
%
%   * gf_plot_mesh(M, 'vertices') displays also vertices numbers. 
%   * gf_plot_mesh(M, 'convexes') displays also convexes numbers.
%   * gf_plot_mesh(M, 'dofs') displays also finite element nodes, 
%   * gf_plot_mesh(M, 'boundaries',BLST) shows the boundaries listed in 
%   BLST
%
%   CAUTION:
%     For 'boundaries' and 'dofs', M should be a mesh_fem identifier, 
%   not a simple mesh object.
%
%   You can also specify an optionnal convex list CVLST to restrict
%   the drawing to the elements listed.
%  
%   $Id: gf_plot_mesh.m,v 1.3 2002/09/05 14:48:34 pommier Exp $


%   A. Huard, Y. Renard, J. Pommier
if nargin<1,
  error('Too few input arguments')
end

% parse argument list
k=1;
show_vertices_num = 0;
show_cv_num = 0;
show_dof_num = 0;
show_boundaries = 0;
cvlst=[];
while (k<=length(varargin))
  switch varargin{k}
   case 'vertices'
    show_vertices_num = 1;
   case 'convexes'
    show_cv_num = 1;
   case 'dofs'
    show_dof_num = 1;
   case 'cvlst'
    if (k<length(varargin)), cvlst=varargin{k+1}; k=k+1;
    else, error('missing value for "cvlst" argument'); 
    end
   case 'boundaries'
    show_boundaries = 1;    
    if k<length(varargin) & ~isstr(varargin{k+1}),
      k=k+1;
      blist=varargin{k};
    else
      blist=gf_mesh_fem_get(M, 'boundaries');
    end
   otherwise
    error('wrong argument');
  end
  k=k+1;
end
if (cvlst == []), cvlst = gf_mesh_get(M,'cvid'); end;


PXY = gf_mesh_get(M, 'pts coords');
all_PID = gf_mesh_get(M, 'pid');
E = gf_mesh_get(M, 'edges', cvlst);

% get the viewable pts id
PID = all_PID(union(E(1,:),E(2,:)));

dim = size(PXY,1);
if (dim > 3), error('sorry, only mesh of dimension <= 3 allowed'); end;
nbpts = size(PXY,2);
nbe = size(E,2);

Bmax  = max(PXY')';
Bmin  = min(PXY')';
Bdiff = Bmax - Bmin;
Bdiff = Bdiff + (Bdiff == 0); % remplace 0.0 par 1.0

ecart=Bdiff/150;


X = [PXY(1,E(1,:)); PXY(1,E(2,:))];

if (dim == 1),
  Y = zeros(size(X));
  ecart = [ecart; 0.05];
  PXY = [PXY; zeros(size(PXY))];
elseif (dim >= 2),
  Y = [PXY(2,E(1,:)); PXY(2,E(2,:))];
  if (dim == 3),
    Z = [PXY(3,E(1,:)); PXY(3,E(2,:))];
  end;
end;

if show_cv_num,
  cv_center = zeros(max(dim,2),numel(cvlst));
  % find convexes centers
  [cv_idx, cv_pid] = gf_mesh_get(M, 'pid from cvid',cvlst);
  for i=1:length(cvlst),
    cv_center(:,i) = mean(PXY(:, cv_pid(cv_idx(i):cv_idx(i+1)-1)),2);
  end;
end;

if show_dof_num,
  [dofpos, dofid] = gf_mesh_fem_get(M, 'dofs nodes', cvlst);
  if (dim == 1) dofpos = [dofpos; zeros(size(dofpos))]; end;
end;

if (show_boundaries),
  for bnum=1:length(blist),
    cvf = gf_mesh_fem_get(M, 'faces on boundary', blist(bnum));
    
    bid = gf_mesh_get(M, 'edges', cvf, 'merge convex');
    if (bnum == 8) disp(bid); end;

    bedge{bnum} = zeros(2, size(bid,2), dim);
    for i=1:max(dim,2),
      bedge{bnum}(:,:,i) = [PXY(i,bid(1,:)); PXY(i,bid(2,:))];
    end;
  end;
end;

is_hold = ishold;

if (dim <= 2),
  h = plot(X, Y, 'b-+');
  if show_boundaries,
    for bnum=1:length(blist),
%      disp(bedge{bnum}(:,:,:));
      hold on;
      h = [h; plot(bedge{bnum}(:,:,1), bedge{bnum}(:,:,2), 'r-o')];
    end;
  end
  if show_vertices_num,
    text(PXY(1,PID)+ecart(1), PXY(2,PID)+ecart(2), num2str(PID'),...
	 'HorizontalAlignment','center','VerticalAlignment','middle');
  end;
  if show_cv_num,
    text(cv_center(1,:), cv_center(2,:), num2str(cvlst'), ...
	 'HorizontalAlignment','center','VerticalAlignment','middle',...
	 'Color', [.7 0 0]);
  end;
  if show_dof_num,
    text(dofpos(1,:)-ecart(1), dofpos(2,:)-ecart(2), num2str(dofid'),...
	 'HorizontalAlignment','center','VerticalAlignment','middle',...
	 'Color', [0 .4 0]);
  end;
  if (dim == 2)
    axis([Bmin(1)-3*ecart(1) Bmax(1)+3*ecart(1) Bmin(2)-3*ecart(2) Bmax(2)+3*ecart(2)]);
  end
else
  h = plot3(X, Y, Z, 'b-+');
  if show_boundaries,
    for bnum=1:length(blist),
%      disp(bedge{bnum}(:,:,:));
      hold on;
      h = [h; plot3(bedge{bnum}(:,:,1), bedge{bnum}(:,:,2), bedge{bnum}(:,:,3), 'r-o')];
    end;
  end
  if show_vertices_num,
    text(PXY(1,PID)+ecart(1), PXY(2,PID)+ecart(2), PXY(3,PID)+ecart(3), num2str(PID'));
  end;
  if show_cv_num,
    text(cv_center(1,:), cv_center(2,:), cv_center(3,:), num2str(cvlst'), 'Color', [.7 0 0]);
  end;
  if show_dof_num,
    text(dofpos(1,:)-ecart(1), dofpos(2,:)-ecart(2), dofpos(3,:)-ecart(3), num2str(dofid'), 'Color', [0 .4 0]);
  end;
%  axis([(xm-ecartx/10) (xM+ecartx/10) (ym-ecarty/10) (yM+ecarty/10) (zm-ecartz/10) (zM+ecartz/10)]);
end

if (is_hold)
  hold on;
else
  hold off;
end;