#include <matlabint_misc.h>
#include <matlabint_mesh_fem.h>

using namespace matlabint;


static void set_fem(getfem::mesh_fem *mf, matlabint::mexargs_in& in)
{
  getfem::pfem               fem = in.pop().to_fem();
  getfem::pintegration_method pfi = in.pop().to_fem_interpolation();

  /* check or build the convex list */
  dal::bit_vector bv;
  if (in.remaining() == 1) {
    bv = in.pop().to_bit_vector(&mf->linked_mesh().convex_index(), -1);
  } else {
    bv.add(0, mf->linked_mesh().nb_convex()-1);
  }
  
  /* check for the validity of the operation */
  dal::bit_vector nn = bv; 
  size_type k;
  for (k << nn; k != size_type(-1); k << nn) {
    if (!mf->linked_mesh().convex_index().is_in(k))
      DAL_THROW(matlabint_error,"Convex " << k+1 << " was not found in mesh");
    if (fem->basic_structure() != mf->linked_mesh().structure_of_convex(k))
      DAL_THROW(matlabint_error, "Structure of the FEM is Incompatible "
		"with the structure of convex " << k+1);
  }
    
  /* all the work done here */
  mf->set_finite_element(bv, fem, pfi);
}

/* set the classical fem of order on the mesh_fem, with a classical integration
   method */
static void set_classical_fem(getfem::mesh_fem *mf, matlabint::mexargs_in& in) {
  size_type K = in.pop().to_integer(1,255);
  dal::bit_vector bv;
  if (in.remaining() == 1) {
    bv = in.pop().to_bit_vector(&mf->linked_mesh().convex_index(), -1);
  } else {
    bv.add(0, mf->linked_mesh().nb_convex()-1);
  }

  for (size_type cv = bv.take_first(); cv != size_type(-1); cv << bv) {
    bgeot::pgeometric_trans pgt = mf->linked_mesh().trans_of_convex(cv);
    mf->set_finite_element(cv, getfem::classical_fem(pgt,K), 
			   getfem::exact_classical_im(pgt));
  }
}

static void set_boundary_num(getfem::mesh_fem *mf, matlabint::mexargs_in& in)
{
  unsigned boundary_num  = in.pop().to_integer(1,100000);
  mlab_vect v            = in.pop().to_int_vector(2,-1);

  const getfem::getfem_mesh &mesh = mf->linked_mesh();

  /* loop over the edges of mxEdge */
  for (size_type j=0; j < v.getn(); j++) {
    unsigned cv = (int)v(0,j)-1;
    unsigned f  = (int)v(1,j)-1;
    if (!mesh.convex_index().is_in(cv)) {
      DAL_THROW(matlabint_bad_arg, "Invalid convex number '" << cv+1 << "' at column " << j+1);
    }
    if (f >= mesh.structure_of_convex(cv)->nb_faces()) {
      DAL_THROW(matlabint_bad_arg, "Invalid face number '" << f+1 << "' at column " << j+1);
    }
    mf->add_boundary_elt(boundary_num, cv, f);
    if (!mf->is_convex_on_boundary(cv, boundary_num)) THROW_INTERNAL_ERROR;
  }
}

/*MLABCOM
  FUNCTION [x] = gf_mesh_fem_set(meshfem MF, operation [, args])

  General function for editing mesh_fem objects
  

  * gf_mesh_fem_set(MF, 'fem', fem FEM, integration PPI [, vec CVIDX])

  Sets the finite element method to FEM with PPI as integration method for all
  the convexes of index CVIDX in the mesh linked to MF. If CVIDX is not used,
  all the elements are assigned the FEM/PPI.

  * gf_mesh_fem_set(MF, 'classical fem', fem FEM, int K [, vec CVIDX])

  Sets the classical fem of order K on the listed convexes (PK for simplexes,
  QK for parallelepipeds, etc..)

  * gf_mesh_fem_set(MF, 'boundary number', int bnum, mat FACES)

  Assigns the boundary number bnum to the convex faces stored in each column of
  the matrix FACES. Each column contains the list of point numbers that
  composes the face.

  * gf_mesh_fem_set(MF, 'delete boundary', BLST)

  Remove the boundaries listed in BLST

  $Id: gf_mesh_fem_set.C,v 1.6 2002/09/02 09:29:58 pommier Exp $
MLABCOM*/

void gf_mesh_fem_set(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 2) {
    DAL_THROW(matlabint_bad_arg, "Wrong number of input arguments");
  }

  getfem::mesh_fem *mf = in.pop().to_mesh_fem();
  std::string cmd        = in.pop().to_string();
  if (check_cmd(cmd, "fem", in, out, 2, 3, 0, 0)) {
    set_fem(mf, in);
  } else if (check_cmd(cmd, "classical fem", in, out, 1, 1, 0, 0)) {
    set_classical_fem(mf, in);
  } else if (check_cmd(cmd, "boundary number", in, out, 2, 2, 0, 0)) {
    set_boundary_num(mf, in);
  } else if (check_cmd(cmd, "delete boundary", in, out, 1, 1, 0, 0)) {
    dal::bit_vector bv = in.pop().to_bit_vector(&mf->get_valid_boundaries());
    size_type b;
    for (b << bv; b != size_type(-1); b << bv) mf->sup_boundary(b);
  } else bad_cmd(cmd);
}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  catch_errors(nlhs, plhs, nrhs, prhs, gf_mesh_fem_set);
}
