#include <matlabint_misc.h>
#include <matlabint_mesh_fem.h>

/*
  $Id: gf_mesh_fem_get.C,v 1.7 2002/09/05 14:48:34 pommier Exp $

  ChangeLog:
  *Log:$
 */


using namespace matlabint;
typedef enum {IS_LAGRANGE, IS_EQUIVALENT, IS_POLYNOMIAL} test_what;
static void
test_fems(test_what what, const getfem::mesh_fem *mf, mexargs_in& in, mexargs_out& out)
{
  dal::bit_vector cvlst;
  bool return_bool = false;
  dal::bit_vector islst;
  if (in.remaining()) cvlst = in.pop().to_bit_vector(&mf->convex_index());
  else { cvlst = mf->convex_index(); return_bool = true; }
  size_type cv;
  for (cv << cvlst; cv != size_type(-1); cv << cvlst) {
    bool it_is = false;
    if (!mf->convex_index()[cv]) DAL_THROW(matlabint_error, "convex " << cv << "does not exist");
    switch (what) { /* hope the compiler will optimize that */
    case IS_LAGRANGE:   it_is = mf->fem_of_element(cv)->is_lagrange(); break;
    case IS_EQUIVALENT: it_is = mf->fem_of_element(cv)->is_equivalent(); break;
    case IS_POLYNOMIAL: it_is = mf->fem_of_element(cv)->is_polynomial(); break;
    }
    if (it_is) {
      islst.add(cv);
    }
  }
  if (return_bool) {
    out.pop().from_integer(islst.card() == mf->convex_index().card() ? 1 : 0);
  } else {
    out.pop().from_bit_vector(islst);
  }
}

dal::bit_vector
get_cv_dof_list(getfem::mesh_fem *mf, mexargs_in& in) {
  dal::bit_vector cvlst = in.remaining() ? 
    in.pop().to_bit_vector(&mf->convex_index()) : mf->convex_index();
  dal::bit_vector dofs;
  size_type cv;
  for (cv << cvlst; cv != size_type(-1); cv << cvlst) {
    if (!mf->convex_index()[cv]) THROW_INTERNAL_ERROR;
    getfem::ref_mesh_dof_ind_ct cvdof = mf->ind_dof_of_element(cv);
    getfem::ref_mesh_dof_ind_ct::const_iterator it;
    for (it = cvdof.begin(); it != cvdof.end(); it++) {       
      dofs.add(*it);
    }
  }
  return dofs;
}

/*MLABCOM
  FUNCTION [x] = gf_mesh_fem_get(meshfem MF, operation [, args])

  General function for inquiring mesh_fem objects

  * I = gf_mesh_fem_get(MF, 'nbdof')
  Returns the number of degree of freedom of the mesh_fem mf.

  * I = gf_mesh_fem_get(MF, 'dofs from cvs', CVLIST)
  Returns the dofs of the convexes listed in CVLIST

  * I = gf_mesh_fem_get(MF, 'u_dim')
  Returns the 'natural' number of dimensions of unknowns (1 for scalar pde,
  2 for pde whose solution is a field etc..)

  * I = gf_mesh_fem_get(MF, {'is_lagrangian' |
                             'is_equivalent' |
                             'is_polynomial'} [,CVLST])

  if CVLST is omitted, returns 1 if all convexes in the mesh are
  lagrangian (resp. equivalents, resp. polynomials), or 0. If CVLST is
  present, returns the convex numbers (from the ones in CVLST) which
  are lagrangian (resp. etc..)

  * M = gf_mesh_fem_get(MF, 'eltm', eltm MET, int CV)
  Returns the elementary matrix evaluated on the convex CV for the
  elementary matrix type MET.
  !!WARNING!! Be sure that the fem used for the construction of MET is compatible
  with the fem assigned to element CV ! This is not checked by the function !

  * M = gf_mesh_fem_get(MF, 'eltm on face', eltm MET, int CV, int F)
  Returns the elementary matrix evaluated on the face F of convex CV, for the
  elementary matrix type MET.

  * BLST = gf_mesh_fem_get(MF, 'boundaries', bnum)
  Returns the list of valid boundaries

  * I = gf_mesh_fem_get(MF, 'faces on boundary', bnum)
  Returns the list of faces on the boundary 'bnum'. On output, the first row of
  I contains the convex numbers, and the second row contains the face numbers.

  * [DOF_XY] = gf_mesh_fem_get(MF, 'dofs nodes'[, CVLIST])
  Returns the list of interpolation points for the convexes listed in CVLIST
  (if CVLIST is omitted, all convexes are considered).
  
  $Id: gf_mesh_fem_get.C,v 1.7 2002/09/05 14:48:34 pommier Exp $
MLABCOM*/

void gf_mesh_fem_get(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 2) {
    DAL_THROW(matlabint_bad_arg, "Wrong number of input arguments");
  }

  mesh_fem_int *mf       = in.pop().to_mesh_fem();
  std::string cmd        = in.pop().to_string();
  if (check_cmd(cmd, "nbdof", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(mf->nb_dof());
  } else if (check_cmd(cmd, "dofs from cvs", in, out, 0, 1, 0, 1)) {
    dal::bit_vector dofs = get_cv_dof_list(mf, in);

    out.pop().from_bit_vector(dofs);
  } else if (check_cmd(cmd, "u_dim", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(mf->get_u_dim());
  } else if (check_cmd(cmd, "is lagrangian", in, out, 0, 1, 0, 1)) {
    test_fems(IS_LAGRANGE , mf, in, out);
  } else if (check_cmd(cmd, "is equivalent", in, out, 0, 1, 0, 1)) {
    test_fems(IS_EQUIVALENT, mf, in, out);
  } else if (check_cmd(cmd, "is polynomial", in, out, 0, 1, 0, 1)) {
    test_fems(IS_POLYNOMIAL, mf, in, out);
  } else if (check_cmd(cmd, "eltm", in, out, 2, 2, 0, 1)) {
    getfem::pmat_elem_type pmet = in.pop().to_mat_elem_type();
    id_type cv                  = in.pop().to_integer(1,1000000000);
    cv--; /* from matlab numbering to getfem numbering */
    if (!(mf->convex_index().is_in(cv))) 
      DAL_THROW(matlabint_bad_arg, "The convex " << cv << " in not part of the mesh");

    /* one should check that the fem given to the MET is 
       compatible with the fem of the element (not easy ..) */
    getfem::base_tensor t;
    getfem::pmat_elem_computation pmec = 
      getfem::mat_elem(pmet, 
		       mf->int_method_of_element(cv) , 
		       mf->linked_mesh().trans_of_convex(cv));
    pmec->gen_compute(t, mf->linked_mesh().points_of_convex(cv));
    out.pop().from_tensor(t);
  } else if (check_cmd(cmd, "eltm on face", in, out, 3, 3, 0, 1)) {
    getfem::pmat_elem_type pmet = in.pop().to_mat_elem_type();
    id_type cv = in.pop().to_integer(1); 
    cv--;
    if (!(mf->convex_index().is_in(cv))) 
      DAL_THROW(matlabint_bad_arg, "The convex " << cv << " in not part of the mesh");
    id_type f = in.pop().to_integer(1, mf->linked_mesh().structure_of_convex(cv)->nb_faces());
    f--;
    getfem::base_tensor t;
    getfem::pmat_elem_computation pmec = 
      getfem::mat_elem(pmet, 
		       mf->int_method_of_element(cv) , 
		       mf->linked_mesh().trans_of_convex(cv));
    pmec->gen_compute_on_face(t, mf->linked_mesh().points_of_convex(cv), f);
    out.pop().from_tensor(t);
  } else if (check_cmd(cmd, "boundaries", in, out, 0, 0, 0, 1)) {
    dal::bit_vector bv = mf->get_valid_boundaries();
    //mexPrintf("card=%d\n", bv.card());
    mlab_vect w = out.pop().create_vector(bv.card());
    size_type i=0, k; 
    for (k << bv; k != size_type(-1); k << bv, i++) {
      w[i] = (double)k;
      //mexPrintf("k=%d\n", k);
    }
    if (i != w.size()) THROW_INTERNAL_ERROR;
  } else if (check_cmd(cmd, "faces on boundary", in, out, 1, 1, 0, 1)) {
    int bnum = in.pop().to_integer(1,100000);
    std::vector<unsigned> cvlst;
    std::vector<unsigned> facelst;
    dal::bit_vector bv = mf->convex_index();
    unsigned cv;
    for (cv << bv; cv != size_type(-1); cv << bv) {
      if (mf->is_convex_on_boundary(cv,bnum)) {
	const dal::bit_vector &faces_on_bound = mf->faces_of_convex_on_boundary(cv,bnum);
	
	/* loop over faces belonging to the boundary */
	for (dal::bit_vector::const_iterator bbfit=faces_on_bound.begin();
	     bbfit != faces_on_bound.end(); bbfit++) {
	  if (!*bbfit) continue;
	  cvlst.push_back(cv);
	  facelst.push_back(bbfit.index());
	}
      }
    }
    mlab_vect w = out.pop().create_vector(2, cvlst.size());
    for (size_type j=0; j < cvlst.size(); j++) { w(0,j) = cvlst[j]+1; w(1,j) = facelst[j]+1; }

  } else if (check_cmd(cmd, "dofs nodes", in, out, 0, 1, 0, 2)) {
    dal::bit_vector dof_lst = get_cv_dof_list(mf, in);

    mlab_vect w = out.pop().create_vector(mf->linked_mesh().dim(), dof_lst.card());
    size_type dof, j = 0;
    for (dof << dof_lst; dof != size_type(-1); dof << dof_lst) {
      if (mf->point_of_dof(dof).size() != w.getm() || j >= w.getn()) THROW_INTERNAL_ERROR;
      for (size_type i=0; i < w.getm(); i++) w(i,j)= mf->point_of_dof(dof)[i];
      //      std::copy(mf->point_of_dof(dof).begin(),mf->point_of_dof(dof).end(), &w(0,j));
      j++;
    }
  } else bad_cmd(cmd);
}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  catch_errors(nlhs, plhs, nrhs, prhs, gf_mesh_fem_get);
}
