#include <matlabint.h>
#include <matlabint_pfi.h>

using namespace matlabint;

/*MLABCOM
  FUNCTION gf_integ(method [, args..])
    General function for obtaining handle to various fem integrations methods
    
    * gf_integ('exact/simplex', dim) : 
    Exact integration on a simplex (assuming the base functions are 
    polynomials).

    * gf_integ('NC/simplex', dim, npts) : 
    Newton-Cotes on a simplex.

    * gf_integ('Gauss/segment', npts) : 
    Integration using Gauss points on a segment.

    * gf_integ('triangle1')    ; (1 + 1+1+1)
    * gf_integ('triangle2')    ; (3 + 2+2+2)
    * gf_integ('triangle2bis') ; (3 + 2+2+2)
    * gf_integ('triangle3')    ; (4 + 2+2+2)
    * gf_integ('triangle4')    ; (6 + 3+3+3)
    * gf_integ('triangle5')    ; (7 + 3+3+3)
    * gf_integ('triangle6')    ; (12+ 4+4+4)
    * gf_integ('triangle7')    ; (13+ 4+4+4)
    * gf_integ('quad2')        ; (3 + 2+2+2+2)
    * gf_integ('quad3')        ; (4 + 2+2+2+2)
    * gf_integ('quad5')        ; (7 + 3+3+3+3)
    * gf_integ('tetrahedron1') ; (1 + 1+1+1+1)
    * gf_integ('tetrahedron2') ; (4 + 3+3+3)
    * gf_integ('tetrahedron3') ; (5 + 4+4+4+4)
    * gf_integ('tetrahedron5') ; (15+ 7+7+7+7)
    Specialized approximative integration on various elements (the
    number in the nameis the order of the approximation, the numbers
    written on the left indicate the number of integration points for
    volume integrals, and for integrals on each of its faces.

    * gf_integ('product', i1, i2) :
    Approximative integration on the tensorial product of 2 approx 
    integration methods pi1 and pi2. For example, an integration 
    method on a 2D quadrangle may be obtained from the product of
    two integrations methods on a 1D simplex (a segment).
    * IM_EXACT_SIMPLEX(n)        : exact integration on simplexes.
    * IM_PRODUCT(a, b)           : product of two integration methods
    * IM_EXACT_PARALLELEPIPED(n) : exact integration on parallelepipeds
    * IM_EXACT_PRISM(n)          : exact integration on prisms
    * IM_GAUSS1D(K)       : Gauss method on the segment, order K
    * IM_NC(N,K)          : Newton-Cotes approximative integration 
    *                       on simplexes, order K
    * IM_NC_PARALLELEPIPED(N,K) :  product of Newton-Cotes integration 
    *                              on parallelepipeds
    * IM_NC_PRISM(N,K)    : product of Newton-Cotes integration on prisms
    * IM_GAUSS_PARALLELEPIPED(N,K) :  product of Gauss1D integration
    *                                 on parallelepipeds
    * IM_TRIANGLE(K)      : Gauss methods on triangles (K=1..7)
    * IM_QUAD(K)          : Gauss methods on quadrilaterons (K=2, 3 or 5)
    * IM_TETRAHEDRON(K)   : Gauss methods on tetrahedrons (K=1, 2, 3 or 5)

MLABCOM*/

void gf_integ(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 1) {
    DAL_THROW(matlabint_bad_arg, "Wrong number of input arguments");
  }
  std::string cmd = in.pop().to_string();
  out.pop().from_object_id(matlabint::ind_pfi(getfem::int_method_descriptor(cmd))
			   + MATLABINT_PFI_MASK);
}
/*
  if (check_cmd(cmd, "exact/simplex", in, out, 1, 1, 0, 1)) {
    unsigned dim = in.pop().to_integer();
    if (dim > 256) DAL_THROW(matlabint_bad_arg, cmd << " : argument out of range");
    out.pop().from_object_id(matlabint::ind_pfi(getfem::simplex_poly_integration(dim))
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "NC/simplex", in, out, 2, 2, 0, 1)) {
    unsigned dim  = in.pop().to_integer();
    unsigned npts = in.pop().to_integer();
    if (dim > 256 || npts > 256) DAL_THROW(matlabint_bad_arg, cmd << " : argument out of range");
    out.pop().from_object_id(matlabint::ind_pfi(getfem::Newton_Cotes_approx_integration(dim,npts))
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "Gauss/simplex", in, out, 1, 1, 0, 1)) {
    unsigned npts = in.pop().to_integer();
    if (npts > 256) DAL_THROW(matlabint_bad_arg, cmd << " : argument out of range");
    out.pop().from_object_id(matlabint::ind_pfi(getfem::Gauss_approx_integration(npts))
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "triangle1", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::triangle1_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "triangle2", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::triangle2_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "triangle2bis", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::triangle2bis_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "triangle3", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::triangle3_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "triangle4", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::triangle4_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "triangle5", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::triangle5_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "triangle6", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::triangle6_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "triangle7", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::triangle7_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "quad2", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::quad2_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "quad3", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::quad3_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "quad5", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::quad5_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "tetrahedron1", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::tetrahedron1_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "tetrahedron2", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::tetrahedron2_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "tetrahedron3", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::tetrahedron3_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "tetrahedron5", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(matlabint::ind_pfi(getfem::tetrahedron5_approx_integration())
			     +MATLABINT_PFI_MASK);
  } else if (check_cmd(cmd, "product", in, out, 2, 2, 0, 1)) {
    getfem::pintegration_method pfi1 = in.pop().to_fem_interpolation();
    getfem::pintegration_method pfi2 = in.pop().to_fem_interpolation();
    getfem::pintegration_method prod = getfem::convex_product_approx_integration(pfi1.method.pai,
										pfi2.method.pai);
    out.pop().from_object_id(matlabint::ind_pfi(prod)+MATLABINT_PFI_MASK);
  } else bad_cmd(cmd);
}
*/
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  catch_errors(nlhs, plhs, nrhs, prhs, gf_integ);
}
