/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Command (matlabcom) version 1.0                        */
/* File    : gf_delete.C : deletes a getfem object                         */
/*     									   */
/*                                                                         */
/* Date : August 2002.                                                     */
/* Author : Julien Pommier, pommier@@gmm.insa-tlse.fr                      */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */

#include "mex.h"
#include <matlabint.h>
#include <matlabint_workspace.h>

using namespace matlabint;

/*MLABCOM
  FUNCTION gf_delete(I,[J,K,...])
 
    Deletes an existing getfem object from memory (integer set, polynomial, 
    mesh, mesh_fem). i should be a descriptor given by new_intset, new_mesh, 
    new_mesh_fem, etc..

  REMARK:
    I may be an array of objects id. And GF_DELETE may also be called with a 
    sequence (I,J,...) of objects id.

    Note that if another object uses I, then object I will be deleted only
    when both have been asked for deletion.

  SEE ALSO:
    gf_new_intset, gf_new_poly, gf_new_mesh, gf_new_mesh_fem,
    gf_new_mesh_from_pdetool
 MLABCOM*/

void gf_delete(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.remaining() < 1)
    mexErrMsgTxt("Wrong number of input arguments, should be at least 1.");
  if (out.narg() != 0)
    mexErrMsgTxt("No output argument needed.");
  
  while (in.remaining()) {
    const mxArray *arg = in.pop().arg;
    if (mxGetClassID(arg) != mxUINT32_CLASS) {
      DAL_THROW(matlabint_bad_arg, "wrong type for argument, expecting an UINT32 getfem identifier");
    }
    for (int j=0; j < mxGetNumberOfElements(arg); j++) {
      matlabint::id_type id = ((matlabint::id_type *)mxGetData(arg))[j];
      
      mexPrintf("delete id %d\n", id);
      
      if (matlabint::workspace().object(id)) { /* throw an exeption if object not found */
	mexPrintf("ready to delete %d\n", id);
	matlabint::workspace().delete_object(id);
      } else {
	mexPrintf("bobob\n");
      }
    }
  }
}


void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  catch_errors(nlhs, plhs, nrhs, prhs, gf_delete);
}
