#include <getfem_assembling.h>
#include <matlabint_misc.h>

using namespace matlabint;


/*MLABCOM
  
  FUNCTION gf_asm(operation[, arg])

  * F = gf_assem('volumic', mf_u, mf_d, Fd)
  Assembles a volumic source term, on the mesh_fem MF_U, using the 
  data vector FD defined on the data mesh_fem MF_D.

  * M = gf_assem('mass matrix', mf_u[, dim])

  * M = gf_assem('linear elasticity', mf_u, mf_d, lambda_d, mu_d)
  assembles elementary matrices for linear elasticity (and partialy stockes)
  
  * M = gf_assem('stockes pressure term', mesh_fem mf_u, mesh_fem mf_p, mesh_fem mf_data, VEC data)
  A ssembles elementary matrices for the pressure term in stockes mixed elements formulation:
  output: B is a sparse matrix corresponding to $\int p.div v$

  * F = gf_assem('neumann',boundary_num, mesh_fem mf_u, esh_fem mf_d, vec g)
  Assembles a Neumann boundary condition. 
  G should be be a [U_DIM x N] matrix, where n is the number of degree of freedom
  of mf_d, and U_DIM is dimension of the unknown u (that is set when creating the 
  mesh_fem).

  * M = gf_assem('boundary qu term',boundary_num, mesh_fem mf_u, esh_fem mf_d,mat q)
  G should be be a [(U_DIM*U_DIM) x N] matrix, where n is the number of degree of freedom
  of mf_d, and U_DIM is dimension of the unkown u (that is set when creating the 
  mesh_fem). Q is stored in the following order
   Q = TRANSPOSE[q11_x, q21_x, q12_x, q22_x; <--- dof #1, x component
                 q11_y, q21_y, q12_y, q22_y; <--- dof #1, y component
                 q11_x, q21_x, q12_x, q22_x; <--- dof #2, x component
                 q11_y, q21_y, q12_y, q22_y; <--- dof #2, y component
                  ...]

  * [Q,G,H,R,F]=gf_asm('pdetool boundary conditions',mf_u, mf_d, b, e[, f_expr])

  Assembles pdetool boundary conditions: B is the boundary matrix
  exported by the pdetool, and E is the edges array. 'F_EXPR' is an
  optionnal expression (or vector) for the volumic term. On return
  Q,G,H,R,F contain the assembled boundary conditions (Q and H are
  matrices), similar to the ones returned by the function ASSEMB
  from PDETOOL.

  MLABCOM*/
void gf_assem(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 1) {
    DAL_THROW(matlabint_bad_arg, "Wrong number of input arguments");
  }
  std::string cmd = in.pop().to_string();

  if (check_cmd(cmd, "volumic", in, out, 3, 3, 1, 1)) {
    getfem::mesh_fem *mf_u = in.pop().to_mesh_fem();
    getfem::mesh_fem *mf_d = in.pop().to_mesh_fem();
    mlab_vect Fd           = in.pop().to_scalar_vector(-1, mf_d->nb_dof());
    unsigned Fdim = Fd.getm();
    mlab_vect Fu           = out.pop().create_vector(Fdim, mf_u->nb_dof());

    getfem::assembling_volumic_source_term(Fu, *mf_u, *mf_d, Fd, Fdim);
  } else if (check_cmd(cmd, "mass matrix", in, out, 2, 2,1, 1)) {
    matlabint::mesh_fem_int *mf_u = in.pop().to_mesh_fem();
    unsigned dim = in.remaining() ? in.pop().to_integer() : mf_u->get_u_dim();
    gf_sparse_matrix M(mf_u->nb_dof() * dim, 
		       mf_u->nb_dof() * dim);
    getfem::mass_matrix(M, *mf_u, dim);
    out.pop().from_sparse(M);
  } else if (check_cmd(cmd, "linear elasticity", in, out, 4, 4,1, 1)) {
    getfem::mesh_fem *mf_u = in.pop().to_mesh_fem();
    getfem::mesh_fem *mf_d = in.pop().to_mesh_fem();
    mlab_vect lambda       = in.pop().to_scalar_vector(mf_d->nb_dof());
    mlab_vect mu           = in.pop().to_scalar_vector(mf_d->nb_dof());
    gf_sparse_matrix M(mf_u->nb_dof() * mf_u->linked_mesh().dim(),
		       mf_u->nb_dof() * mf_u->linked_mesh().dim());
    getfem::assembling_rigidity_matrix_for_linear_elasticity(M, *mf_u, *mf_d, lambda, mu);
    out.pop().from_sparse(M);
  } else if (check_cmd(cmd, "stockes pressure term", in, out, 4, 4, 1, 1)) {
    getfem::mesh_fem *mf_u = in.pop().to_mesh_fem();
    getfem::mesh_fem *mf_p = in.pop().to_mesh_fem();
    getfem::mesh_fem *mf_d = in.pop().to_mesh_fem();
    mlab_vect        vec_d = in.pop().to_scalar_vector(mf_d->nb_dof());
    gf_sparse_matrix  M(mf_u->nb_dof() * mf_u->linked_mesh().dim(), 
			mf_p->nb_dof());
    getfem::assembling_mixed_pressure_term(M, *mf_u, *mf_p, *mf_d, vec_d);
    out.pop().from_sparse(M);
  } else if (check_cmd(cmd, "neumann", in, out, 4, 4, 1, 1)) {
    int boundary_num       = in.pop().to_integer();
    mesh_fem_int *mf_u     = in.pop().to_mesh_fem();
    mesh_fem_int *mf_d     = in.pop().to_mesh_fem();
    unsigned u_dim = mf_u->get_u_dim();
    mlab_vect g            = in.pop().to_scalar_vector(u_dim, mf_d->nb_dof());
    mlab_vect F            = out.pop().create_vector(u_dim*mf_u->nb_dof(),1);
    getfem::assembling_Neumann_condition(F, *mf_u, boundary_num,
					 *mf_d, g, u_dim);
  } else if (check_cmd(cmd, "boundary qu term", in, out, 4, 4, 1, 1)) {
    int boundary_num       = in.pop().to_integer();
    mesh_fem_int *mf_u     = in.pop().to_mesh_fem();
    mesh_fem_int *mf_d     = in.pop().to_mesh_fem();

    unsigned u_dim = mf_u->get_u_dim();
    unsigned u_dim2 = u_dim*u_dim;
    mlab_vect q            = in.pop().to_scalar_vector(u_dim2, 
						       mf_d->nb_dof());
    /*
    check_bc_dimensions(h, mf_d->nb_dof(), 0, u_dim, 7);
    check_bc_dimensions(r, mf_d->nb_dof(), 1, u_dim, 8);
    if (mesh_dim != mf_d->linked_mesh().dim()) {
      DAL_THROW(matlabint_bad_arg,
		"?? what are you doing? mf_u and mf_d should refer to the same mesh");
		}*/

    gf_sparse_matrix Q(mf_u->nb_dof() * u_dim, mf_u->nb_dof() * u_dim);
    getfem::assembling_boundary_qu_term(Q, *mf_u, boundary_num, *mf_d, q, u_dim);
    out.pop().from_sparse(Q);
  } else if (check_cmd(cmd, "pdetool boundary conditions", in, out)) {
    call_matlab_function("gf_asm_pdetoolbc", in,out);
  } else bad_cmd(cmd);
}


void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  catch_errors(nlhs, plhs, nrhs, prhs, gf_assem);
}
