//  Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>

#include <gtkmm.h>
#include <glibmm.h>
#include <iostream>
#include <giomm.h>
#include "main.h"
#include "cbc-osd-connector.h"
#include <libnotify/notify.h>
#include <libsoupmm/session.h>
#include <libsoupmm/message.h>
#include <libsoupmm/uri.h>
#include <libsoupmm/message-body.h>
#include <libsoupmm/cookie-jar.h>
#include "timing.h"
#include "ucompose.hpp"
#include "json-glibmm.hpp"

CBCOSDConnector* CBCOSDConnector::create(bool news, bool sports)
{
  return new CBCOSDConnector(news, sports);
}

CBCOSDConnector::CBCOSDConnector(bool news, bool sports)
{
  d_watch_news = news;
  d_watch_sports = sports;
  web = Soup::Session::create (true);
  web->property_timeout () = 5;
  web->property_idle_timeout () = 5;
  web->property_max_conns_per_host () = 2;
  web->add_feature_by_type (SOUP_TYPE_COOKIE_JAR);
}

void open_browser(NotifyNotification *notification, char *action, gpointer user_data)
{
  Glib::ustring url =std::string((gchar*)user_data);
  Glib::ustring cmd = Glib::find_program_in_path("gvfs-open") + " '" + url +"'";
  Glib::spawn_command_line_async(cmd);
}

void CBCOSDConnector::notify(Glib::ustring show_name, Glib::ustring show_id, Glib::ustring url)
{
  NotifyNotification *notification = notify_notification_new("Streaming Live",Glib::locale_from_utf8(show_name).c_str(), NULL);
  Glib::RefPtr<Gdk::Pixbuf> pixbuf = Gdk::Pixbuf::create_from_file
    (Main::get_data_path() + "/data/cbc-osd.png");
  notify_notification_set_icon_from_pixbuf  (notification, pixbuf->gobj());
  gpointer urlp = (gpointer) g_strdup(Glib::locale_from_utf8(url).c_str());
  notify_notification_add_action(notification, Glib::locale_from_utf8(show_id).c_str(), "Start Streaming", open_browser, urlp, g_free);
  notify_notification_set_timeout(notification, 20 * 1000);
  notify_notification_show(notification,NULL);
}

CBCOSDConnector::~CBCOSDConnector()
{
}
    
void CBCOSDConnector::connect()
{
  if (!notify_is_initted ())
    if (!notify_init (PACKAGE))
      {
        connect_succeeded.emit(false);
        return;
      }

  Glib::RefPtr < Soup::Message > msg = Soup::Message::create ("GET", 
                                                              CBC_VIDEO_URL);
  web->queue_message (msg, sigc::mem_fun (*this, &CBCOSDConnector::on_get_id_attempted));

}

void CBCOSDConnector::start_polling(int msecs)
{
  poll();
  timer = Timing::instance().register_timer
    (sigc::mem_fun(this, &CBCOSDConnector::poll), msecs);
}

void CBCOSDConnector::poll_news()
{
  Glib::ustring url = String::ucompose("http://cbc.feeds.theplatform.com/ps/JSON/PortalService/2.2/getReleaseList?PID=%1&startIndex=1&endIndex=6&field=ID&field=title&field=description&field=airdate&field=length&field=PID&query=Categories|News&contentCustomField=show&contentCustomField=featureImage&contentCustomField=sport", id);
  Glib::RefPtr < Soup::Message > msg = Soup::Message::create ("GET", url);
  web->queue_message (msg, sigc::mem_fun (*this, &CBCOSDConnector::on_news_polled));
}

void CBCOSDConnector::poll_sports()
{
  Glib::ustring url = String::ucompose("http://cbc.feeds.theplatform.com/ps/JSON/PortalService/2.2/getReleaseList?PID=%1&startIndex=1&endIndex=6&field=ID&field=title&field=description&field=airdate&field=length&field=PID&query=Categories|Sports&contentCustomField=show&contentCustomField=featureImage&contentCustomField=sport", id);
  Glib::RefPtr < Soup::Message > msg = Soup::Message::create ("GET", url);
  web->queue_message (msg, sigc::mem_fun (*this, &CBCOSDConnector::on_sports_polled));
}

bool CBCOSDConnector::poll()
{
  if (d_watch_news)
    poll_news();
  if (d_watch_sports)
    poll_sports();
  return Timing::CONTINUE;
}

void CBCOSDConnector::on_sports_polled(Glib::RefPtr<Soup::Message> &msg)
{
  if (msg->property_status_code () == 200)
    {
      web->cancel_message (msg, msg->property_status_code ());
      Soup::MessageBody body (msg->gobj ()->response_body);
      body.set_accumulate (true);
      Soup::Buffer buffer = body.flatten ();
      const char *data = buffer.gobj ()->data;
      guint len = buffer.gobj ()->length;
      if (len != 0)
        sports_polled.emit(data, len);
    }
}

void CBCOSDConnector::on_news_polled(Glib::RefPtr<Soup::Message> &msg)
{
  if (msg->property_status_code () == 200)
    {
      web->cancel_message (msg, msg->property_status_code ());
      Soup::MessageBody body (msg->gobj ()->response_body);
      body.set_accumulate (true);
      Soup::Buffer buffer = body.flatten ();
      const char *data = buffer.gobj ()->data;
      guint len = buffer.gobj ()->length;
      if (len != 0)
        news_polled.emit(data, len);
    }
}

void CBCOSDConnector::on_get_id_attempted(Glib::RefPtr<Soup::Message> &msg)
{
  if (msg->property_status_code () == 200)
    {
      web->cancel_message (msg, msg->property_status_code ());
      Soup::MessageBody body (msg->gobj ()->response_body);
      body.set_accumulate (true);
      Soup::Buffer buffer = body.flatten ();
      const char *data = buffer.gobj ()->data;
      guint len = buffer.gobj ()->length;
      if (len != 0)
        {
          std::vector < Glib::ustring > a = Glib::Regex::split_simple
            ("var feedPID = \"", data);
          if (a.size() < 2)
            {
              connect_succeeded.emit(false);
              return;
            }
          std::vector < Glib::ustring > b = Glib::Regex::split_simple
            ("\"", a[1]);
          if (b.size() < 2)
            {
              connect_succeeded.emit(false);
              return;
            }
          id = b[0];
          Glib::ustring url = String::ucompose("http://cbc.feeds.theplatform.com/ps/JSON/PortalService/2.2/getCategoryList?PID=%1&field=fullTitle&field=treeOrder&field=hasChildren&field=customData&field=description&field=parentID&field=title&field=ID&customField=AdCategory&customField=SortOrder&customField=Sub-Event&customField=Segment&customField=Producers&customField=Organizations&customField=LiveOnDemand&customField=EpisodeNumber&customField=ClipType&customField=Characters&customField=BylineCredit&customField=AudioVideo&customField=Aired&customField=CBCPersonalities&customField=People&customField=Event&customField=Region&customField=Sport&customField=SeasonNumber&customField=MaxClips&customField=SortField&customField=Show&customField=Genre&customField=CreatedBefore&customField=CreatedAfter&customField=Account&customField=GroupOrder&customField=GroupLevel&customField=IsDynamicPlaylist&customField=Keywords&customField=backgroundImage&query=FullTitles|News,News/Live%%20Streaming", id);
          Glib::RefPtr < Soup::Message > followupmsg = 
            Soup::Message::create ("GET",  url);
          web->queue_message (followupmsg, sigc::mem_fun (*this, &CBCOSDConnector::on_get_news_id_attempted));
          return;
        }
    }
  connect_succeeded.emit(false);
}

void CBCOSDConnector::on_get_news_id_attempted(Glib::RefPtr<Soup::Message> &msg)
{
  if (msg->property_status_code () == 200)
    {
      web->cancel_message (msg, msg->property_status_code ());
      Soup::MessageBody body (msg->gobj ()->response_body);
      body.set_accumulate (true);
      Soup::Buffer buffer = body.flatten ();
      const char *data = buffer.gobj ()->data;
      guint len = buffer.gobj ()->length;
      if (len != 0)
        {
          JsonParserMM parser;
          bool success = parser.load_from_data(std::string(data));
          if (!success)
            {
              connect_succeeded.emit(false);
              return;
            }
          JsonNodeMM *items_node = 
            parser.get_root()->get_object()->get_member("items");
          JsonNodeMM *news_node = (*(items_node->get_array()))[0]->get_object()->get_member("ID");

          news_id = String::ucompose("%1", news_node->get_int());
          Glib::ustring url = String::ucompose("http://cbc.feeds.theplatform.com/ps/JSON/PortalService/2.2/getCategoryList?PID=%1&field=fullTitle&field=treeOrder&field=hasChildren&field=customData&field=description&field=parentID&field=title&field=ID&customField=AdCategory&customField=SortOrder&customField=Sub-Event&customField=Segment&customField=Producers&customField=Organizations&customField=LiveOnDemand&customField=EpisodeNumber&customField=ClipType&customField=Characters&customField=BylineCredit&customField=AudioVideo&customField=Aired&customField=CBCPersonalities&customField=People&customField=Event&customField=Region&customField=Sport&customField=SeasonNumber&customField=MaxClips&customField=SortField&customField=Show&customField=Genre&customField=CreatedBefore&customField=CreatedAfter&customField=Account&customField=GroupOrder&customField=GroupLevel&customField=IsDynamicPlaylist&customField=Keywords&customField=backgroundImage&query=FullTitles|Sports,Sports/Live%%20Streaming", id);
          Glib::RefPtr < Soup::Message > followupmsg = 
            Soup::Message::create ("GET",  url);
          web->queue_message (followupmsg, sigc::mem_fun (*this, &CBCOSDConnector::on_get_sports_id_attempted));
        }
      else
        connect_succeeded.emit(false);
    }
  else
    connect_succeeded.emit(false);
}

void CBCOSDConnector::on_get_sports_id_attempted(Glib::RefPtr<Soup::Message> &msg)
{
  if (msg->property_status_code () == 200)
    {
      web->cancel_message (msg, msg->property_status_code ());
      Soup::MessageBody body (msg->gobj ()->response_body);
      body.set_accumulate (true);
      Soup::Buffer buffer = body.flatten ();
      const char *data = buffer.gobj ()->data;
      guint len = buffer.gobj ()->length;
      if (len != 0)
        {
          JsonParserMM parser;
          bool success = parser.load_from_data(std::string(data));
          if (!success)
            {
              connect_succeeded.emit(false);
              return;
            }
          JsonNodeMM *items_node = 
            parser.get_root()->get_object()->get_member("items");
          JsonNodeMM *sports_node = (*(items_node->get_array()))[0]->get_object()->get_member("ID");

          sports_id = String::ucompose("%1", sports_node->get_int());
          connect_succeeded.emit(true);
        }
      else
        connect_succeeded.emit(false);
    }
  else
    connect_succeeded.emit(false);
}

void CBCOSDConnector::stop_polling()
{
  timer.disconnect();
}
