
/*
 * Copyright (c) 1997 by Procom Technology,Inc.
 *
 * This program can be redistributed or modified under the terms of the 
 * GNU General Public License as published by the Free Software Foundation.
 * This program is distributed without any warranty or implied warranty
 * of merchantability or fitness for a particular purpose.
 *
 * See the GNU General Public License for more details.
 *
 */
 
#ifndef __LINUX_AF_NETB_H
#define __LINUX_AF_NETB_H

#include <linux/sockios.h>

/* General definitions */

#define NB_MAX_DATALEN		(64 * 1024)  /* 64K-bytes max data buffer len */
#define NB_MAX_NAMES		255          /* Maximum number of local names */
#define NB_MAX_ADAPTERS		16        /* Maximum number of local adapters */
#define NB_MAX_LINKS		256      /* Maximum number of links to others */
#define NB_MAX_SESSIONS		255   /* Maximum number of sessions on a link */

#define SOCK_NAME		SOCK_RAW  /* Socket type for Name registration */
#define NB_NAME_LEN		16


/* NetBEUI address type definitions */

typedef enum {
	NB_NAME_UNIQUE = 1,
	NB_NAME_GROUP
} name_type_t;

struct netbeui_addr {
	char		name [NB_NAME_LEN];
	unsigned char	reserved;	/* Safety region */
	name_type_t	name_type;
};

/* Maximum permitted length of this structure is MAX_SOCK_ADDR defined in 'net/socket.c' */
struct sockaddr_netbeui {
	unsigned short		snb_family;         /* s.b. AF_NETBEUI */
	struct netbeui_addr	snb_addr;
};


/* NetBEUI Socket Layer Default Values */

#define NBSO_DFLT_STO_SEC	0   /* Seconds of 'Send Time Out' value */
#define NBSO_DFLT_STO_MICSEC	0   /* Micro-Seconds of 'Send Time Out' value */

#define NBSO_DFLT_RTO_SEC	0   /* Seconds of 'Receive Time Out' value */
#define NBSO_DFLT_RTO_MICSEC	0   /* Micro-Seconds of 'Receive Time Out' value */

/* Macro definitions */

/*
 * Macro to set up NETBEUI SOCKADDRs.
 *
 * _snb  : pointer to a 'struct sockaddr_netbeui'.
 * _type : type of NetBIOS name.
 * _name : a NULL terminated char string.
 * _port : port number that will be located at end of NetBIOS name.
 */
#define SET_NETBEUI_SOCKADDR(_snb, _type, _name, _port)			\
{									\
	int _i;								\
									\
	(_snb)->snb_family = AF_NETBEUI;				\
	(_snb)->snb_addr.name_type = (_type);				\
									\
	for (_i = 0 ; _i < NB_NAME_LEN-1 ; _i++) {			\
		(_snb)->snb_addr.name[_i] = ' ';			\
	}								\
									\
	for (_i = 0 ; (*((_name)+_i) != '\0') && (_i < NB_NAME_LEN-1) ; _i++) {	\
		(_snb)->snb_addr.name[_i] = *((_name)+_i);		\
	}								\
									\
	(_snb)->snb_addr.name[NB_NAME_LEN-1] = (_port);			\
									\
	(_snb)->snb_addr.reserved = 0;					\
}


/* NetBEUI ioctls */

/* From 0x89E0 to 0x89EF are protocol private 'ioctl numbers'                 *\
 * For more information see  SIOCDEVPRIVATE and SIOCPROTOPRIVATE  definitions *
\* in 'include/linux/sockios.h' file                                          */
/*
 * 	NetBEUI specific I/O ConTroL calls (ioctl)
 */
#define SIOCGSTATE	(SIOCPROTOPRIVATE)   /* Gets NetBEUI socket state */
#define SIOCTRIMDATA	(SIOCPROTOPRIVATE+1) /* Trims the last bytes of current message */
#define SIOCSENDZERO	(SIOCPROTOPRIVATE+2) /* Sends a dummy session message with zero size */
#define SIOCRUWDGF	(SIOCPROTOPRIVATE+3) /* Removes UnWanted DataGram Frames */
#define NBIOCGSTATUS	(SIOCPROTOPRIVATE+4) /* Gets NetBIOS STATUS of local or remote node */
#define NBIOCCONFIG	(SIOCPROTOPRIVATE+5) /* Configures NetBEUI during running */


/* NetBEUI 'SIOCGSTATE' ioctl() Constants */
typedef enum {
	NBSO_INIT = 0,
	NBSO_LISTENING,
	NBSO_RUNNING
} nbso_state_t;


/* NetBEUI Configuration (via NBIOCCONFIG ioctl) */

/* NetBEUI Configuration request constants */
#define NB_MAX_NIF_NAME_LEN	16 /* Maximum of Network InterFace Name Length */
#define NB_UNBIND_FLAG_SAFE	0 /* Unbinds device only if no connection exist on it */
#define NB_UNBIND_FLAG_DROP	1 /* Drops connections that exist on device and then unbinds it */

#define NB_CFGCMD_NIF_UNBIND	0
#define NB_CFGCMD_NIF_BIND	1
#define NB_CFGCMD_DROP_SESS	2
#define NB_CFGCMD_DROP_LINK	3

/* NetBEUI Configuration request data structure */
struct netbeui_cfg {
	unsigned short	command;	/* NB_CFGCMD_... */

	char		nif_name [NB_MAX_NIF_NAME_LEN];
	unsigned char	reserved;	/* Safety region, always must be zero */
	unsigned char	flag;		/* NB_UNBIND_FLAG_... */

	int		ln_num;		/* 0 <= ln_num < NB_MAX_LINKS */
	int		sn_num;		/* 0 < sn_num < NB_MAX_SESSIONS */
};


/* NetBEUI setsockopt / getsockopt */

/*
 * NetBEUI specific level definitions for setsockopt() & getsockopt() 
 *
 * Note: SOL_SOCKET defined in 'include/asm/socket.h' to 1
 *       and other protocol constants defined in 'include/linux/socket.h'
 */
#define SOL_NETBEUI	300
#define SOL_LLC		301
#define SOL_LANA	302

/*
 * NetBEUI specific option names for setsockopt() & getsockopt()
 *
 * Note: Remainder of option names defined in 'asm/socket.h' from 1 to 15
 */
#define SO_SNDTIMEO	101
#define SO_RCVTIMEO	102
#define SO_SNDLOWAT	103
#define SO_RCVLOWAT	104
#define SO_USELOOPBACK	105
#define SO_URGENTACK	106	
#define SO_NBPARAM	107


/*
 * NetBEUI configurable parameters use by SO_NBPARAM
 *
 * Note: Setting value of a parameter to zero means no change to the
 *	 current value.
 */
struct netbeui_config 
{
	__u8			inactivity_timeout;
	__u8			transmit_timeout;
	__u8			transmit_count;
	__u8			resource_timeout;
	__u8			data_ack_timeout;
};


/* Definition of default and maximum value of NetBIOS configurable parameters */

#define DEFAULT_LINK_INACTIVITY_TIMEOUT	30	/* Unit is second */
#define MAX_LINK_INACTIVITY_TIMEOUT	255	/* Unit is second */

#define DEFAULT_TRANSMIT_TIMEOUT	1	/* Unit is 1/2 second */
#define MAX_TRANSMIT_TIMEOUT		10	/* Unit is 1/2 second */

#define DEFAULT_TRANSMIT_COUNT		6
#define MAX_TRANSMIT_COUNT		10

#define DEFAULT_RESOURCE_TIMEOUT	1	/* Unit is 1/10 second */
#define MAX_RESOURCE_TIMEOUT		10	/* Unit is 1/10 second */

#define DEFAULT_DATA_ACK_TIMEOUT	1	/* Unit is 1/10 second */
#define MAX_DATA_ACK_TIMEOUT		10	/* Unit is 1/10 second */



/* NetBEUI STATUS Service */

/* NetBEUI status service constants */
#define NB_MIN_STATUS_BUFF_LEN          60
#define NB_MAX_STATUS_BUFF_LEN          (NB_MIN_STATUS_BUFF_LEN + 18 * NB_MAX_NAMES)

/* NetBEUI STATUS request data structure */
struct netbeui_status {
	char 	called_name[NB_NAME_LEN];
	char	reserved;	/* Safety region, always must be zero */
	int 	buff_len;
	char 	status_buff[0]; /* A dynamic length array for status information */
};

/* Adapter types in status information */
enum {
	_3174_PEER = 0xFB,
	IBM_FDDI,
	ETHERNET,
	PC_NETWORK,
	TOKEN_RING
};

/* Name structure in status information */
struct nb_status_names {
	char	name[NB_NAME_LEN];
	__u8	name_number;
	__u8	name_status;
};

/* STATUS information structure */
typedef struct {
	__u8	adptr_addr[6];
	__u8	sftwr_release_no;
	__u8	zero;

	struct {
		__u8	adptr_type;
		__u8	sftwr_level;
	}	adptr_type_AND_sftwr_level;

	__u16	duration;
	__u16	no_rx_FRMR;			/* NOT SUPPORTED YET */
	__u16	no_tx_FRMR;			/* NOT SUPPORTED YET */
	__u16	no_rx_Iformat_LPDUs;		/* NOT SUPPORTED YET */
	__u16	no_abrtd_transmissions;		/* NOT SUPPORTED YET */
	__u32	no_succ_tx_packets;		/* NOT SUPPORTED YET */
	__u32	no_succ_rx_packets;		/* NOT SUPPORTED YET */
	__u16	no_tx_Iformat_LPDUs;		/* NOT SUPPORTED YET */
	__u16	lost_data_OR_buff_fails;	/* NOT SUPPORTED YET */
	__u16	no_DLC_T1_expired;		/* NOT SUPPORTED YET */
	__u16	no_DLC_Ti_expired;		/* NOT SUPPORTED YET */
	__u32	ext_status_inf_addr;		/* NOT SUPPORTED YET */
	__u16	no_free_NCBs;			/* NOT SUPPORTED YET */
	__u16	config_max_NCBs;		/* NOT SUPPORTED YET */
	__u16	max_no_NCBs;			/* NOT SUPPORTED YET */
	__u16	local_busy_OR_buff_fails;	/* NOT SUPPORTED YET */
	__u16	max_dgram_packet_size;
	__u16	no_pend_sess;			/* NOT SUPPORTED YET */
	__u16	config_max_pend_sess;		/* NOT SUPPORTED YET */
	__u16	max_no_pend_sess;
	__u16	max_size_sess_data_packet;
	__u16	no_names_in_local_name_tbl;

	struct nb_status_names	local_names[NB_MAX_NAMES];
} nb_status_buffer_t;


#endif __LINUX_AF_NETB_H
