#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2009 Free Software Foundation
#
# FILE:
# GRLayout.py
#
# DESCRIPTION:
# Classes related to the Layout section
#
# NOTES:
#


from gnue.common.definitions.GObjects import *
from gnue.common.apps import GDebug
from gnue.common.datasources import GConditions

from gnue.reports.base import GRExceptions
from GRSection import GRSection
from GRLayoutElement import GRLayoutElement
from GRField import GRField
from GRSumm import GRSumm
from GRFormula import GRFormula

import string
from xml.sax import saxutils


############################################################
#
# GRLayout
#
class GRLayout (GObj):
  def __init__(self, parent):
    GObj.__init__(self, parent, type='GRLayout')
    self.name = "__GR%s" % id(self)
    self._inits = (self.initialize,)
    self.suppressGNUeTags = 0


  def initialize(self):
    # Find the xml namespace in use by any child passthru objects
    # If there is more than one object attached to a layout
    # object, then create an unbound section to contain them.
    # This is a convenience for the layout engine code :)

    if  len(self._children) > 1 or \
       ( len(self._children) and \
         self._children[0]._type != 'GRSection' ):

      temp = self._children[:]
      self._children = []
      section = GRSection(self)
      section._children = temp
      section._buildObject()
      
    elif len(self._children) == 0:
      raise GRExceptions.NoLayoutDefined, \
        "The layout section does not contain any instructions. What do I do?"

    self._xmlchildnamespace = self._findNamespace()
    self.getParent ()._namespace = self._xmlchildnamespace


  # Find the xml namespace in use by any child passthru objects
  def _findNamespace(self, object=None):
    if object==None: object = self
    for child in object._children:
      try:
        if child._xmlnamespace:
          namespace =  child._xmlnamespace
        else:
          namespace = self._findNamespace(child)
        if namespace:
          return namespace
      except AttributeError:
        pass


  #
  # Prepare for running
  #
  def prepare(self, mapper, parameters):

    self._mapper = mapper
    self._parameters = parameters
    self.walk(self.__prepare)

  # Used by prepare()
  def __prepare(self, object):
    if isinstance(object, GRLayoutElement):

      object.prepare()

      # This all looks confusing, but basically it simply figures out
      # what section a field or summary actually pulls from and also
      # makes sure a section, field, or summary isn't trying to use
      # a source outside of the source's scope.  After it's satisfied
      # with the results, it adds the element to the mapper object.

      # TODO: This can probably be simplified by moving
      # TODO: the various functions to the actual child
      # TODO: object's phaseInits.

      parentSection = object.findParentOfType('GRSection',includeSelf=0)

      if isinstance(object, GRSection):
        if parentSection is None:
          self._mapper.addSection(object._name, object._source, None)
        else:
          self._mapper.addSection(object._name,
                                  object._source,
                                  parentSection._name)
        object._mymapper = self._mapper.sectionMap[object._name]
        object._mymapper._object = object

      elif isinstance(object, GRField):
        s = parentSection.getAncestorWithSource(object._source)
        if s == None:
          raise GRExceptions.SourceOutOfScope, \
                'Field "%s" in section "%s" uses out-of-scope source "%s"' \
                     % (object.name, parentSection._name, object._source)

        object._section = s._name

        if hasattr(object,'section') and object.section:
          sec = string.lower(object.section)
          curr = s
          section = None
          while curr:
            if curr._name == sec:
              section = curr
              break
            curr = curr.getParent ().findParentOfType('GRSection')

          if section == None:
            raise GRExceptions.SourceOutOfScope, \
                'Field "%s" in section "%s" uses out-of-scope section "%s"' \
                     % (object.name, parentSection._name, object.section)

          object._section = section._name

        object._mymapper = self._mapper.sectionMap[object._section]

        assert gDebug(6,'Mapping field "%s" to section "%s"' \
                   % (object.name, object._section))
        self._mapper.addFieldToSection(object._section, object.name)

      elif isinstance(object, GRSumm):
        if object.section == None:
          s = object.getParent ().findParentOfType('GRSection')
        else:
          sec = string.lower(object.section)
          s = self._mapper.sectionMap[sec]._object
        if s == None:
          raise GRExceptions.SourceOutOfScope, \
            'Summary "%s" in section "%s" uses out-of-scope source "%s"' \
                 % (object._field, object.getParent ().name, object._source)

        object._section = s._name
        object._mymapper = self._mapper.sectionMap[s._name]

        assert gDebug(6,'Mapping summary %s [%s] to section %s' \
                   % (object._field, object.function, object._section))
        self._mapper.addSummaryToSection(object.function,
                                 object._section, object._field, object._formula)

      elif isinstance(object, GRFormula):
        if object.section == None:
          s = object.getParent ().findParentOfType('GRSection')
        else:
          sec = string.lower(object.section)
          s = self._mapper.sectionMap[sec]._object
        if s == None:
          raise GRExceptions.SourceOutOfScope, \
            'Formula "%s" in section "%s" uses out-of-scope source "%s"' \
                 % (object.name, object.getParent ().name, object._source)

        object._section = s._name
        object._mymapper = self._mapper.sectionMap[s._name]

        assert gDebug(6,'Mapping formula "%s" to section "%s"' \
                   % (object.name, object._section))
        self._mapper.addFormulaToSection(object._section, object.name, object)


  #
  # Process layout and dump output.
  # self.prepare *must* be called prior to processing
  #
  def process(self, dest, includeStructuralComments=0):

##    global structuralComment
##    if includeStructuralComments:
##      structuralComment = _structuralComment

    for child in self._children:
      ## structuralComment(dest, '<!-- [layout] -->')
      child.processAsController(dest, self._mapper)
      ## structuralComment(dest, '<!-- [/layout] -->\n')



