
#include "qtc.h"

volatile sig_atomic_t exit_requested;

void warn_memory()
{
    warnx(_("Out of memory"));
}

void err_bug(const char * const file, const int line)
{
    errx(1, "Bug! File: [%s] Line: [%d]", file, line);
}

int safe_write(const int fd, const void *buf_, size_t count)
{
    const char *buf = (const char *) buf_;
    ssize_t written;
    struct pollfd pfd;
    pfd.fd = fd;
    pfd.events = POLLOUT;
        
    while (count > (size_t) 0) {
        for (;;) {
            if ((written = write(fd, buf, count)) <= (ssize_t) 0) {
                if (errno == EAGAIN) {
                    (void) poll(&pfd, 1, -1);
                } else if (errno != EINTR) {
                    return -1;
                }
                continue;
            }
            break;
        }
        buf += written;
        count -= written;
    }
    return 0;
}

void init_client_shared_data(ClientsSharedData * const csd)
{
    csd->nb_listening_sockets = 0U;
    csd->first_client = NULL;
    csd->mutex_clients_list = PTHREAD_MUTEX_INITIALIZER;
}

int prepare_listeners(ClientsSharedData * const csd)
{
    struct addrinfo hints, *res, *res0;
    int error;
    int listening_socket;
    const char *cause = NULL;
    int on = 1;
    
    memset(&hints, 0, sizeof hints);
    hints.ai_family = PF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_flags = AI_PASSIVE;
    if ((error = getaddrinfo(NULL, DEFAULT_PORT, &hints, &res0)) != 0) {
        errx(1, "getaddrinfo: [%s]", gai_strerror(error));
    }
    res = res0;
    while (res != NULL && csd->nb_listening_sockets < MAX_LISTENING_SOCKETS) {
        listening_socket = socket(res->ai_family, res->ai_socktype,
                                  res->ai_protocol);
        if (listening_socket == -1) {
            cause = "socket";
            goto next;
        }
        setsockopt(listening_socket, SOL_SOCKET, SO_REUSEADDR,
                   &on, (socklen_t) sizeof on);
        if (bind(listening_socket, res->ai_addr, res->ai_addrlen) < 0) {
            cause = "bind";
            close(listening_socket);
            goto next;
        }
        if (listen(listening_socket, DEFAULT_BACKLOG) < 0) {
            cause = "listen";
            close(listening_socket);
            goto next;
        }
        csd->listening_sockets[csd->nb_listening_sockets]
            = listening_socket;
        csd->pfds[csd->nb_listening_sockets].fd = listening_socket;
        csd->pfds[csd->nb_listening_sockets].events = POLLIN;
        csd->nb_listening_sockets++;
        next:
        res = res->ai_next;
    }
    if (csd->nb_listening_sockets == 0U) {
        err(1, "%s", cause);
    }
    freeaddrinfo(res0);

    return 0;
}

int register_client(ClientsSharedData * const csd, Client * const client)
{
    if (csd->first_client == NULL) {
        csd->first_client = client;
        return 0;
    }
    client->next = csd->first_client;
    csd->first_client = client;
    
    return 0;
}

int unregister_client(ClientsSharedData * const csd, Client * const client)
{
    Client *scanned_client = csd->first_client;
    
    if (scanned_client == client) {
        csd->first_client = NULL;
        return 0;
    }
    while (scanned_client != NULL) {
        if (scanned_client->next == client) {
            scanned_client->next = client->next;
            break;
        }
        scanned_client = scanned_client->next;
    }
    return 0;
}

int http_error(const Client * const client, const char * const error)
{
    return safe_write(client->socket, error, strlen(error));
}

int send_reply_header(ClientsSharedData * const csd, Client * const client)
{
    (void) csd;
    return safe_write(client->socket, HTTP_REPLY_HEADER,
                      (sizeof HTTP_REPLY_HEADER) - (size_t) 1U);
}

int read_client_data(ClientsSharedData * const csd, Client * const client)
{
    char *read_buffer;
    size_t sizeof_read_buffer = HTTP_READ_BUFFER_SIZE;
    char *client_data = NULL;
    char *client_data0;
    size_t sizeof_client_data = (size_t) 0U;
    size_t wanted_sizeof_client_data;
    size_t wanted_sizeof_body = (size_t) 0U;
    size_t sizeof_body = (size_t) 0U;
    ssize_t readen;
    int header_received = 0;
    size_t body_pos = (size_t) 0U;
    char *found;
    char *body = NULL;
    
    if ((read_buffer = malloc(sizeof_read_buffer)) == NULL) {
        warn_memory();
        return -1;
    }
    for (;;) {
        readen = read(client->socket, read_buffer, sizeof_read_buffer);
        if (readen == (ssize_t) 0) {
            break;
        }
        if (readen < (ssize_t) 0) {
            if (errno == EINTR) {
                continue;
            }
            warn(_("read"));
            free(read_buffer);
            free(client_data);
            return -1;
        }
        wanted_sizeof_client_data = sizeof_client_data + readen +
            (size_t) HTTP_CLIENT_DATA_GAP_SIZE;
        if (wanted_sizeof_client_data > HTTP_MAX_CLIENT_DATA_SIZE) {
            warnx(_("Data too large"));
            http_error(client, HTTP_ERROR_TOO_LARGE);
            free(read_buffer);
            free(client_data);
            return -1;
        }
        if (wanted_sizeof_client_data <= sizeof_client_data) {
            err_bug(__FILE__, __LINE__);
        }
        client_data0 = client_data;
        if ((client_data = realloc(client_data, wanted_sizeof_client_data))
            == NULL) {
            client_data = client_data0;
            warn_memory();
            http_error(client, HTTP_ERROR_UNAVAILABLE);
            free(read_buffer);
            free(client_data);
            return -1;
        }
        memcpy(client_data + sizeof_client_data, read_buffer, readen);
        sizeof_client_data = sizeof_client_data + readen;
        if (wanted_sizeof_client_data - HTTP_CLIENT_DATA_GAP_SIZE
            != sizeof_client_data) {
            err_bug(__FILE__, __LINE__);
        }
        *(client_data + sizeof_client_data) = 0;
        if (header_received == 0) {
            if (sizeof_client_data > HTTP_MAX_CLIENT_HEADER_SIZE) {
                warnx(_("Header too long"));
                http_error(client, HTTP_ERROR_TOO_LARGE);
                free(read_buffer);
                free(client_data);
                return -1;
            }
            if ((found = strstr(client_data, HTTP_HEADER_BODY_SEP)) == NULL) {
                continue;
            }
            body_pos = (size_t) (found - client_data) +
                (sizeof HTTP_HEADER_BODY_SEP) - (size_t) 1U;
            if ((found = strcasestr(client_data, HTTP_CONTENT_LENGTH_HEADER))
                == NULL ||
                (size_t) (found - client_data) >= body_pos) {
                warnx(_("Missing content length"));
                http_error(client, HTTP_ERROR_LENGTH_REQUIRED);
                free(read_buffer);
                free(client_data);
                return -1;
            }
            found += (sizeof HTTP_CONTENT_LENGTH_HEADER) - (size_t) 1U;
            while (*found == ' ' || *found == '\t') {
                found++;
            }
            wanted_sizeof_body = (size_t) strtoul(found, NULL, 10);
            if (wanted_sizeof_body <= (size_t) 0U ||
                body_pos >= HTTP_MAX_CLIENT_DATA_SIZE ||
                wanted_sizeof_body >= HTTP_MAX_CLIENT_DATA_SIZE - body_pos) {
                warnx(_("Invalid content length"));
                http_error(client, HTTP_ERROR_LENGTH_REQUIRED);                
                free(read_buffer);
                free(client_data);
                return -1;                
            }
            header_received = 1;
            continue;
        }
        if (header_received == 0 || body_pos > sizeof_client_data ||
            wanted_sizeof_body <= (size_t) 0U) {
            err_bug(__FILE__, __LINE__);
        }
        sizeof_body = sizeof_client_data - body_pos;
        if (sizeof_body >= wanted_sizeof_body) {
            break;
        }
    }
    free(read_buffer);
    read_buffer = NULL;
    if (header_received == 0) {
        warnx(_("No header"));
        http_error(client, HTTP_ERROR_BAD_REQUEST);
        free(client_data);
        return -1;                
    }
    if (sizeof_body <= (size_t) 0U) {
        warnx(_("No data"));
        http_error(client, HTTP_ERROR_PRECONDITION_FAILED);
        free(client_data);
        return -1;
    }
    if (body_pos >= sizeof_client_data ||
        body_pos + sizeof_body > sizeof_client_data) {
        err_bug(__FILE__, __LINE__);
    }
    body = client_data + body_pos;
    if (sizeof_body > wanted_sizeof_body) {
        warnx(_("Content should have been [%lu] bytes long, "
                "but [%lu] bytes were received. Truncating."),
              (unsigned long) wanted_sizeof_body,
              (unsigned long) sizeof_body);
        sizeof_body = wanted_sizeof_body;
        *(body + sizeof_body) = 0;
    } else if (sizeof_body < wanted_sizeof_body) {
        warnx(_("We were waiting for [%lu] bytes, "
                "but only [%lu] bytes were received. Aborting."),
              (unsigned long) wanted_sizeof_body,
              (unsigned long) sizeof_body);
        http_error(client, HTTP_ERROR_PRECONDITION_FAILED);
        free(client_data);
        return -1;
    }
    if (*(body + sizeof_body) != 0) {
        err_bug(__FILE__, __LINE__);
    }
    if (send_reply_header(csd, client) != 0) {
        warnx(_("Unable to send any reply to the client"));
        free(client_data);
        return -1;
    }
    free(client_data);
    return 0;
}

void *client_thread(void * const client_thread_data_)
{
    ClientThreadData * const client_thread_data =
        (ClientThreadData * const) client_thread_data_;
    ClientsSharedData *csd = client_thread_data->csd;
    Client *client = client_thread_data->client;
    
    free(client_thread_data);
    pthread_mutex_lock(&csd->mutex_clients_list);
    register_client(csd, client);
    pthread_mutex_unlock(&csd->mutex_clients_list);
    read_client_data(csd, client);
    close(client->socket);
    client->socket = -1;
    pthread_mutex_lock(&csd->mutex_clients_list);
    unregister_client(csd, client);
    pthread_mutex_unlock(&csd->mutex_clients_list);
    free(client);

    return NULL;
}

void init_client(ClientsSharedData * const csd, Client * const client,
                 const int socket)
{
    (void) csd;
    client->next = NULL;    
    client->socket = socket;
    memset(&client->thread, 0, sizeof client->thread);
}

int create_client_thread(ClientsSharedData * const csd,
                         const int socket)
{
    Client *client;
    ClientThreadData *client_thread_data;

    if (exit_requested != (sig_atomic_t) 0) {
        close(socket);
        return -1;
    }    
    if ((client_thread_data = malloc(sizeof *client_thread_data)) == NULL) {
        warn_memory();
        return -1;        
    }
    if ((client = malloc(sizeof *client)) == NULL) {
        warn_memory();
        free(client_thread_data);
        return -1;
    }
    init_client(csd, client, socket);
    client_thread_data->csd = csd;
    client_thread_data->client = client;
    if (pthread_create(&client->thread, NULL, client_thread,
                       client_thread_data) != 0) {
        warn("pthread_create");
        http_error(client, HTTP_ERROR_UNAVAILABLE);
        free(client_thread_data);
        close(socket);
        return -1;
    }    
    return 0;
}

int accept_client(ClientsSharedData * const csd, const int listening_socket)
{
    struct sockaddr_storage remote_addr;
    socklen_t sizeof_remote_addr = sizeof remote_addr;
    int client_socket;

    for(;;) {
        if ((client_socket = accept(listening_socket,
                                    (struct sockaddr *) &remote_addr,
                                    &sizeof_remote_addr)) != -1) {
            break;
        }
        if (errno == EINTR) {
            continue;
        }
        warn("listen");
        return -1;
    }
    return create_client_thread(csd, client_socket);
}

int watch_listening_sockets(ClientsSharedData * const csd)
{
    int nb_events_;
    unsigned int nb_events;
    unsigned int t;
    
    if ((nb_events_ = poll(csd->pfds, csd->nb_listening_sockets,
                           LISTENER_TIMEOUT * 1000)) == -1) {
        if (errno == EINTR) {
            return 0;
        }
        warn("poll error");
        return -1;
    }
    nb_events = (unsigned int) nb_events_;
    if (nb_events == 0U) {
        return 0; /* Time out */
    }
    t = 0U;
    do {
        if (csd->pfds[t].revents == 0 ||
            (csd->pfds[t].revents & (POLLERR | POLLHUP | POLLNVAL)) != 0) {
            continue;
        }
        accept_client(csd, csd->listening_sockets[t]);
    } while (++t < csd->nb_listening_sockets);

    return 0;
}

void sigaction_exit_requested(const int signal)
{
    (void) signal;
    exit_requested = (sig_atomic_t) 1;
}

int trap_signals(void)
{
    struct sigaction action;
    
    sigemptyset(&action.sa_mask);
    action.sa_flags = SA_NOCLDSTOP | SA_NOCLDWAIT;
    action.sa_handler = sigaction_exit_requested;
    sigaction(SIGINT, &action, NULL);
    sigaction(SIGHUP, &action, NULL);
    sigaction(SIGTERM, &action, NULL);
    
    return 0;
}

int wait_for_running_threads(ClientsSharedData * const csd)
{
    Client *scanned_client;
    pthread_t scanned_thread;

    for(;;) {
        pthread_mutex_lock(&csd->mutex_clients_list);        
        if ((scanned_client = csd->first_client) == NULL) {
            pthread_mutex_unlock(&csd->mutex_clients_list);            
            break;
        }        
        scanned_thread = scanned_client->thread;
        pthread_mutex_unlock(&csd->mutex_clients_list);
        pthread_join(scanned_thread, NULL);
    }
    return 0;
}

int main(void)
{   
    ClientsSharedData csd;
    
    init_client_shared_data(&csd);
    prepare_listeners(&csd);
    trap_signals();
    while (exit_requested == (sig_atomic_t) 0) {
        watch_listening_sockets(&csd);
    }
    wait_for_running_threads(&csd);
    
    return 0;
}
