#include "udm_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <regex.h>
#include <string.h>
#include <ctype.h>
#ifdef   HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_WINSOCK_H
#include <winsock.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif


#include "udm_utils.h"
#include "udm_common.h"
#include "udm_charset.h"
#include "udm_spell.h"
#include "udm_db.h"
#include "udm_xmalloc.h"

#define MAXNORMLEN 56
#ifndef MSG_WAITALL
#define MSG_WAITALL 0
#endif


static int cmpspell(const void *s1,const void *s2){
  int lc;
  lc = strcmp(((const UDM_SPELL*)s1)->lang,((const UDM_SPELL*)s2)->lang);
  if (lc == 0)
	return(strcmp(((const UDM_SPELL*)s1)->word,((const UDM_SPELL*)s2)->word));
  else return lc;
}

/* backward string compaire for suffix tree operations */
static int strbcmp(const char *s1, const char *s2) { 
  int l1 = strlen(s1)-1, l2 = strlen(s2)-1;
  while (l1 >= 0 && l2 >= 0) {
    if (s1[l1] < s2[l2]) return -1;
    if (s1[l1] > s2[l2]) return 1;
    l1--;
    l2--;
  }
  if (l1 < l2) return -1;
  if (l1 > l2) return 1;

  return 0;
}
static int strbncmp(const char *s1, const char *s2, size_t count) { 
  int l1 = strlen(s1) - 1, l2 = strlen(s2) - 1, l = count;
  while (l1 >= 0 && l2 >= 0 && l > 0) {
    if (s1[l1] < s2[l2]) return -1;
    if (s1[l1] > s2[l2]) return 1;
    l1--;
    l2--;
    l--;
  }
  if (l == 0) return 0;
  if (l1 < l2) return -1;
  if (l1 > l2) return 1;
  return 0;
}
static int cmpaffix(const void *s1,const void *s2){
  int lc;
  if (((const UDM_AFFIX*)s1)->type < ((const UDM_AFFIX*)s2)->type) return -1;
  if (((const UDM_AFFIX*)s1)->type > ((const UDM_AFFIX*)s2)->type) return 1;
  lc = strcmp(((const UDM_AFFIX*)s1)->lang,((const UDM_AFFIX*)s2)->lang);
  if (lc == 0)
    if (((const UDM_AFFIX*)s1)->type == 'p')
	return(strcmp(((const UDM_AFFIX*)s1)->repl,((const UDM_AFFIX*)s2)->repl));
    else return(strbcmp(((const UDM_AFFIX*)s1)->repl,((const UDM_AFFIX*)s2)->repl));
  else return lc;
}

int UdmAddSpell(UDM_ENV * Conf,const char * word,const char *flag,const char *lang){
	if(Conf->nspell>=Conf->mspell){
		if(Conf->mspell){
			Conf->mspell+=1024*20;
			Conf->Spell=(UDM_SPELL *)UdmXrealloc(Conf->Spell,Conf->mspell*sizeof(UDM_SPELL));
		}else{
			Conf->mspell=1024*20;
			Conf->Spell=(UDM_SPELL *)UdmXmalloc(Conf->mspell*sizeof(UDM_SPELL));
		}
	}
	Conf->Spell[Conf->nspell].word=strdup(word);
	strncpy(Conf->Spell[Conf->nspell].flag,flag,10);
	strncpy(Conf->Spell[Conf->nspell].lang,lang,2);
	Conf->Spell[Conf->nspell].lang[3] = 0;
	Conf->nspell++;
	return(0);
}


__INDLIB__ int UdmImportDictionary(UDM_ENV * Conf,const char *lang,const char *filename,int skip_noflag,const char *first_letters){
	unsigned char str[BUFSIZ];	
	FILE *dict;

	if(!(dict=fopen(filename,"r")))return(1);
	while(fgets(str,sizeof(str),dict)){
		unsigned char *s;
		const unsigned char *flag;

	        flag = NULL;
		if((s=strchr(str,'/'))){
			*s=0;
			s++;flag=s;
			while(*s){
				if((*s>='A')&&(*s<='Z'))s++;
				else{
					*s=0;
					break;
				}
			}
		}else{
			if(skip_noflag)	continue;
			flag="";
		}
		UdmTolower(str,Conf->local_charset);
		/* Dont load words if first letter is not required */
		/* It allows to optimize loading at  search time   */
		if(*first_letters)
			if(!strchr(first_letters,str[0]))
				continue;
		s=str;
		while(*s){
			if(*s=='\r')*s=0;
			if(*s=='\n')*s=0;
			s++;
		}
		UdmAddSpell(Conf,str,flag,lang);
	}
	fclose(dict);
	return(0);
}


__INDLIB__ int UdmDBImportDictionary(UDM_AGENT * Indexer,const char *lang,const char *filename, int dump){
	unsigned char str[BUFSIZ];
	unsigned char *s;
	const unsigned char *flag=NULL;
	int rej=0;
	int imp=0;
	FILE *dict;

	if(!(dict=fopen(filename,"r")))return(1);
	while(fgets(str,sizeof(str),dict)){
		flag=NULL;
		if((s=strchr(str,'/'))){
			*s=0;s++;flag=s;
			while(*s){
				if(*s>='A'&&*s<='Z')s++;
				else{
					*s=0;
					break;
				}
			}
		}
		
		UdmTolower(str,Indexer->Conf->local_charset);
		
		s=str;
		while(*s){
			if(*s=='\r')*s=0;
			if(*s=='\n')*s=0;
			s++;
		}
		
		if(dump){
			printf("INSERT INTO spell (word,flag,lang) VALUES ('%s','%s','%s');\n",str,flag?flag:(const unsigned char*)"",lang);
		}else{
			if(UdmInsertSpell(Indexer,flag,lang,str)) {
				rej++;
				printf("InsertSpell %d: %s\n",rej,UdmDBErrorMsg(Indexer->db));
			}
			else
				imp++;
		}

	}
	fclose(dict);
	if(!dump)
	printf("%d words imported, %d errors\n",imp,rej);
	return(0);
}

static UDM_SPELL * UdmFindWord(UDM_AGENT * Indexer, const char *word, int affixflag) {
int l,c,r,resc,resl,resr, nlang = Indexer->spellang, li, li_from, li_to, i;

        if (nlang == -1) {
	  li_from = 0; li_to = Indexer->Conf->nLang;
	} else {
	  li_from = nlang; li_to = li_from + 1;
	}
	if ((!(Indexer->Conf->ispell_mode & UDM_ISPELL_MODE_DB)) ||
	    (Indexer->Conf->nspell && (Indexer->Conf->ispell_mode & UDM_ISPELL_MODE_DB))) {
	    
	  i = (int)(*word) & 255;
	  for(li = li_from; li < li_to; li++) {
	    l = Indexer->Conf->SpellTree[li].Left[i];
	    r = Indexer->Conf->SpellTree[li].Right[i];
	    if (l == -1) continue;
	    while(l<=r){
		    c = (l + r) >> 1;
		    resc = strcmp(Indexer->Conf->Spell[c].word, word);
		    if( (resc == 0) && 
			((affixflag == NULL) || (strchr(Indexer->Conf->Spell[c].flag, affixflag) != NULL)) ) {
		      return(&Indexer->Conf->Spell[c]);
		    }
		    resl = strcmp(Indexer->Conf->Spell[l].word, word);
		    if( (resl == 0) && 
			((affixflag == NULL) || (strchr(Indexer->Conf->Spell[l].flag, affixflag) != NULL)) ) {
		      return(&Indexer->Conf->Spell[l]);
		    }
		    resr = strcmp(Indexer->Conf->Spell[r].word, word);
		    if( (resr == 0) && 
			((affixflag == NULL) || (strchr(Indexer->Conf->Spell[r].flag, affixflag) != NULL)) ) {
		      return(&Indexer->Conf->Spell[r]);
		    }
		    if(resc < 0){
			    l = c + 1;
			    r--;
		    } else if(resc > 0){
			    r = c - 1;
			    l++;
		    } else {
		      l++;
		      r--;
		    }
	    }
	  }
	} else {
	    return (UDM_SPELL *)UdmFindWordDB(Indexer,word);
	}
	return(NULL);
}


int UdmAddAffix(UDM_ENV * Conf,int flag,const char * lang,const char *mask,const char *find,const char *repl,int type) {
#define ERRSTRSIZE 100

	if ((!(Conf->ispell_mode & UDM_ISPELL_USE_PREFIXES)) &&
		(type=='p')) return(0);
	
	if(Conf->naffixes>=Conf->maffixes){
		if(Conf->maffixes){
			Conf->maffixes+=16;
			Conf->Affix = UdmXrealloc(Conf->Affix,Conf->maffixes*sizeof(UDM_AFFIX));
		}else{
			Conf->maffixes=16;
			Conf->Affix = UdmXmalloc(Conf->maffixes*sizeof(UDM_AFFIX));
		}
	}
	if (type=='s') {
	    sprintf(((UDM_AFFIX*)(Conf->Affix))[Conf->naffixes].mask,"%s$",mask);
	} else {
	    sprintf(((UDM_AFFIX*)(Conf->Affix))[Conf->naffixes].mask,"^%s",mask);
	}
	((UDM_AFFIX*)(Conf->Affix))[Conf->naffixes].compile = 1;
	((UDM_AFFIX*)(Conf->Affix))[Conf->naffixes].flag=flag;
	((UDM_AFFIX*)(Conf->Affix))[Conf->naffixes].type=type;
	strcpy(((UDM_AFFIX*)(Conf->Affix))[Conf->naffixes].lang,lang);
	((UDM_AFFIX*)(Conf->Affix))[Conf->naffixes].lang[2] = 0;
	
	strcpy(((UDM_AFFIX*)(Conf->Affix))[Conf->naffixes].find,find);
	strcpy(((UDM_AFFIX*)(Conf->Affix))[Conf->naffixes].repl,repl);
	((UDM_AFFIX*)(Conf->Affix))[Conf->naffixes].replen=strlen(repl);
	Conf->naffixes++;
	return(0);
}

static char * remove_spaces(char *dist,char *src){
char *d,*s;
	d=dist;
	s=src;
	while(*s){
		if(*s!=' '&&*s!='-'&&*s!='\t'){
			*d=*s;
			d++;
		}
		s++;
	}
	*d=0;
	return(dist);
}


__INDLIB__ int UdmImportAffixes(UDM_ENV * Conf,const char *lang, const char *filename, UDM_AGENT *Indexer, int dump){
unsigned char str[BUFSIZ];
unsigned char flag=0;
unsigned char mask[BUFSIZ]="";
unsigned char find[BUFSIZ]="";
unsigned char repl[BUFSIZ]="";
unsigned char *s;
int i;
int suffixes=0;
int prefixes=0;
int imp=0;
int rej=0;
FILE *affix;

	    if(!(affix=fopen(filename,"r")))
		    return(1);

	    while(fgets(str,sizeof(str),affix)){
		    if(!UDM_STRNCASECMP(str,"suffixes")){
		    	    suffixes=1;
			    prefixes=0;
			    continue;
		    }
		    if(!UDM_STRNCASECMP(str,"prefixes")){
			    suffixes=0;
			    prefixes=1;
			    continue;
		    }
		    if(!UDM_STRNCASECMP(str,"flag ")){
			    s=str+5;
			    while(strchr("* ",*s))s++;
			    flag=*s;
			    continue;
		    }
		    if((!suffixes)&&(!prefixes))continue;
		
		    if((s=strchr(str,'#')))*s=0;
		    if(!*str)continue;

		    UdmTolower(str,Conf->local_charset);
		    strcpy(mask,"");
		    strcpy(find,"");
		    strcpy(repl,"");

		    i=sscanf(str,"%[^>\n]>%[^,\n],%[^\n]",mask,find,repl);

		    remove_spaces(str,repl);strcpy(repl,str);
		    remove_spaces(str,find);strcpy(find,str);
		    remove_spaces(str,mask);strcpy(mask,str);

		    switch(i){
		    case 3:break;
		    case 2:
			    if(*find!='-'){
				    strcpy(repl,find);
				    strcpy(find,"");
			    }
			    break;
		    default:
			    continue;
		    }
		
		    UdmAddAffix(Conf,(int)flag,lang,mask,find,repl,suffixes?'s':'p');
		
		    if (Indexer != NULL) {
			if(dump){
			    if (suffixes) {
				printf("INSERT INTO affix (flag,type,lang,mask,find,repl) VALUES ('%c','%s','%s','%s$','%s','%s');\n",flag,"s",lang,mask,find,repl);
			    } else {
				printf("INSERT INTO affix (flag,type,lang,mask,find,repl) VALUES ('%c','%s','%s','^%s','%s','%s');\n",flag,"p",lang,mask,find,repl);
			    }
			}else{
			    if(UdmInsertAffix(Indexer,(int)flag,lang,mask,find,repl,suffixes?"s":"p")) {
				    rej++;
				    printf("InsertAffix %d: %s\n",rej,UdmDBErrorMsg(Indexer->db));
			    }
			    else
				    imp++;
			}
		    }
	    }
	    fclose(affix);
	    if ((!dump) && (Indexer != NULL)) printf("%d rules imported, %d errors\n",imp,rej);
	    
	    return(0);
}

__INDLIB__ void UdmSortDictionary(UDM_ENV * Conf){
  int  j, CurLet = -1, Let;size_t i;
  char *CurLang = NULL;

        qsort((void*)Conf->Spell,Conf->nspell,sizeof(UDM_SPELL),cmpspell);
	for(i = 0; i < Conf->nspell; i++) {
	  if (CurLang == NULL || strncmp(CurLang, Conf->Spell[i].lang, 2) != 0) {
	    CurLang = Conf->Spell[i].lang;
	    strncpy(Conf->SpellTree[Conf->nLang].lang, CurLang, 2);
	    Conf->SpellTree[Conf->nLang].lang[3] = 0;
	    for(j = 0; j < 256; j++)
	      Conf->SpellTree[Conf->nLang].Left[j] =
		Conf->SpellTree[Conf->nLang].Right[j] = -1;
	    if (Conf->nLang > 0) {
	      CurLet = -1;
	    }
	    Conf->nLang++;
	  }
	  Let = (int)(*(Conf->Spell[i].word)) & 255;
	  if (CurLet != Let) {
	    Conf->SpellTree[Conf->nLang-1].Left[Let] = i;
	    CurLet = Let;
	  }
	  Conf->SpellTree[Conf->nLang-1].Right[Let] = i;
	}
}

__INDLIB__ void UdmSortAffixes(UDM_ENV * Conf) {
  int  j, CurLetP = -1, CurLetS = -1, Let, cl = -1;
  char *CurLangP = NULL, *CurLangS = NULL;
  UDM_AFFIX *Affix; size_t i;
  
  if (Conf->naffixes > 1)
    qsort((void*)Conf->Affix,Conf->naffixes,sizeof(UDM_AFFIX),cmpaffix);
  for(i = 0; i < Conf->nLang; i++)
    for(j = 0; j < 256; j++) {
      Conf->PrefixTree[i].Left[j] = Conf->PrefixTree[i].Right[j] = -1;
      Conf->SuffixTree[i].Left[j] = Conf->SuffixTree[i].Right[j] = -1;
    }

  for(i = 0; i < Conf->naffixes; i++) {
    Affix = &(((UDM_AFFIX*)Conf->Affix)[i]);
    if(Affix->type == 'p') {
      if (CurLangP == NULL || strcmp(CurLangP, Affix->lang) != 0) {
	cl = -1;
	for (j = 0; j < Conf->nLang; j++) {
	  if (strncmp(Conf->SpellTree[j].lang, Affix->lang, 2) == 0) {
	    cl = j;
	    break;
	  }
	}
	CurLangP = Affix->lang;
	strcpy(Conf->PrefixTree[cl].lang, CurLangP);
	CurLetP = -1;
      }
      if (cl < 0) continue; /* we have affixes without spell for this lang */
      Let = (int)(*(Affix->repl)) & 255;
      if (CurLetP != Let) {
	Conf->PrefixTree[cl].Left[Let] = i;
	CurLetP = Let;
      }
      Conf->PrefixTree[cl].Right[Let] = i;
    } else {
      if (CurLangS == NULL || strcmp(CurLangS, Affix->lang) != 0) {
	cl = -1;
	for (j = 0; j < Conf->nLang; j++) {
	  if (strcmp(Conf->SpellTree[j].lang, Affix->lang) == 0) {
	    cl = j;
	    break;
	  }
	}
	CurLangS = Affix->lang;
	strcpy(Conf->SuffixTree[cl].lang, CurLangS);
	CurLetS = -1;
      }
      if (cl < 0) continue; /* we have affixes without spell for this lang */
      Let = (int)(Affix->repl[Affix->replen-1]) & 255;
      if (CurLetS != Let) {
	Conf->SuffixTree[cl].Left[Let] = i;
	CurLetS = Let;
      }
      Conf->SuffixTree[cl].Right[Let] = i;
    }
  }
}

static char * CheckSuffix(const char *word, size_t len, UDM_AFFIX *Affix, int *res, UDM_AGENT *Indexer) {
  regmatch_t subs[1];
  char newword[2*MAXNORMLEN] = "";
  int err;
  
  *res = strbncmp(word, Affix->repl, Affix->replen);
  if (*res < 0) {
    return NULL;
  }
  if (*res > 0) {
    return NULL;
  }
  strcpy(newword, word);
  strcpy(newword+len-Affix->replen, Affix->find);

  if (Affix->compile) {
    err = regcomp(&(Affix->reg),Affix->mask,REG_EXTENDED|REG_ICASE|REG_NOSUB);
    if(err){
      /*regerror(err, &(Affix->reg), regerrstr, ERRSTRSIZE);*/
      regfree(&(Affix->reg));
      return(NULL);
    }
    Affix->compile = 0;
  }
  if(!(err=regexec(&(Affix->reg),newword,1,subs,0))){
    UDM_SPELL * curspell;

    if((curspell=UdmFindWord(Indexer, newword, Affix->flag))){
    /*  if(strchr(curspell->flag,Affix->flag)){*/
	
        return strdup(newword);
    /*  }*/
    }
  }
  return NULL;
}

#define NS 1
#define MAX_NORM 512
static int CheckPrefix(const char *word, size_t len, UDM_AFFIX *Affix, UDM_AGENT *Indexer, int li, int pi,
		char **forms, char ***cur ) {
  regmatch_t subs[NS];
  char newword[2*MAXNORMLEN] = "";
  int err, ls,cs,rs, lres,rres,cres, res;
  size_t newlen;
  UDM_AFFIX *CAffix = Indexer->Conf->Affix;
  
  res = strncmp(word, Affix->repl, Affix->replen);
  if (res != 0) {
    return res;
  }
  strcpy(newword, Affix->find);
  strcat(newword, word+Affix->replen);

  if (Affix->compile) {
    err = regcomp(&(Affix->reg),Affix->mask,REG_EXTENDED|REG_ICASE|REG_NOSUB);
    if(err){
      /*regerror(err, &(Affix->reg), regerrstr, ERRSTRSIZE);*/
      regfree(&(Affix->reg));
      return (0);
    }
    Affix->compile = 0;
  }
  if(!(err=regexec(&(Affix->reg),newword,1,subs,0))){
    UDM_SPELL * curspell;

    if((curspell=UdmFindWord(Indexer, newword, Affix->flag))){
      if ((*cur - forms) < (MAX_NORM-1)) {
	**cur =  strdup(newword);
	(*cur)++; **cur = NULL;
      }
    } 
    newlen = strlen(newword);
    ls = Indexer->Conf->SuffixTree[li].Left[pi];
    rs = Indexer->Conf->SuffixTree[li].Right[pi];
    while (ls >= 0 && ls <= rs) {
      cs = (ls + rs) >> 1; /*cs = ls;*/
      if ((*cur - forms) < (MAX_NORM-1)) {
	**cur = CheckSuffix(newword, newlen, &CAffix[cs], &cres, Indexer);
	if (**cur) {
	  (*cur)++; **cur = NULL;
	}
      }
      if ( (ls < cs) && ((*cur - forms) < (MAX_NORM-1)) ) {
	**cur = CheckSuffix(newword, newlen, &CAffix[ls], &lres, Indexer);
	if (**cur) {
	  (*cur)++; **cur = NULL;
	}
      }
      if ( (rs > cs) && ((*cur - forms) < (MAX_NORM-1)) ) {
	**cur = CheckSuffix(newword, newlen, &CAffix[rs], &rres, Indexer);
	if (**cur) {
	  (*cur)++; **cur = NULL;
	}
      }
      if (cres < 0) {
	rs = cs - 1;
	ls++;
      } else if (rres > 0) {
	ls = cs + 1;
	rs--;
      } else {
	ls++;
	rs--;
      }
    }
  }
  return 0;
}


__INDLIB__ char ** UdmNormalizeWord(UDM_AGENT * Indexer,const char *word){
/*regmatch_t subs[NS];*/
size_t len;
char ** forms;
char **cur;
UDM_AFFIX * Affix;
int ri, pi, lp, rp, cp, ls, rs, cs, nlang = Indexer->spellang;
int li, li_from, li_to, lres, cres, rres;

	len=strlen(word);
	if (len < Indexer->Conf->min_word_len 
		|| len > MAXNORMLEN
		|| len > Indexer->Conf->max_word_len
		)
		return(NULL);

	forms=(char **) UdmXmalloc(MAX_NORM*sizeof(char **));
	cur=forms;*cur=NULL;

	ri = (int)(*word) & 255;
	pi = (int)(word[strlen(word)-1]) & 255;
	if (nlang == -1) {
	  li_from = 0; li_to = Indexer->Conf->nLang;
	} else {
	  li_from  = nlang;
	  li_to = nlang + 1;
	}
	Affix=(UDM_AFFIX*)Indexer->Conf->Affix;

	/* Check that the word itself is normal form */
	if(UdmFindWord(Indexer, word, NULL)){
		*cur=strdup(word);
		cur++;*cur=NULL;
	}

	/* Find all other NORMAL forms of the 'word' */

	for (li = li_from; li < li_to; li++) {
	  /* check prefix */
	  lp = Indexer->Conf->PrefixTree[li].Left[ri];
	  rp = Indexer->Conf->PrefixTree[li].Right[ri];
	  while (lp >= 0 && lp <= rp) {
	    cp = (lp + rp) >> 1;
	    if ((cur - forms) < (MAX_NORM-1)) {
	      cres = CheckPrefix(word, len, &Affix[cp], Indexer, li, pi, forms, &cur);
	    }
	    if ((lp < cp) && ((cur - forms) < (MAX_NORM-1)) ) {
	      lres = CheckPrefix(word, len, &Affix[lp], Indexer, li, pi, forms, &cur);
	    }
	    if ( (rp > cp) && ((cur - forms) < (MAX_NORM-1)) ) {
	      rres = CheckPrefix(word, len, &Affix[rp], Indexer, li, pi, forms, &cur);
	    }
	    if (cres < 0) {
	      rp = cp - 1;
	      lp++;
	    } else if (rres > 0) {
	      lp = cp + 1;
	      rp--;
	    } else {
	      lp++;
	      rp--;
	    }
	  }

	  /* check suffix */
	  ls = Indexer->Conf->SuffixTree[li].Left[pi];
	  rs = Indexer->Conf->SuffixTree[li].Right[pi];
	  while (ls >= 0 && ls <= rs) {
	    cs = (ls + rs) >> 1;  /*cs = ls;*/
	    if ((cur - forms) < (MAX_NORM-1)) {
	      *cur = CheckSuffix(word, len, &Affix[cs], &cres, Indexer);
	      if (*cur) {
		cur++; *cur = NULL;
	      }
	    }
	    if ( (ls < cs) && ((cur - forms) < (MAX_NORM-1)) ) {
	      *cur = CheckSuffix(word, len, &Affix[ls], &lres, Indexer);
	      if (*cur) {
		cur++; *cur = NULL;
	      }
	    }
	    if ( (rs > cs) && ((cur - forms) < (MAX_NORM-1)) ) {
	      *cur = CheckSuffix(word, len, &Affix[rs], &rres, Indexer);
	      if (*cur) {
		cur++; *cur = NULL;
	      }
	    }
	    if (cres < 0) {
	      rs = cs - 1;
	      ls++;
	    } else if (rres > 0) {
	      ls = cs + 1;
	      rs--;
	    } else {
	      ls++;
	      rs--;
	    }
	  } /* end while */
	  
	} /* for li */

	if(cur==forms){
		free(forms);
		return(NULL);
	}
	return(forms);
}

__INDLIB__ int UdmSetIspellMode(UDM_ENV * Env,int mode){
	Env->ispell_mode=mode;
	return(mode);
}

int UdmSelectSpellLang(UDM_ENV *Conf, char *lang) {
  int i;
  for(i=0; i < Conf->nLang; i++) {
    if(strcmp(lang, Conf->SpellTree[i].lang) == 0) {
      return i;
    }
  }
  return -1;
}



void UdmSelectLang(UDM_AGENT * Indexer, char *lang) {
	size_t i;
	char *s;
	/* Convert to lower case */
	for(s=lang;*s;*s=tolower(*s),s++);
	Indexer->spellang = UdmSelectSpellLang(Indexer->Conf, lang);
	for (i = 0; i < Indexer->nlangs; i++) {
		if (!UDM_STRNCMP(Indexer->lang[i].lang, lang)) {
			Indexer->curlang = i;
			return;
		}
	}
	if (i < UDM_LANGPERDOC) {
		strncpy(Indexer->lang[i].lang, lang, 3);
		Indexer->lang[i].count = 0;
		Indexer->curlang = i;
		Indexer->nlangs = i + 1;
	}
}

ssize_t recvall(int s, void *buf, size_t len, int flags) {
  size_t received = 0;
  char *b = buf;
  while ((received += recv(s, &b[received], len - received, flags)) < len);
  return received;
}

int UdmLoadSpellFromServer(UDM_ENV *Conf, char *hostname) {
  struct hostent *hp;
  struct sockaddr_in server_addr;
  int i,j,s;
  size_t wlen;
  const char *hello = "HELO\0";
  UDM_AFFIX *Affix;

  if ((hp = gethostbyname(hostname)) == 0 ) {
    return 1;
  }
  bzero(&server_addr, sizeof(server_addr));
  memmove(hp->h_addr, &server_addr.sin_addr, (size_t)hp->h_length);
  server_addr.sin_family = hp->h_addrtype;
  server_addr.sin_port = htons(UDM_SPELL_PORT);

  if((s = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
    return 2;
  }
  
  if(connect(s, (struct sockaddr *)&server_addr, sizeof(server_addr)) == -1) {
    close(s);
    return 3;
  }

  send (s, hello, 4, 0);

  if (recvall(s, &Conf->nLang, sizeof(Conf->nLang), MSG_WAITALL) < 0) {
    close(s);
    return 4;
  }

  
  if (recvall(s, (void *)&Conf->naffixes, sizeof(Conf->naffixes), MSG_WAITALL) < 0) {
    close(s);
    return 4;
  }
  Conf->maffixes = Conf->naffixes;
  Conf->Affix = UdmXmalloc(Conf->naffixes * sizeof(UDM_AFFIX));
  Affix = (UDM_AFFIX *)Conf->Affix;
  for(i = 0; i < Conf->naffixes; i++) {
    if (recvall(s, &Affix[i], sizeof(UDM_AFFIX), MSG_WAITALL) < 0) {
      close(s);
      return 4;
    }
    regcomp(&Affix[i].reg, Affix[i].mask, REG_EXTENDED|REG_ICASE);
  }
  if (recvall(s, (void *)&Conf->nspell, sizeof(Conf->nspell), MSG_WAITALL) < 0) {
    close(s);
    return 4;
  }
  Conf->mspell = Conf->nspell;
  Conf->Spell = (UDM_SPELL *) UdmXmalloc(Conf->nspell * sizeof(UDM_SPELL));
  for(j = 0; j < Conf->nspell; j++) {
    if (recvall(s, &Conf->Spell[j], sizeof(UDM_SPELL), MSG_WAITALL) < 0) {
      close(s);
      return 4;
    }
    if (recvall(s, &wlen, sizeof(int), MSG_WAITALL) < 0) {
      close(s);
      return 4;
    }
    Conf->Spell[j].word = (char *)UdmXmalloc(wlen + 1);
    if (recvall(s, Conf->Spell[j].word, wlen, MSG_WAITALL) < 0) {
      close(s);
      return 4;
    }
    Conf->Spell[j].word[wlen] = 0;
  }

  for(i = 0; i < Conf->nLang; i++) {
    if (recvall(s, &Conf->PrefixTree[i], sizeof(Tree_struct), MSG_WAITALL) < 0) {
      close(s);
      return 4;
    }
    if (recvall(s, &Conf->SuffixTree[i], sizeof(Tree_struct), MSG_WAITALL) < 0) {
      close(s);
      return 4;
    }
    if (recvall(s, &Conf->SpellTree[i], sizeof(Tree_struct), MSG_WAITALL) < 0) {
      close(s);
      return 4;
    }
  }

  close(s);

  return 0;
}

__INDLIB__ char ** UdmNormalizeWordFromServer(UDM_AGENT *Indexer, const char *word) {
  struct hostent *hp;
  struct sockaddr_in server_addr;
  char **forms, **cur;
  int s;
  size_t len;
  const char *hello = "CLIL\0";
  const char  *lang = "  ";
  char buf[1024];

  len=strlen(word);
	if (len < Indexer->Conf->min_word_len 
		|| len > MAXNORMLEN
		|| len > Indexer->Conf->max_word_len
		)
		return(NULL);

  if ((hp = gethostbyname(Indexer->Conf->spellhost)) == 0 ) {
    return NULL;
  }
  bzero(&server_addr, sizeof(server_addr));
  memmove(hp->h_addr, &server_addr.sin_addr, (size_t)hp->h_length);
  server_addr.sin_family = hp->h_addrtype;
  server_addr.sin_port = htons(UDM_SPELL_PORT);

  if((s = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
    return NULL;
  }
  
  if(connect(s, (struct sockaddr *)&server_addr, sizeof(server_addr)) == -1) {
    close(s);
    return NULL;
  }

  forms=(char **) UdmXmalloc(MAX_NORM*sizeof(char **));
  cur=forms;*cur=NULL;

  send (s, hello, 4, 0);
  send (s, lang, 2, 0);   /* Fixme: lang selection from form */
  send (s, &len, sizeof(len), 0); /* word length */
  send (s, word, len, 0); /* word */

  do {
    if (recvall(s, &len, sizeof(len), MSG_WAITALL) < 0) {
      close(s);
      free(forms);
      return NULL;
    }
    if (len > 0) {
      if (recvall(s, buf, len, MSG_WAITALL) < 0) {
	close(s);
	free(forms);
	return NULL;
      }
      buf[len] = 0;
      *cur = strdup(buf);
      cur++;
      *cur = NULL;
    }

  } while (len > 0);

  close(s);

  if(cur==forms){
    free(forms);
    return(NULL);
  }
  return(forms);

}

__INDLIB__ void UdmFreeIspell (UDM_ENV *Conf) {
  int i;
  UDM_AFFIX *Affix = (UDM_AFFIX *)Conf->Affix;

  for (i = 0; i < Conf->naffixes; i++) {
    if (Affix[i].compile == 0) {
      regfree(&(Affix[i].reg));
    }
  }
  UDM_FREE(Conf->spellhost);
  UDM_FREE(Conf->Affix);
  UDM_FREE(Conf->Spell);
  Conf->Affix = NULL;
  Conf->Spell = NULL;
  Conf->nspell = Conf->naffixes = Conf->nLang = 0;
  return;
}
