/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <config.h>
#include <glib/gi18n.h>
#include <glib-object.h>
#include <string.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <dvdread/dvd_reader.h>
#include <dvdread/ifo_types.h>
#include <dvdread/ifo_read.h>
#include <dvdread/nav_read.h>


#include "dvd.h"
#include "dvd-time.h"
#include "dvd-marshal.h"

#undef DEBUG_TITLE_DISK

#ifdef DEBUG_TITLE_DISK
const gchar *DVD_VIDEO_FORMAT[] = {
	"mpeg1",
	"mpeg2"
};

const gchar *DVD_VIDEO_WIDTH[] = {
	"720",
	"704",
	"352",
};

const gchar *DVD_VIDEO_HEIGHT[] = {
	"480",
	"576",
	"240",
	"288"
};

const gchar *DVD_VIDEO_STANDARD[] = {
	"NTSC",
	"PAL"
};

const gchar *DVD_VIDEO_FPS[] = {
	"25",
	"29.97"
};

const gchar *DVD_VIDEO_ASPECT[] = {
	"4X3",
	"16X9"
};

const gchar *DVD_AUDIO_FORMAT[] = {
	"AC3",		/* 5.1ch 448bps */
	"UNKNOWN1",	/* ?? */
	"MPEG1",	/* 2ch 384bps */
	"MPEG2",	/* 7.1ch 912bps */
	"LPCM",		/* 8ch 6144bps*/
	"SDDS",		/* 7.1ch 1280bps [not in use] */
	"DTS"		/* 5.1ch 1536bps*/
};

const gchar *DVD_AUDIO_TRACK[] = {
	"UNDEFINED",
	"NORMAL",
	"IMPAIRED",
	"COMMENTS1",
	"COMMENTS2",
	"UNKNOWN"
};

const gchar *DVD_AUDIO_SAMP[] = {
	"48000",
	"96000",
	"44100",
	"32000"
};

const gchar *DVD_AUDIO_QUANT[] = {
	"16BIT",
	"20BIT",
	"24BIT",
	"DRC",
};
#endif /* DEBUG_TITLE_DISK */

enum {
	TITLE_OUTPUT_CHAPTER,
	TITLE_TIME,
	TITLE_LAST_SIGNAL
};

guint dvd_title_disk_signals[TITLE_LAST_SIGNAL];

static DvdTitleClass *dvd_title_disk_parent_class = NULL;

static void     dvd_title_disk_class_init	(DvdTitleDiskClass *class);
static void     dvd_title_disk_instance_init	(GTypeInstance	 *instance,
						 gpointer	  g_class);
static void     dvd_title_disk_dispose		(GObject	 *object);
static void	output_chapter_cb		(DvdChapterDisk	 *chapter,
						 DvdStream	  stream_type,
						 gint		  track,
						 gint		  bytes,
						 guint8		 *buffer,
						 guint32	  pts,
						 guint64	  frame_clocks,
						 DvdTitleDisk	 *title);
static void	time_chapter_cb			(DvdChapterDisk	 *chapter,
						 const DvdTime	 *cell_time,
						 const DvdTime	 *chapter_time,
						 DvdTitleDisk	 *title);
/* imported by dvd-disk.c */
gboolean	dvd_title_disk_read_properties	(DvdTitleDisk	 *title,
						 dvd_reader_t	 *dvd_handle,
						 const gchar	 *device,
						 guint8		  title_id,
						 GError		**error);
/* import from dvd-chapter-disc.c */
extern gboolean	dvd_chapter_disk_read_properties(DvdChapterDisk	 *chapter,
						 guint8		  chapter_id,
						 dvd_reader_t	 *dvd_handle,
						 const gchar	 *device,
						 guint8		  title_id,
						 GError		**error);
extern gboolean	dvd_chapter_disk_read_chapter	(DvdChapterDisk	 *chapter,
						 dvd_reader_t	 *dvd_handle,
						 GError		**error);
static void
dvd_title_disk_class_init	(DvdTitleDiskClass	*class)
{
	GObjectClass *object_class = (GObjectClass *) class;
	dvd_title_disk_signals[TITLE_OUTPUT_CHAPTER] =
		g_signal_new ("output_chapter",
			      G_TYPE_FROM_CLASS(class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (DvdTitleDiskClass, output_chapter),
			      NULL, NULL,
			      dvd_marshal_VOID__INT_INT_INT_POINTER_UINT_UINT64,
			      G_TYPE_NONE,
			      6,
			      G_TYPE_INT,
			      G_TYPE_INT,
			      G_TYPE_INT,
			      G_TYPE_POINTER,
			      G_TYPE_UINT,
			      G_TYPE_UINT64);
	dvd_title_disk_signals[TITLE_TIME] =
		g_signal_new ("title_time",
			      G_TYPE_FROM_CLASS(class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (DvdTitleDiskClass, title_time),
			      NULL, NULL,
			      dvd_marshal_VOID__POINTER_POINTER_POINTER,
			      G_TYPE_NONE,
			      3,
			      G_TYPE_POINTER,
			      G_TYPE_POINTER,
			      G_TYPE_POINTER);
	dvd_title_disk_parent_class = g_type_class_ref (DVD_TITLE_TYPE);
	object_class->dispose = dvd_title_disk_dispose;
}

static void
dvd_title_disk_instance_init	(GTypeInstance	*instance,
				 gpointer	 g_class)
{
	DvdTitleDisk *title;
	
	title = DVD_TITLE_DISK (instance);
	title->op_mutex = g_mutex_new ();
	DVD_TITLE (title)->source = DVD_SOURCE_DISK;
}

static void
dvd_title_disk_dispose		(GObject	*object)
{
	DvdTitleDisk *title = DVD_TITLE_DISK (object);

	g_free (title->device);

	G_OBJECT_CLASS (dvd_title_disk_parent_class)->dispose (G_OBJECT (title));
	g_mutex_free (title->op_mutex);
}

/**
 * Gets the GType for the dvd title object class
 * @returns GType.
 */
GType
dvd_title_disk_get_type		(void)
{
	static GType title_type = 0;

	if (title_type == 0) {
		GTypeInfo title_info = {
			sizeof (DvdTitleDiskClass),
			NULL,
			NULL,
			(GClassInitFunc) dvd_title_disk_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (DvdTitleDisk),
			0, /* n_preallocs */
			(GInstanceInitFunc) dvd_title_disk_instance_init
	    	};
		title_type = g_type_register_static (DVD_TITLE_TYPE,
						    "DvdTitleDisk",
						    &title_info, 0);
	}
	return title_type;
}

/**
 * Create a new dvd title object with #DvdSource source.
 * You must call g_object_unref() when you are done with the returned object.
 * @returns A #DvdTitleDisk object.
 */
DvdTitleDisk
*dvd_title_disk_new		(void)
{
	DvdTitleDisk *title;
	
	title = g_object_new (dvd_title_disk_get_type (), NULL);
	
	return title;
}

/* imported by dvd-disk.c */
gboolean
dvd_title_disk_read_properties	(DvdTitleDisk	 *title_disk,
				 dvd_reader_t	 *dvd_handle,
				 const gchar	 *device,
				 guint8		  title_id,
				 GError		**error)
{
	ifo_handle_t *ifo_zero;
	ifo_handle_t *ifo_title;
	pgcit_t *vts_pgcit;
	vtsi_mat_t *vtsi_mat;
	vmgi_mat_t *vmgi_mat;
	video_attr_t *video_attr;
	pgc_t *pgc;
	gint audio_no;
	gint chapters;
	gint chapter_no;
	gint vts_ttn;
	guint32 start_ts = 0;
	DvdTitle *title;
	DvdVideo *video;
	
	g_assert (title_disk != NULL);
	g_assert (dvd_handle != NULL);
	g_assert (device != NULL);
	
	title = DVD_TITLE (title_disk);
	ifo_zero = ifoOpen (dvd_handle, 0);
	if (ifo_zero == NULL) {
		g_warning ("Can't open main ifo!");
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_DISK_OPEN,
			     _("Unable to open the DVD video manager. "
			       "The disk may be dirty, damaged, or may not be a DVD video disk."));
		return FALSE;
	}
	
	title_disk->title_set = ifo_zero->tt_srpt->title[title_id].title_set_nr;
	ifo_title = ifoOpen (dvd_handle, title_disk->title_set);
	if (ifo_title == NULL) {
		g_warning ("Can't open title %d ifo (title set %d)", title_id, title_disk->title_set);
		ifoClose (ifo_zero);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_DISK_OPEN,
			     _("Unable to open DVD title set information file for reading. "
			       "The disk may be dirty or damaged."));
		return FALSE;
	} else if (ifo_title->vtsi_mat == NULL) {
		ifoClose (ifo_title);
		ifoClose (ifo_zero);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_DISK_OPEN,
			     _("Unable to open DVD title set information file for reading. "
			       "The disk may be dirty or damaged."));
		return FALSE;
	}

	vtsi_mat   = ifo_title->vtsi_mat;
	vts_pgcit  = ifo_title->vts_pgcit;
	video_attr = &vtsi_mat->vts_video_attr;
	vts_ttn	   = ifo_zero->tt_srpt->title[title_id].vts_ttn;
	vmgi_mat   = ifo_zero->vmgi_mat;
	pgc	   = vts_pgcit->pgci_srp[ifo_title->vts_ptt_srpt->title[vts_ttn - 1].ptt[0].pgcn - 1].pgc;

	/* title */
	chapters	     = ifo_zero->tt_srpt->title[title_id].nr_of_ptts;
	title->time          = dvd_time_new_from_dvd_read_time (&pgc->playback_time);
	title->audio_tracks  = vtsi_mat->nr_of_vts_audio_streams;
	/*g_message ("%d audio tracks", title->audio_tracks);*/
	
	/* video */
	video = dvd_video_new ();
	dvd_video_set_letterboxed (video, video_attr->letterboxed);
	dvd_video_set_film_mode (video, video_attr->film_mode);
	switch (video_attr->bit_rate > 0) {
	case 0:
		dvd_video_set_bit_rate (video, DVD_VIDEO_BIT_RATE_VARIABLE);
	case 1:
		dvd_video_set_bit_rate (video, DVD_VIDEO_BIT_RATE_CONSTANT);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
	switch (video_attr->mpeg_version) {
	case 0:
		dvd_video_set_format (video, DVD_VIDEO_FORMAT_MPEG1);
		break;
	case 1:
		dvd_video_set_format (video, DVD_VIDEO_FORMAT_MPEG2);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
	switch (video_attr->video_format) {
	case 0:
		dvd_video_set_standard (video, DVD_VIDEO_STANDARD_NTSC);
		break;
	case 1:
		dvd_video_set_standard (video, DVD_VIDEO_STANDARD_PAL);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
	switch (video_attr->display_aspect_ratio) {
	case 0:
		dvd_video_set_aspect (video, DVD_VIDEO_ASPECT_4X3);
		break;
	case 1:
	case 2:
		g_assert_not_reached ();
		break;
	case 3:
		dvd_video_set_aspect (video, DVD_VIDEO_ASPECT_16X9);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
	/* video_attr->picture_size
	Resolution NTSC (PAL)
	0 = 720x480 (720x576)
	1 = 704x480 (704x576)
	2 = 352x480 (352x576)
	3 = 352x240 (352x288)
	*/
	if (dvd_video_get_standard (video) == DVD_VIDEO_STANDARD_NTSC) {
		switch (video_attr->picture_size) {
		case 0:
			dvd_video_set_width (video, DVD_VIDEO_WIDTH_720);
			dvd_video_set_height (video, DVD_VIDEO_HEIGHT_480);
			break;
		case 1:
			dvd_video_set_width (video, DVD_VIDEO_WIDTH_704);
			dvd_video_set_height (video, DVD_VIDEO_HEIGHT_480);
			break;
		case 2:
			dvd_video_set_width (video, DVD_VIDEO_WIDTH_352);
			dvd_video_set_height (video, DVD_VIDEO_HEIGHT_480);
			break;
		case 3:
			dvd_video_set_width (video, DVD_VIDEO_WIDTH_352);
			dvd_video_set_height (video, DVD_VIDEO_HEIGHT_240);
			break;
		default:
			g_assert_not_reached ();
			break;
		}
	} else {
		switch (video_attr->picture_size) {
		case 0:
			dvd_video_set_width (video, DVD_VIDEO_WIDTH_720);
			dvd_video_set_height (video, DVD_VIDEO_HEIGHT_576);
			break;
		case 1:
			dvd_video_set_width (video, DVD_VIDEO_WIDTH_704);
			dvd_video_set_height (video, DVD_VIDEO_HEIGHT_576);
			break;
		case 2:
			dvd_video_set_width (video, DVD_VIDEO_WIDTH_352);
			dvd_video_set_height (video, DVD_VIDEO_HEIGHT_576);
			break;
		case 3:
			dvd_video_set_width (video, DVD_VIDEO_WIDTH_352);
			dvd_video_set_height (video, DVD_VIDEO_HEIGHT_288);
			break;
		default:
			g_assert_not_reached ();
			break;
		}
	}
	switch ((pgc->playback_time.frame_u & 0xc0) >> 6) {
	case 0:
		g_assert_not_reached ();
		break;
	case 1:
		dvd_video_set_fps (video, DVD_VIDEO_FPS_25);
		break;
	case 2:
		g_assert_not_reached ();
		break;
	case 3:
		dvd_video_set_fps (video, DVD_VIDEO_FPS_30);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
	title->video = video;
#ifdef DEBUG_TITLE_DISK
	g_message ("Title %d (title %d in title set %d)", title_id + 1, vts_ttn, title_disk->title_set);
	
	g_message ("video type     = %s", DVD_VIDEO_FORMAT[dvd_video_get_format (video)]);
	g_message ("video width    = %s", DVD_VIDEO_WIDTH[dvd_video_get_width (video)]);
	g_message ("video height   = %s", DVD_VIDEO_HEIGHT[dvd_video_get_height (video)]);
	g_message ("video standard = %s", DVD_VIDEO_STANDARD[dvd_video_get_standard (video)]);
	g_message ("video aspect   = %s", DVD_VIDEO_ASPECT[dvd_video_get_aspect (video)]);
	g_message ("video fps      = %s", DVD_VIDEO_FPS[dvd_video_get_fps (video)]);
	g_message ("chapters       = %d", title->chapter_count);
	g_message ("letterboxed    = %d", dvd_video_get_letterboxed (video));
	g_message ("film mode      = %d", dvd_video_get_film_mode (video));
#endif /* DEBUG_TITLE_DISK */

	/* audio's */
	for (audio_no = 0;
	     audio_no < title->audio_tracks;
	     audio_no++) {
		audio_attr_t *audio_attr;
		DvdAudio *audio;

#ifdef DEBUG_TITLE_DISK
		g_message ("reading audio track %d", audio_no);
#endif /* DEBUG_TITLE_DISK */

		audio		= dvd_audio_new ();
		audio_attr	= &vtsi_mat->vts_audio_attr[audio_no];
		dvd_audio_set_lang (audio, dvd_language_get_from_code16 (audio_attr->lang_code));
		dvd_audio_set_format (audio, audio_attr->audio_format);
		dvd_audio_set_channels (audio, audio_attr->channels + 1);
		dvd_audio_set_number (audio, audio_no);
		dvd_audio_set_purpose (audio, audio_attr->code_extension);
		dvd_audio_set_application (audio, audio_attr->application_mode);
		dvd_audio_set_quant (audio, audio_attr->quantization);
		dvd_audio_set_samplerate (audio, audio_attr->sample_frequency);
		title->audio[audio_no] = audio;

#ifdef DEBUG_TITLE_DISK
		g_message ("audio lang     = %s", dvd_language_get_name (dvd_audio_get_lang (audio)));
		g_message ("audio type     = %s", DVD_AUDIO_FORMAT[dvd_audio_get_format (audio)]);
		g_message ("audio channels = %d", dvd_audio_get_channels (audio));
		g_message ("audio purpose  = %s", DVD_AUDIO_TRACK[dvd_audio_get_purpose (audio)]);
		g_message ("audio quant    = %s", DVD_AUDIO_QUANT[dvd_audio_get_quant (audio)]);
		g_message ("audio samp     = %s", DVD_AUDIO_SAMP[dvd_audio_get_samplerate (audio)]);
#endif /* DEBUG_TITLE_DISK */

	}
	ifoClose (ifo_title);
	ifoClose (ifo_zero);
	title_disk->blocks = 0;
	for (chapter_no = 0;
	     chapter_no < chapters;
	     chapter_no++) {
		DvdChapterDisk *chapter;
		
		chapter = dvd_chapter_disk_new ();
		g_message ("scanning chapter %d of %d", chapter_no + 1, chapters);
		dvd_chapter_set_id (DVD_CHAPTER (chapter), chapter_no);
		
		
		if (dvd_chapter_disk_read_properties (chapter,
						      chapter_no,
						      dvd_handle,
						      device,
						      title_id,
						      error) == TRUE) {
			DVD_CHAPTER (chapter)->start_time = dvd_time_new_from_time_stamp (start_ts);
			start_ts += DVD_CHAPTER (chapter)->time->ts;
			title_disk->blocks += chapter->blocks;
			dvd_title_append_chapter (title, DVD_CHAPTER (chapter));
		} else {
			g_warning ("Failed to scan chapter %d title %d", chapter_no, title->title_id);
			g_object_unref (G_OBJECT (chapter));
			return FALSE;
		}
	}
	if (title_disk->blocks < 1) {
		g_warning ("Unable to determine title %d block count", title_id);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_DISK_READ,
			     _("Unable to determine title size."));
		return FALSE;
	}
	/* Rough estimate */
	title->kbs = (title_disk->blocks - (title_disk->blocks / 145)) * 2048;

#ifdef DEBUG_TITLE_DISK
	g_message ("%d audio tracks", title->audio_tracks);
	g_message ("chapters = %d", title->chapter_count);
	g_message ("length   = %dh:%dm:%d.%02ds", title->time->bk_hours,
					  title->time->bk_minutes,
					  title->time->bk_seconds,
					  title->time->bk_milliseconds);
	g_message ("Est kb   = %d\n\n", title->kbs);
#endif /* DEBUG_TITLE_DISK */

	if (title->time->ts > start_ts) {
		g_warning ("title time=%lu is greater than chapter times=%lu.", title->time->ts, start_ts);
	}
	if (title->time->ts < start_ts) {
		g_warning ("title time=%lu is less than chapter times=%lu.", title->time->ts, start_ts);
	}
	title_disk->device = g_strdup (device);
	title->title_id = title_id;
	return TRUE;
}

/**
 * Gets the number of disk blocks used by the titles vob files.
 * @returns Number of disk blocks.
 */
guint32
dvd_title_disk_get_blocks	(DvdTitleDisk	 *title)
{
	g_assert (title != NULL);
	
	return (guint32) title->blocks;
}

/**
 * Reads the title data from disk or file and emits data through the #DvdTitleDiskClass::output_chapter signal handler.
 * Also outputs time information through the #DvdTitleDiskClass::title_time signal handler.
 * @Note You will most likely need to run this in a separate thread to allow the gui to be updated.
 * @param title A #DvdTitleDisk.
 * @param error The return location for an allocated GError, or NULL to ignore errors.
 * @returns TRUE if successful, FALSE on fail.
 */
gboolean
dvd_title_disk_read		(DvdTitleDisk	 *title,
				 GError		**error)
{
	return dvd_title_disk_read_chapters (title, 1,
					     DVD_TITLE (title)->chapter_count,
					     error);
}

static void
output_chapter_cb		(DvdChapterDisk	*chapter,
				 DvdStream	 stream_type,
				 gint		 track,
				 gint		 bytes,
				 guint8		*buffer,
				 guint32	 pts,
				 guint64	 frame_clocks,
				 DvdTitleDisk	*title)
{
	g_signal_emit (G_OBJECT (title),
		       dvd_title_disk_signals[TITLE_OUTPUT_CHAPTER],
		       0,
	 	       stream_type,
	 	       track,
	 	       bytes,
	 	       buffer,
	 	       pts,
	 	       frame_clocks);
}

static void
time_chapter_cb			(DvdChapterDisk	*chapter,
				 const DvdTime	*cell_time,
				 const DvdTime	*chapter_time,
				 DvdTitleDisk	*title)
{
	guint32 elapsed_ts;
	
	elapsed_ts = title->start_ts + chapter_time->ts;
	title->elapsed_time = dvd_time_set_from_time_stamp (title->elapsed_time, elapsed_ts);

	g_signal_emit (G_OBJECT (title),
		       dvd_title_disk_signals[TITLE_TIME],
		       0,
	 	       cell_time,
	 	       chapter_time,
	 	       title->elapsed_time);
}

/**
 * Reads the title data from disk or file and emits data through the #DvdTitleDiskClass::output_chapter signal handler.
 * Also outputs time information through the #DvdTitleDiskClass::title_time signal handler.
 * @Note You will most likely need to run this in a separate thread to allow the gui to be updated.
 * @param start_chapter A valid start chapter starting from 1. Read will start from beginning of chapter.
 * @param end_chapter A valid end chapter that is greater than specified start chapter. Read ends at end of chapter.
 * @param title A #DvdTitleDisk.
 * @param error The return location for an allocated GError, or NULL to ignore errors.
 * @returns TRUE if successful, FALSE on fail.
 */
gboolean
dvd_title_disk_read_chapters		(DvdTitleDisk	 *title,
					 guint8		  start_chapter,
					 guint8		  end_chapter,
					 GError		**error)
{
	gboolean ok = FALSE;
	dvd_reader_t *dvd_handle;
	gulong output_signal;
	gulong time_signal;
	guint32 start_ts;
	guint8 chapter;
	const DvdTime *start_dvdtime;
	DvdDemux *demux;
	
	g_assert (title != NULL);
	g_assert (start_chapter <= end_chapter);
	g_assert (end_chapter <= DVD_TITLE (title)->chapter_count);
	g_assert (title->device != NULL);
	
	g_mutex_lock (title->op_mutex);
	if (title->op_canned == TRUE) {
		title->op_reading = FALSE;
		g_mutex_unlock (title->op_mutex);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_CANCELLED,
			     _("The read operation has been cancelled"));
		return FALSE;
	}
	title->op_reading = TRUE;
	g_mutex_unlock (title->op_mutex);
	dvd_handle = DVDOpen (title->device);
	if (dvd_handle == NULL) {
		g_warning ("Can't open disc %s!", title->device);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_DISK_OPEN,
			     _("Failed to open DVD disk. "
			       "There may not be a disk in the drive or the disk is unreadable"));
		return FALSE;
	}
	title->op_chapter = DVD_CHAPTER_DISK (dvd_title_get_chapter (DVD_TITLE (title), start_chapter));
	g_assert (title->op_chapter != NULL);
	start_dvdtime = dvd_chapter_get_start_time (DVD_CHAPTER (title->op_chapter));
	g_object_unref (G_OBJECT (title->op_chapter));
	title->start_ts = start_dvdtime->ts;
	/*g_message ("starting at %d", title->start_ts);*/
	start_ts = start_dvdtime->ts;
	title->elapsed_time = dvd_time_new_from_time_stamp (start_ts);
	/*g_message ("dvd-title-priv reading chapters");*/
	demux = dvd_demux_new ();
	for (chapter = start_chapter;
	     chapter <= end_chapter;
	     chapter++) {
		
		g_mutex_lock (title->op_mutex);
		if (title->op_canned == TRUE) {
			g_free (title->elapsed_time);
			title->start_ts = 0;
			g_signal_emit (G_OBJECT (title),
				       dvd_title_disk_signals[TITLE_OUTPUT_CHAPTER],
				       0,
				       DVD_STREAM_VOB,
				       0,
				       -1,
				       NULL,
				       0);
			DVDClose (dvd_handle);
			g_mutex_unlock (title->op_mutex);
			g_object_unref (G_OBJECT (demux));
			g_set_error (error,
				     DVD_ERROR,
				     DVD_ERROR_CANCELLED,
				     _("The operation has been cancelled"));
			return FALSE;
		}
		title->op_chapter = DVD_CHAPTER_DISK (dvd_title_get_chapter (DVD_TITLE (title), chapter));
		g_assert (title->op_chapter != NULL);
		output_signal = g_signal_connect (title->op_chapter,
						  "output_cell",
						  G_CALLBACK (output_chapter_cb),
						  title);
		time_signal = g_signal_connect (title->op_chapter,
						"chapter_time",
						G_CALLBACK (time_chapter_cb),
						title);
		/* turn chapter reading on */
		g_mutex_lock (title->op_chapter->op_mutex);
		title->op_chapter->op_reading = TRUE;
		g_mutex_unlock (title->op_chapter->op_mutex);
		g_mutex_unlock (title->op_mutex);
		
		/* read chapter */
		title->op_chapter->demux = demux;
		ok = dvd_chapter_disk_read_chapter (title->op_chapter, dvd_handle, error);
		title->op_chapter->demux = NULL;
		g_mutex_lock (title->op_mutex);
		
		/* turn chapter reading off */
		g_mutex_lock (title->op_chapter->op_mutex);
		title->op_chapter->op_reading = FALSE;
		g_mutex_unlock (title->op_chapter->op_mutex);
		
		g_signal_handler_disconnect (title->op_chapter, output_signal);
		g_signal_handler_disconnect (title->op_chapter, time_signal);
		if (DVD_CHAPTER (title->op_chapter)->start_time->ts != start_ts) {
			g_warning ("chapter start time = %d and start_ts %d", DVD_CHAPTER (title->op_chapter)->start_time->ts, start_ts);
		}
		start_ts += DVD_CHAPTER (title->op_chapter)->time->ts;
		title->start_ts = start_ts;
		/* callback discards all (-1 length) end of cells */
		g_signal_emit (G_OBJECT (title),
			       dvd_title_disk_signals[TITLE_OUTPUT_CHAPTER],
			       0,
	 		       DVD_STREAM_VOB,
	 		       0,
	 		       -1,
	 		       NULL,
	 		       0);
	 	g_mutex_unlock (title->op_mutex);
	 	g_object_unref (G_OBJECT (title->op_chapter));
	 	if (ok == FALSE) {
			break;
		}
	}
	g_object_unref (G_OBJECT (demux));
	g_free (title->elapsed_time);
	DVDClose (dvd_handle);
	title->start_ts = 0;
	g_mutex_lock (title->op_mutex);
	title->op_reading = FALSE;
	g_mutex_unlock (title->op_mutex);
	return ok;
}

/**
 * Stops the current read process invoked by #dvd_title_disk_read ().
 * @Note The read emits signals that may update your gui and will be 
 * running in a separate thread, so remember to leave gdk threads first!
 * @param title A #DvdTitleDisk.
 * @returns void.
 */
void
dvd_title_disk_stop_read	(DvdTitleDisk	*title)
{
	gboolean stopped;
	
	g_assert (title != NULL);
	
	g_mutex_lock (title->op_mutex);
	if (title->op_reading == FALSE) {
		g_warning ("dvd_title_disk_stop_read called and title is not reading");
		g_mutex_unlock (title->op_mutex);
		return;
	}
	g_assert (title->op_canned != TRUE);
	if (title->op_chapter != NULL) {
		dvd_chapter_disk_stop_read (title->op_chapter);
	} else {
		title->op_canned = TRUE;
	}
	g_mutex_unlock (title->op_mutex);
	
	/*g_message ("title waiting");*/
	/* wait untill stopped */
	for (stopped = FALSE;
	     stopped == FALSE;
	     g_usleep (100)) {
		g_mutex_lock (title->op_mutex);
		if (title->op_reading == FALSE) {
			stopped = TRUE;
			title->op_canned = FALSE;
		}
		g_mutex_unlock (title->op_mutex);
	}
	/*g_message ("title stopped");*/
}
