/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <config.h>
#include <glib-object.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <dvd.h>
#include "dvd-marshal.h"

static GObjectClass *dvd_decoder_parent_class = NULL;
static void     dvd_decoder_class_init	(DvdDecoderClass	*class);
static void     dvd_decoder_instance_init	(GTypeInstance	*instance,
					 gpointer	 g_class);
static void     dvd_decoder_dispose	(GObject	*object);

enum {
	DECODER_OUTPUT_FRAME,
	DECODER_LAST_SIGNAL
};

guint dvd_decoder_signals[DECODER_LAST_SIGNAL];

static void
dvd_decoder_class_init	(DvdDecoderClass *class)
{
	GObjectClass *object_class = (GObjectClass *) class;
	dvd_decoder_signals[DECODER_OUTPUT_FRAME] =
		g_signal_new ("output_frame",
			      G_TYPE_FROM_CLASS(class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (DvdDecoderClass, output_frame),
			      NULL, NULL,
			      dvd_marshal_VOID__INT_INT_INT_POINTER_UINT_UINT64,
			      G_TYPE_NONE,
			      6,
			      G_TYPE_INT,
			      G_TYPE_INT,
			      G_TYPE_INT,
			      G_TYPE_POINTER,
			      G_TYPE_UINT,
			      G_TYPE_UINT64);
	dvd_decoder_parent_class = g_type_class_ref (G_TYPE_OBJECT);
	object_class->dispose = dvd_decoder_dispose;
}

static void
dvd_decoder_instance_init(GTypeInstance	*instance,
			 gpointer	 g_class)
{
	DvdDecoder *decoder;
	
	decoder = DVD_DECODER (instance);
}

static void
dvd_decoder_dispose	(GObject	*object)
{
	DvdDecoder *decoder;
	
	decoder = DVD_DECODER (object);
	g_free (decoder->buffer);
	
	G_OBJECT_CLASS (dvd_decoder_parent_class)->dispose (G_OBJECT (decoder));
}

/**
 * dvd_decoder_get_type
 * @return The GType for the DvdDecoder class.
 */
GType
dvd_decoder_get_type	(void)
{
	static GType decoder_type = 0;

	if (decoder_type == 0) {
		GTypeInfo decoder_info = {
			sizeof (DvdDecoderClass),
			NULL,
			NULL,
			(GClassInitFunc) dvd_decoder_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (DvdDecoder),
			0, /* n_preallocs */
			(GInstanceInitFunc) dvd_decoder_instance_init
	    	};
		decoder_type = g_type_register_static (G_TYPE_OBJECT,
						     "DvdDecoder",
						     &decoder_info, 0);
	}
	return decoder_type;
}

DvdDecoder
*dvd_decoder_new		(DvdStream	  stream_type)
{
	DvdDecoder *decoder = NULL;
	
	switch (stream_type) {
	case DVD_STREAM_VOB:
	case DVD_STREAM_MPEG1_VIDEO:
	case DVD_STREAM_MPEG1_AUDIO:
	case DVD_STREAM_MPEG2_AUDIO:
	case DVD_STREAM_SDDS_AUDIO:
	case DVD_STREAM_SUB_PICTURE:
		/* does not exist yet */
		g_warning ("No decoder available for specified stream type, returning NULL decoder");
		decoder = NULL;
		break;
	case DVD_STREAM_MPEG2_VIDEO:
		decoder = g_object_new (dvd_decoder_m2v_get_type (), NULL);
		break;
	case DVD_STREAM_AC3_AUDIO:
		decoder = g_object_new (dvd_decoder_ac3_get_type (), NULL);
		g_message ("new ac3 decoder");
		break;
	case DVD_STREAM_LPCM_AUDIO:
		decoder = g_object_new (dvd_decoder_lpcm_get_type (), NULL);
		break;
	case DVD_STREAM_DTS_AUDIO:
		decoder = g_object_new (dvd_decoder_dts_get_type (), NULL);
		break;
	default:
		g_assert_not_reached ();
	}
	return decoder;
}

DvdStream
dvd_decoder_get_stream_type	(DvdDecoder	 *decoder)
{
	DvdStream stream_type;
	
	g_assert (decoder != NULL);
	
	stream_type = decoder->stream_type;
	return stream_type;
}

void
dvd_decoder_set_track		(DvdDecoder	 *decoder,
				 guint8		  track)
{
	g_assert (decoder != NULL);
	g_assert (track < 8);
	
	decoder->track = track;
}

guint8
dvd_decoder_get_track		(DvdDecoder	 *decoder)
{
	guint8 track;
	
	g_assert (decoder != NULL);
	
	track = decoder->track;
	return track;
}

guint32
dvd_decoder_get_start_pts	(DvdDecoder	 *decoder)
{
	guint32 start_pts;
	
	g_assert (decoder != NULL);
	
	start_pts = decoder->start_pts;
	return start_pts;
}

guint32
dvd_decoder_get_current_pts	(DvdDecoder	 *decoder)
{
	guint32 current_pts;
	
	g_assert (decoder != NULL);
	
	current_pts = decoder->current_pts;
	return current_pts;
}

guint16
dvd_decoder_get_frames		(DvdDecoder	 *decoder)
{
	guint16 frames;
	
	g_assert (decoder != NULL);
	
	frames = decoder->frames;
	return frames;
}

DvdDecoderState
dvd_decoder_get_state		(DvdDecoder	 *decoder)
{
	DvdDecoderState state;
	
	g_assert (decoder != NULL);
	
	state = decoder->state;
	return state;
}

void
dvd_decoder_write		(DvdDecoder	 *decoder,
				 guint8		 *data,
				 guint16	  length,
				 guint32	  pts)
{
	guint8 *end;
	
	g_assert (decoder != NULL);
	g_assert (data != NULL);
	
	end = data + length;

	while (data < end) {
		switch (decoder->state) {
		case DVD_DECODER_STATE_STARTCODE:
			decoder->read_startcode (decoder, &data, end, pts);
			break;
		case DVD_DECODER_STATE_HEADER:
			decoder->read_header (decoder, &data, end, pts);
			break;
		case DVD_DECODER_STATE_FRAME:
			decoder->read_frame (decoder, &data, end, pts);
			break;
		default:
			g_assert_not_reached ();
		}
	}
#if 0
	/* for debugging decoders by writing 1 byte at a time */
	while (data < end) {
		switch (decoder->state) {
		case DVD_DECODER_STATE_STARTCODE:
			decoder->read_startcode (decoder, &data, data + 1, pts);
			break;
		case DVD_DECODER_STATE_HEADER:
			decoder->read_header (decoder, &data, data + 1, pts);
			break;
		case DVD_DECODER_STATE_FRAME:
			decoder->read_frame (decoder, &data, data + 1, pts);
			break;
		default:
			g_assert_not_reached ();
		}
	}
#endif
}

/* for children only! */
void
dvd_decoder_set_start_pts	(DvdDecoder	 *decoder,
				 guint32	  start_pts)
{
	decoder->start_pts = start_pts;
	decoder->have_start_pts = TRUE;
}

void
dvd_decoder_set_current_pts	(DvdDecoder	 *decoder,
				 guint32	  current_pts)
{
	/*g_message ("frame was %u clocks", current_pts - decoder->current_pts);*/
	decoder->current_pts = current_pts;
}

void
dvd_decoder_add_to_current_pts	(DvdDecoder	 *decoder,
				 guint32	  add_pts)
{
	/*g_message ("frame was %u clocks", add_pts);*/
	decoder->current_pts += add_pts;
}

void
dvd_decoder_set_frame_clocks	(DvdDecoder	 *decoder,
				 guint64	  frame_clocks)
{
	decoder->frame_clocks = frame_clocks;
}


void
dvd_decoder_set_state		(DvdDecoder	 *decoder,
				 DvdDecoderState  state)
{
	decoder->state = state;
}

void
dvd_decoder_discard_frame	(DvdDecoder	 *decoder)
{
	decoder->buffer_pos = 0;
}

void
dvd_decoder_discard_bytes	(DvdDecoder	 *decoder,
				 guint16	  bytes)
{
	if (decoder->buffer_pos > bytes) {
		decoder->buffer_pos -= bytes;
	} else {
		decoder->buffer_pos = 0;
	}
}

void
dvd_decoder_output_frame	(DvdDecoder	 *decoder)
{
	g_signal_emit (G_OBJECT (decoder),
			dvd_decoder_signals[DECODER_OUTPUT_FRAME], 0,
			decoder->stream_type,
			(gint) decoder->track,
			decoder->buffer_pos,
			decoder->buffer,
			decoder->current_pts,
			decoder->frame_clocks);
	/*g_message ("decoder pts %u clocks - format = %d packet length = %d packet clocks = %d\n",
		 decoder->current_pts,
		 decoder->stream_type,
		 decoder->buffer_pos,
		 decoder->frame_clocks / 300);*/
	decoder->frames += 1;
	decoder->buffer_pos = 0;
	/*decoder->state = DVD_DECODER_STATE_STARTCODE;*/
}

void
dvd_decoder_ensure_buffer_size	(DvdDecoder	 *decoder,
				 guint32	  buffer_size)
{
	if (decoder->buffer_size == 0) {
		decoder->buffer = g_malloc (buffer_size);
		decoder->buffer_size = buffer_size;
	} else if (decoder->buffer_size < buffer_size) {
		decoder->buffer = g_realloc (decoder->buffer, buffer_size);
		g_assert (decoder->buffer != NULL);
		decoder->buffer_size = buffer_size;
	} else {
		/* leave buffer alone */
	}
}

void
dvd_decoder_write_frame_byte	(DvdDecoder	 *decoder,
				 guint8		  byte)
{
	dvd_decoder_ensure_buffer_size (decoder, decoder->buffer_pos + 1);
	g_assert (decoder->buffer_size > decoder->buffer_pos);
		
	decoder->buffer[decoder->buffer_pos] = byte;
	decoder->buffer_pos += 1;
}

void
dvd_decoder_write_frame_bytes	(DvdDecoder	 *decoder,
				 const guint8	 *bytes,
				 guint32	  length)
{
	const guint8 *end;
	guint8 *ptr;
	
	dvd_decoder_ensure_buffer_size (decoder, decoder->buffer_pos + length);
	ptr = decoder->buffer + decoder->buffer_pos;
	end = bytes + length;
	while (bytes < end) {
		*ptr = *bytes;
		bytes += 1;
		ptr += 1;
	}
	decoder->buffer_pos += length;
}

