/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
/** $Revision: 1.01 $
 * Libdvd common header
 * @file dvd-chapter-disk.h DvdChapterDisk object header file
 * @ingroup core
 *
 * libdvd
 */

#if !defined (__DVD_H_INSIDE__) && !defined (LIBDVD_COMPILATION)
#error "Only <dvd.h> can be included directly."
#endif

#ifndef __DVD_CHAPTER_DISK_H__
#define __DVD_CHAPTER_DISK_H__


G_BEGIN_DECLS


/* --- type macros --- */
#define DVD_CHAPTER_DISK_TYPE			(dvd_chapter_disk_get_type())
#define DVD_CHAPTER_DISK(o)			(G_TYPE_CHECK_INSTANCE_CAST (o, DVD_CHAPTER_DISK_TYPE, DvdChapterDisk))
#define DVD_CHAPTER_DISK_CLASS(k)		(G_TYPE_CHECK_CLASS_CAST    (k, DVD_CHAPTER_DISK_TYPE, DvdChapterDiskClass))
#define DVD_CHAPTER_DISK_IS_CHAPTER(o)		(G_TYPE_CHECK_INSTANCE_TYPE (o, DVD_CHAPTER_DISK_TYPE))
#define DVD_CHAPTER_DISK_IS_CHAPTER_CLASS(k)	(G_TYPE_CHECK_CLASS_TYPE    (k, DVD_CHAPTER_DISK_TYPE))

/* --- typedefs & structures --- */
typedef struct	_DvdChapterDisk		DvdChapterDisk;
typedef struct	_DvdChapterDiskClass	DvdChapterDiskClass;

struct _DvdChapterDisk {
	DvdChapter	 parent;

	guint		 blocks;
	guint		 cell_count;
	GSList		*cells;
	gchar		*device;
	DvdDemux	*demux;
	/* For reading ops - always lock op_mutex before accessing op_* */
	GMutex		*op_mutex;
	gboolean	 op_reading;
	gboolean	 op_canned;
};

/** Dvd chapter disk class definition */
struct _DvdChapterDiskClass {
	DvdChapterClass	parent;
	/**
	 * output_cell Outputs all available data when chapter is being read.
	 * Bytes = -1 signals end of data stream.
	 * @param chapter Pointer to #DvdChapter emitting signal
	 * @param output #DvdStream data output type
	 * @param track Track data belongs to as a gint
	 * @param bytes Bytes output as a gint
	 * @param buffer Pointer to guint8 output buffer
	 * @returns void
	 */
	void (*output_cell)	(DvdChapter	*chapter,
				 DvdStream	 stream_type,
				 gint		 track,
				 gint		 bytes,
				 guint8		*buffer,
				 guint32	 pts,
				 guint64	 frame_clocks);
	/**
	 * chapter_time Outputs current cell and chapter times when chapter is being read
	 * @param chapter Pointer to #DvdChapter emitting signal
	 * @param cell_time Const pointer to current cell #DvdTime
	 * @param chapter_time Const pointer to current chapter #DvdTime
	 * @returns void
	 */
	void (*chapter_time)	(DvdChapter	*chapter,
				 const DvdTime	*cell_time,
				 const DvdTime	*chapter_time);
};

/* --- prototypes --- */
GType		dvd_chapter_disk_get_type	(void);
DvdChapterDisk*	dvd_chapter_disk_new		(void);
guint		dvd_chapter_disk_get_cell_count	(DvdChapterDisk	 *chapter);
G_CONST_RETURN
DvdTime*	dvd_chapter_disk_get_cell_time	(DvdChapterDisk	 *chapter,
						 guint		  cell_id);
gboolean	dvd_chapter_disk_read		(DvdChapterDisk	 *chapter,
						 GError		**error);
void		dvd_chapter_disk_stop_read	(DvdChapterDisk	 *chapter);

G_END_DECLS

#endif /*__DVD_CHAPTER_DISK_H__*/
