//  Copyright (C) 2007-2008  CEA/DEN, EDF R&D, OPEN CASCADE
//
//  Copyright (C) 2003-2007  OPEN CASCADE, EADS/CCR, LIP6, CEA/DEN,
//  CEDRAT, EDF R&D, LEG, PRINCIPIA R&D, BUREAU VERITAS
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
//
//  See http://www.salome-platform.org/ or email : webmaster.salome@opencascade.com
//
//  SMESH SMESH_I : idl implementation based on 'SMESH' unit's calsses
//  File   : SMESH_Gen_i.hxx
//  Author : Paul RASCLE, EDF
//  Module : SMESH
//
#ifndef _SMESH_GEN_I_HXX_
#define _SMESH_GEN_I_HXX_

#include "SMESH.hxx"

#include <SALOMEconfig.h>
#include CORBA_SERVER_HEADER(SMESH_Gen)
#include CORBA_SERVER_HEADER(SMESH_Mesh)
#include CORBA_SERVER_HEADER(SMESH_Hypothesis)
#include CORBA_CLIENT_HEADER(GEOM_Gen)
#include CORBA_CLIENT_HEADER(SALOMEDS)
#include CORBA_CLIENT_HEADER(SALOMEDS_Attributes)

#include "SMESH_Mesh_i.hxx"
#include "SMESH_Hypothesis_i.hxx"
#include "SALOME_Component_i.hxx"
#include "SALOME_NamingService.hxx"

#include "SMESH_Gen.hxx"
#include "GEOM_Client.hxx"

#include <TCollection_AsciiString.hxx>
#include <Resource_DataMapOfAsciiStringAsciiString.hxx>
#include <TColStd_HSequenceOfAsciiString.hxx>

#include <map>
#include <sstream>

class SMESH_Mesh_i;
class SALOME_LifeCycleCORBA;

// ===========================================================
// Study context - stores study-connected objects references
// ==========================================================
class SMESH_I_EXPORT StudyContext
{
public:
  // constructor
  StudyContext() {}
  // destructor
  ~StudyContext() 
  { 
    mapIdToIOR.clear();
    mapIdToId.clear();
  }
  // register object in the internal map and return its id
  int addObject( string theIOR )
  {
    int nextId = getNextId();
    mapIdToIOR[ nextId ]  = theIOR;
    return nextId;
  }
  // find the object id in the internal map by the IOR
  int findId( string theIOR )
  {
    map<int, string>::iterator imap;
    for ( imap = mapIdToIOR.begin(); imap != mapIdToIOR.end(); ++imap ) {
      if ( imap->second == theIOR )
        return imap->first;
    }
    return 0;
  }
  // get object's IOR by id
  string getIORbyId( const int theId )
  {
    if ( mapIdToIOR.find( theId ) != mapIdToIOR.end() )
      return mapIdToIOR[ theId ];
    return string( "" );
  }
  // get object's IOR by old id
  string getIORbyOldId( const int theOldId )
  {
    if ( mapIdToId.find( theOldId ) != mapIdToId.end() )
      return getIORbyId( mapIdToId[ theOldId ] );
    return string( "" );
  }
  // maps old object id to the new one (used when restoring data)
  void mapOldToNew( const int oldId, const int newId ) {
    mapIdToId[ oldId ] = newId;
  }
  // get old id by a new one
  int getOldId( const int newId ) {
    map<int, int>::iterator imap;
    for ( imap = mapIdToId.begin(); imap != mapIdToId.end(); ++imap ) {
      if ( imap->second == newId )
        return imap->first;
    }
    return 0;
  }
    
private:
  // get next free object identifier
  int getNextId()
  {
    int id = 1;
    while( mapIdToIOR.find( id ) != mapIdToIOR.end() )
      id++;
    return id;
  }

  map<int, string> mapIdToIOR;      // persistent-to-transient map
  map<int, int>    mapIdToId;       // used to translate object from persistent to transient form
};

// ===========================================================
// SMESH module's engine
// ==========================================================
class SMESH_I_EXPORT SMESH_Gen_i:
  public virtual POA_SMESH::SMESH_Gen,
  public virtual Engines_Component_i 
{
public:
  // Get last created instance of the class
  static SMESH_Gen_i* GetSMESHGen() { return mySMESHGen;}
  // Get ORB object
  static CORBA::ORB_var GetORB() { return myOrb;}
  // Get SMESH module's POA object
  static PortableServer::POA_var GetPOA() { return myPoa;}
  // Get Naming Service object
  static SALOME_NamingService* GetNS();
  // Get SALOME_LifeCycleCORBA object
  static SALOME_LifeCycleCORBA* GetLCC();
  // Retrieve and get GEOM engine reference
  static GEOM::GEOM_Gen_ptr GetGeomEngine();
  // Get object of the CORBA reference
  static PortableServer::ServantBase_var GetServant( CORBA::Object_ptr theObject );
  // Get CORBA object corresponding to the SALOMEDS::SObject
  static CORBA::Object_var SObjectToObject( SALOMEDS::SObject_ptr theSObject );
  // Get the SALOMEDS::SObject corresponding to a CORBA object
  static SALOMEDS::SObject_ptr ObjectToSObject(SALOMEDS::Study_ptr theStudy,
                                               CORBA::Object_ptr   theObject);
  // Get GEOM Object correspoding to TopoDS_Shape
  GEOM::GEOM_Object_ptr ShapeToGeomObject (const TopoDS_Shape& theShape );
  // Get TopoDS_Shape correspoding to GEOM_Object
  TopoDS_Shape GeomObjectToShape(GEOM::GEOM_Object_ptr theGeomObject);
  
  // Default constructor
  SMESH_Gen_i();
  // Standard constructor
  SMESH_Gen_i( CORBA::ORB_ptr            orb,
               PortableServer::POA_ptr   poa,
               PortableServer::ObjectId* contId, 
               const char*               instanceName, 
               const char*               interfaceName );
  // Destructor
  virtual ~SMESH_Gen_i();

  // *****************************************
  // Interface methods
  // *****************************************
  // Set a new Mesh object name
  void SetName(const char* theIOR,
               const char* theName);

  //GEOM::GEOM_Gen_ptr SetGeomEngine( const char* containerLoc );
  void SetGeomEngine( GEOM::GEOM_Gen_ptr geomcompo );

  // Set current study
  void SetEmbeddedMode( CORBA::Boolean theMode );
  // Get current study
  CORBA::Boolean IsEmbeddedMode();

  // Set current study
  void SetCurrentStudy( SALOMEDS::Study_ptr theStudy );
  // Get current study
  SALOMEDS::Study_ptr GetCurrentStudy();

  // Create hypothesis/algorothm of given type
  SMESH::SMESH_Hypothesis_ptr CreateHypothesis (const char* theHypType,
                                                const char* theLibName)
    throw ( SALOME::SALOME_Exception );
  
  // Return hypothesis of given type holding parameter values of the existing mesh
  SMESH::SMESH_Hypothesis_ptr GetHypothesisParameterValues (const char*           theHypType,
                                                            const char*           theLibName,
                                                            SMESH::SMESH_Mesh_ptr theMesh,
                                                            GEOM::GEOM_Object_ptr theGeom,
                                                            CORBA::Boolean        byMesh)
    throw ( SALOME::SALOME_Exception );
  
  /*!
   * Sets number of segments per diagonal of boundary box of geometry by which
   * default segment length of appropriate 1D hypotheses is defined
   */
  void SetBoundaryBoxSegmentation( CORBA::Long theNbSegments ) throw ( SALOME::SALOME_Exception );
  /*!
   * \brief Sets default number of segments per edge
   */
  void SetDefaultNbSegments(CORBA::Long theNbSegments) throw ( SALOME::SALOME_Exception );

  // Create empty mesh on a shape
  SMESH::SMESH_Mesh_ptr CreateMesh( GEOM::GEOM_Object_ptr theShapeObject )
    throw ( SALOME::SALOME_Exception );

  // Create empty mesh
  SMESH::SMESH_Mesh_ptr CreateEmptyMesh()
    throw ( SALOME::SALOME_Exception );

  //  Create mesh(es) and import data from UNV file
  SMESH::SMESH_Mesh_ptr CreateMeshesFromUNV( const char* theFileName )
    throw ( SALOME::SALOME_Exception );

  //  Create mesh(es) and import data from MED file
  SMESH::mesh_array* CreateMeshesFromMED( const char* theFileName,
                                          SMESH::DriverMED_ReadStatus& theStatus )
    throw ( SALOME::SALOME_Exception );

  //  Create mesh(es) and import data from STL file
  SMESH::SMESH_Mesh_ptr CreateMeshesFromSTL( const char* theFileName )
    throw ( SALOME::SALOME_Exception );

  // Compute mesh on a shape
  CORBA::Boolean Compute( SMESH::SMESH_Mesh_ptr theMesh,
                          GEOM::GEOM_Object_ptr theShapeObject )
    throw ( SALOME::SALOME_Exception );
  /*!
   * \brief Return errors of mesh computation
   */
  SMESH::compute_error_array* GetComputeErrors(SMESH::SMESH_Mesh_ptr theMesh,
                                               GEOM::GEOM_Object_ptr  theShapeObject )
    throw ( SALOME::SALOME_Exception );

  /*!
   * Evaluate mesh on a shape and
   *  returns statistic of mesh elements
   * Result array of number enityties
   */
  SMESH::long_array* Evaluate(SMESH::SMESH_Mesh_ptr theMesh,
                              GEOM::GEOM_Object_ptr theShapeObject)
    throw ( SALOME::SALOME_Exception );

  // Returns true if mesh contains enough data to be computed
  CORBA::Boolean IsReadyToCompute( SMESH::SMESH_Mesh_ptr theMesh,
                                   GEOM::GEOM_Object_ptr theShapeObject )
    throw ( SALOME::SALOME_Exception );
  
  /*!
   * Calculate Mesh as preview till indicated dimension on shape
   * First, verify list of hypothesis associated with the subShape.
   * Return mesh preview structure
   */
  SMESH::MeshPreviewStruct* Precompute( SMESH::SMESH_Mesh_ptr theMesh,
                                        GEOM::GEOM_Object_ptr theSubObject,
                                        SMESH::Dimension      theDimension,
                                        SMESH::long_array&    theShapesId )
    throw ( SALOME::SALOME_Exception );

  // Returns errors of hypotheses definintion
  SMESH::algo_error_array* GetAlgoState( SMESH::SMESH_Mesh_ptr theMesh, 
                                         GEOM::GEOM_Object_ptr theSubObject )
      throw ( SALOME::SALOME_Exception );

  // Return mesh elements preventing computation of a subshape
  SMESH::MeshPreviewStruct* GetBadInputElements( SMESH::SMESH_Mesh_ptr theMesh,
                                                 CORBA::Short          theSubShapeID )
    throw ( SALOME::SALOME_Exception );

  // Get sub-shapes unique ID's list
  SMESH::long_array* GetSubShapesId( GEOM::GEOM_Object_ptr      theMainShapeObject,
                                     const SMESH::object_array& theListOfSubShape )
    throw ( SALOME::SALOME_Exception );

  // Return geometrical object the given element is built on. Publish it in study.
  GEOM::GEOM_Object_ptr GetGeometryByMeshElement( SMESH::SMESH_Mesh_ptr  theMesh,
                                                  CORBA::Long            theElementID,
                                                  const char*            theGeomName)
    throw ( SALOME::SALOME_Exception );

  // Return geometrical object the given element is built on. Don't publish it in study.
  GEOM::GEOM_Object_ptr FindGeometryByMeshElement( SMESH::SMESH_Mesh_ptr  theMesh,
                                                   CORBA::Long            theElementID)
    throw ( SALOME::SALOME_Exception );

  // Concatenate the given meshes into one mesh
  SMESH::SMESH_Mesh_ptr ConcatenateCommon(const SMESH::mesh_array& theMeshesArray, 
                                          CORBA::Boolean           theUniteIdenticalGroups, 
                                          CORBA::Boolean           theMergeNodesAndElements, 
                                          CORBA::Double            theMergeTolerance,
                                          CORBA::Boolean           theCommonGroups)
    throw ( SALOME::SALOME_Exception );

  // Concatenate the given meshes into one mesh
  SMESH::SMESH_Mesh_ptr Concatenate(const SMESH::mesh_array& theMeshesArray, 
                                    CORBA::Boolean           theUniteIdenticalGroups, 
                                    CORBA::Boolean           theMergeNodesAndElements, 
                                    CORBA::Double            theMergeTolerance)
    throw ( SALOME::SALOME_Exception );

  // Concatenate the given meshes into one mesh
  // Create the groups of all elements from initial meshes
  SMESH::SMESH_Mesh_ptr ConcatenateWithGroups(const SMESH::mesh_array& theMeshesArray, 
                                              CORBA::Boolean           theUniteIdenticalGroups, 
                                              CORBA::Boolean           theMergeNodesAndElements, 
                                              CORBA::Double            theMergeTolerance)
    throw ( SALOME::SALOME_Exception );

  // ****************************************************
  // Interface inherited methods (from SALOMEDS::Driver)
  // ****************************************************

  // Save SMESH data
  SALOMEDS::TMPFile* Save( SALOMEDS::SComponent_ptr theComponent,
                         const char*              theURL,
                         bool                     isMultiFile );
  // Load SMESH data
  bool Load( SALOMEDS::SComponent_ptr theComponent,
             const SALOMEDS::TMPFile& theStream,
             const char*              theURL,
             bool                     isMultiFile );
  // Save SMESH data in ASCII format
  SALOMEDS::TMPFile* SaveASCII( SALOMEDS::SComponent_ptr theComponent,
                                const char*              theURL,
                                bool                     isMultiFile );
  // Load SMESH data in ASCII format
  bool LoadASCII( SALOMEDS::SComponent_ptr theComponent,
                  const SALOMEDS::TMPFile& theStream,
                  const char*              theURL,
                  bool                     isMultiFile );

  // Create filter manager
  SMESH::FilterManager_ptr CreateFilterManager();

  // Return a pattern mesher
  SMESH::SMESH_Pattern_ptr GetPattern();

  // Clears study-connected data when it is closed
  void Close( SALOMEDS::SComponent_ptr theComponent );
  
  // Get component data type
  char* ComponentDataType();
    
  // Transform data from transient form to persistent
  char* IORToLocalPersistentID( SALOMEDS::SObject_ptr theSObject,
                                const char*           IORString,
                                CORBA::Boolean        isMultiFile,
                                CORBA::Boolean        isASCII );
  // Transform data from persistent form to transient
  char* LocalPersistentIDToIOR( SALOMEDS::SObject_ptr theSObject,
                                const char*           aLocalPersistentID,
                                CORBA::Boolean        isMultiFile,
                                CORBA::Boolean        isASCII );

  // Returns true if object can be published in the study
  bool CanPublishInStudy( CORBA::Object_ptr theIOR );
  // Publish object in the study
  SALOMEDS::SObject_ptr PublishInStudy( SALOMEDS::Study_ptr   theStudy,
                                        SALOMEDS::SObject_ptr theSObject,
                                        CORBA::Object_ptr     theObject,
                                        const char*           theName ) 
    throw ( SALOME::SALOME_Exception );

  // Copy-paste methods - returns true if object can be copied to the clipboard
  CORBA::Boolean CanCopy( SALOMEDS::SObject_ptr theObject ) { return false; }
  // Copy-paste methods - copy object to the clipboard
  SALOMEDS::TMPFile* CopyFrom( SALOMEDS::SObject_ptr theObject, CORBA::Long& theObjectID ) { return false; }
  // Copy-paste methods - returns true if object can be pasted from the clipboard
  CORBA::Boolean CanPaste( const char* theComponentName, CORBA::Long theObjectID ) { return false; }
  // Copy-paste methods - paste object from the clipboard
  SALOMEDS::SObject_ptr PasteInto( const SALOMEDS::TMPFile& theStream,
                                   CORBA::Long              theObjectID,
                                   SALOMEDS::SObject_ptr    theObject ) {
    SALOMEDS::SObject_var aResultSO;
    return aResultSO._retn();
  }

  // ============
  // Dump python
  // ============

  virtual Engines::TMPFile* DumpPython(CORBA::Object_ptr theStudy, 
                                       CORBA::Boolean isPublished, 
                                       CORBA::Boolean& isValidScript);

  void AddToPythonScript (int theStudyID, const TCollection_AsciiString& theString);

  void RemoveLastFromPythonScript (int theStudyID);

  void SavePython (SALOMEDS::Study_ptr theStudy);

  TCollection_AsciiString DumpPython_impl (SALOMEDS::Study_ptr theStudy, 
                                           Resource_DataMapOfAsciiStringAsciiString& theObjectNames,
                                           Resource_DataMapOfAsciiStringAsciiString& theNames,
                                           bool isPublished, 
                                           bool& aValidScript,
                                           const TCollection_AsciiString& theSavedTrace);

  TCollection_AsciiString GetNewPythonLines (int theStudyID);

  void CleanPythonTrace (int theStudyID);


  // *****************************************
  // Internal methods
  // *****************************************
public:
  // Get shape reader
  GEOM_Client* GetShapeReader();

  // Tags definition 
  static long GetHypothesisRootTag();
  static long GetAlgorithmsRootTag();
  static long GetRefOnShapeTag();
  static long GetRefOnAppliedHypothesisTag();
  static long GetRefOnAppliedAlgorithmsTag();
  static long GetSubMeshOnVertexTag();
  static long GetSubMeshOnEdgeTag();
  static long GetSubMeshOnFaceTag();
  static long GetSubMeshOnSolidTag();
  static long GetSubMeshOnCompoundTag();
  static long GetSubMeshOnWireTag();
  static long GetSubMeshOnShellTag();
  static long GetNodeGroupsTag();
  static long GetEdgeGroupsTag();
  static long GetFaceGroupsTag();
  static long GetVolumeGroupsTag();

  // publishing methods
  SALOMEDS::SComponent_ptr PublishComponent(SALOMEDS::Study_ptr theStudy);
  SALOMEDS::SObject_ptr PublishMesh (SALOMEDS::Study_ptr   theStudy,
                                     SMESH::SMESH_Mesh_ptr theMesh,
                                     const char*           theName = 0);
  SALOMEDS::SObject_ptr PublishHypothesis (SALOMEDS::Study_ptr         theStudy,
                                           SMESH::SMESH_Hypothesis_ptr theHyp,
                                           const char*                 theName = 0);
  SALOMEDS::SObject_ptr PublishSubMesh (SALOMEDS::Study_ptr      theStudy,
                                        SMESH::SMESH_Mesh_ptr    theMesh,
                                        SMESH::SMESH_subMesh_ptr theSubMesh,
                                        GEOM::GEOM_Object_ptr    theShapeObject,
                                        const char*              theName = 0);
  SALOMEDS::SObject_ptr PublishGroup (SALOMEDS::Study_ptr    theStudy,
                                      SMESH::SMESH_Mesh_ptr  theMesh,
                                      SMESH::SMESH_GroupBase_ptr theGroup,
                                      GEOM::GEOM_Object_ptr  theShapeObject,
                                      const char*            theName = 0);
  bool AddHypothesisToShape(SALOMEDS::Study_ptr         theStudy,
                            SMESH::SMESH_Mesh_ptr       theMesh,
                            GEOM::GEOM_Object_ptr       theShapeObject,
                            SMESH::SMESH_Hypothesis_ptr theHyp);
  bool RemoveHypothesisFromShape(SALOMEDS::Study_ptr         theStudy,
                                 SMESH::SMESH_Mesh_ptr       theMesh,
                                 GEOM::GEOM_Object_ptr       theShapeObject,
                                 SMESH::SMESH_Hypothesis_ptr theHyp);
  SALOMEDS::SObject_ptr GetMeshOrSubmeshByShape (SALOMEDS::Study_ptr   theStudy,
                                                 SMESH::SMESH_Mesh_ptr theMesh,
                                                 GEOM::GEOM_Object_ptr theShape);
  static void SetName(SALOMEDS::SObject_ptr theSObject,
                      const char*           theName,
                      const char*           theDefaultName = 0);

  static void SetPixMap(SALOMEDS::SObject_ptr theSObject,
                        const char*           thePixMap);

  //  Get study context
  StudyContext* GetCurrentStudyContext();

  // Register an object in a StudyContext; return object id
  int RegisterObject(CORBA::Object_ptr theObject);

  // Return id of registered object
  CORBA::Long GetObjectId(CORBA::Object_ptr theObject);

  // Return an object that previously had an oldID
  template<class TInterface> 
  typename TInterface::_var_type GetObjectByOldId( const int oldID )
  {
    if ( StudyContext* myStudyContext = GetCurrentStudyContext() ) {
      string ior = myStudyContext->getIORbyOldId( oldID );
      if ( !ior.empty() )
        return TInterface::_narrow(GetORB()->string_to_object( ior.c_str() ));
    }
    return TInterface::_nil();
  }

  // Get current study ID
  int GetCurrentStudyID();

  /*!
   * \brief Find SObject for an algo
   */
  SALOMEDS::SObject_ptr GetAlgoSO(const ::SMESH_Algo* algo);

  void UpdateParameters(CORBA::Object_ptr theObject, const char* theParameters);
  char* GetParameters(CORBA::Object_ptr theObject);
  char* ParseParameters(const char* theParameters);
  
 
private:
  // Create hypothesis of given type
  SMESH::SMESH_Hypothesis_ptr createHypothesis( const char* theHypName,
                                                const char* theLibName)
    throw ( SALOME::SALOME_Exception );
  
  // Create empty mesh on shape
  SMESH::SMESH_Mesh_ptr createMesh()
    throw ( SALOME::SALOME_Exception );

  static void loadGeomData( SALOMEDS::SComponent_ptr theCompRoot );
  
private:
  static GEOM::GEOM_Gen_var      myGeomGen;
  static CORBA::ORB_var          myOrb;         // ORB reference
  static PortableServer::POA_var myPoa;         // POA reference
  static SALOME_NamingService*   myNS;          // Naming Service
  static SALOME_LifeCycleCORBA*  myLCC;         // Life Cycle CORBA
  static SMESH_Gen_i*            mySMESHGen;    // Point to last created instance of the class
  ::SMESH_Gen               myGen;              // SMESH_Gen local implementation

  // hypotheses managing
  map<string, GenericHypothesisCreator_i*> myHypCreatorMap;

  map<int, StudyContext*>   myStudyContextMap;  // Map of study context objects

  GEOM_Client*              myShapeReader;      // Shape reader
  SALOMEDS::Study_var       myCurrentStudy;     // Current study
  CORBA::Boolean            myIsEmbeddedMode;   // Current mode

  // Dump Python: trace of API methods calls
  std::map < int, Handle(TColStd_HSequenceOfAsciiString) > myPythonScripts;
};


namespace SMESH
{
  template<class T>
  T
  DownCast(CORBA::Object_ptr theArg)
  {
    return dynamic_cast<T>(SMESH_Gen_i::GetServant(theArg).in());
  }
}


#endif
