# -*- coding: utf-8 -*-

# Copyright (C) 2010-2011 by Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# Python X2go is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Python X2go is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

"""\
X2goProxyBASE class - proxying your connection through NX3 and others.

"""
__NAME__ = 'x2goproxy-pylib'

# modules
import gevent
import os
import sys
import types
import time
import copy
import threading
import cStringIO

# Python X2go modules
import x2go.forward as forward
import x2go.log as log
import x2go.utils as utils
import x2go.x2go_exceptions as x2go_exceptions

from x2go.defaults import X2GOCLIENT_OS as _X2GOCLIENT_OS
if _X2GOCLIENT_OS in ("Windows"):
    import subprocess
else:
    import x2go.gevent_subprocess as subprocess
    from x2go.x2go_exceptions import WindowsError

from x2go.defaults import LOCAL_HOME as _LOCAL_HOME
from x2go.defaults import X2GO_SESSIONS_ROOTDIR as _X2GO_SESSIONS_ROOTDIR


class X2goProxyBASE(threading.Thread):
    """\
    X2goProxy is an abstract class for X2go proxy connections.

    This class needs to be inherited from a concrete proxy class. Only 
    currently available proxy class is: L{X2goProxyNX3}.

    """
    PROXY_CMD = ''
    """Proxy command. Needs to be set by a potential child class, might be OS specific."""
    PROXY_ARGS = []
    """Arguments to be passed to the proxy command. This needs to be set by a potential child class."""
    PROXY_ENV = {}
    """Provide environment variables to the proxy command. This also needs to be set by a child class."""

    session_info = None
    session_log_stdout = None
    session_log_stderr = None
    fw_tunnel = None
    proxy = None

    def __init__(self, session_info=None, 
                 ssh_transport=None, session_log="session.log", 
                 sessions_rootdir=os.path.join(_LOCAL_HOME, _X2GO_SESSIONS_ROOTDIR),
                 proxy_options={},
                 session_instance=None,
                 logger=None, loglevel=log.loglevel_DEFAULT, ):
        """\
        @param session_info: session information provided as an C{X2goServerSessionInfo*} backend
            instance
        @type session_info: C{X2goServerSessionInfo*} instance
        @param ssh_transport: SSH transport object from C{paramiko.SSHClient}
        @type ssh_transport: C{paramiko.Transport} instance
        @param session_log: name of the proxy's session logfile
        @type session_log: C{str}
        @param sessions_rootdir: base dir where X2go session files are stored (by default: ~/.x2go)
        @type sessions_rootdir: C{str}
        @param proxy_options: a set of very C{X2goProxy*} backend specific options; any option that is not known
            to the C{X2goProxy*} backend will simply be ignored
        @type proxy_options: C{dict}
        @param logger: you can pass an L{X2goLogger} object to the
            L{X2goProxy} constructor
        @param session_instance: the L{X2goSession} instance this C{X2goProxy*} instance belongs to
        @type session_instance: L{X2goSession} instance
        @type logger: L{X2goLogger} instance
        @param loglevel: if no L{X2goLogger} object has been supplied a new one will be
            constructed with the given loglevel
        @type loglevel: int

        """
        if logger is None:
            self.logger = log.X2goLogger(loglevel=loglevel)
        else:
            self.logger = copy.deepcopy(logger)
        self.logger.tag = __NAME__

        self.sessions_rootdir = sessions_rootdir
        self.session_info = session_info
        self.ssh_transport = ssh_transport
        self.session_log = session_log
        self.proxy_options = proxy_options
        self.session_instance = session_instance
        self.PROXY_ENV = os.environ.copy()
        self.proxy = None

        threading.Thread.__init__(self)
        self.daemon = True

    def __del__(self):
        """\
        On instance destruction make sure the this proxy thread is stopped properly.

        """
        self.stop_thread()

    def _tidy_up(self):
        """\
        Close any left open port forwarding tunnel, also close session log file, 
        if left open.

        """
        if self.proxy:
            self.logger('Shutting down X2go proxy subprocess', loglevel=log.loglevel_DEBUG)
            try:
                self.proxy.kill()
            except OSError, e:
                self.logger('X2go proxy shutdown gave a message that we may ignore: %s' % str(e), loglevel=log.loglevel_WARN)
            self.proxy = None
        if self.fw_tunnel is not None:
            self.logger('Shutting down Paramiko/SSH forwarding tunnel', loglevel=log.loglevel_DEBUG)
            forward.stop_forward_tunnel(self.fw_tunnel)
            self.fw_tunnel = None
        if self.session_log_stdout is not None:
            self.session_log_stdout.close()
        if self.session_log_stderr is not None:
            self.session_log_stderr.close()

    def stop_thread(self):
        """\
        End the thread runner and tidy up.

        """
        self._keepalive = False
        # wait for thread loop to finish...
        gevent.sleep(.5)
        self._tidy_up()

    def run(self):
        """\
        Start the X2go proxy command. The X2go proxy command utilizes a
        Paramiko/SSH based forwarding tunnel (openssh -L option). This tunnel
        gets started here and is forked into background (Greenlet/gevent).

        """
        self._keepalive = True
        self.proxy = None

        if self.session_info is None or self.ssh_transport is None:
            return None

        try:
            os.mkdir(self.session_info.local_container)
        except OSError, e:
            if e.errno == 17:
                # file exists
                pass

        local_graphics_port = self.session_info.graphics_port
        local_graphics_port = utils.detect_unused_port(bind_address='', preferred_port=local_graphics_port)
        self.fw_tunnel = forward.start_forward_tunnel(local_port=local_graphics_port, 
                                                      remote_port=self.session_info.graphics_port, 
                                                      ssh_transport=self.ssh_transport, 
                                                      session_instance=self.session_instance,
                                                      logger=self.logger,
                                                     )

        # update the proxy port in PROXY_ARGS
        self._update_local_proxy_socket(local_graphics_port)
        cmd_line = self._generate_cmdline()

        self.session_log_stdout = open('%s/%s' % (self.session_info.local_container, self.session_log, ), 'a')
        self.session_log_stderr = open('%s/%s' % (self.session_info.local_container, self.session_log, ), 'a')
        self.logger('forking threaded subprocess: %s' % " ".join(cmd_line), loglevel=log.loglevel_DEBUG)

        _stdin = None
        _shell = False
        if _X2GOCLIENT_OS == 'Windows':
            _stdin = file('nul', 'r')
            _shell = True

        # allow inheriting classes to do something with backend specific proxy_options...
        self.process_proxy_options()

        while not self.proxy:
            gevent.sleep(.2)
            p = self.proxy = subprocess.Popen(cmd_line,
                                              env=self.PROXY_ENV,
                                              stdin=_stdin,
                                              stdout=self.session_log_stdout,
                                              stderr=self.session_log_stderr,
                                              shell=_shell)

        while self._keepalive:
            gevent.sleep(.2)

        if _X2GOCLIENT_OS == 'Windows':
            _stdin.close()
        try:
            p.terminate()
            self.logger('terminating proxy: %s' % p, loglevel=log.loglevel_DEBUG)
        except OSError, e:
            if e.errno == 3:
                # No such process
                pass
        except WindowsError:
            pass

    def process_proxy_options(self):
        """\
        Override this method to incorporate elements from C{proxy_options} 
        into actual proxy subprocess execution.

        This method (if overridden) should (by design) never fail nor raise an exception.
        Make sure to catch all possible errors appropriately.

        If you want to log ignored proxy_options then

            1. remove processed proxy_options from self.proxy_options
            2. once you have finished processing the proxy_options call
            the parent class method X2goProxyBASE.process_proxy_options()

        """
        # do the logging of remaining options
        if self.proxy_options:
            self.logger('ignoring non-processed proxy options: %s' % self.proxy_options, loglevel=log.loglevel_INFO)

    def _update_local_proxy_socket(self, port):
        pass

    def _generate_cmdline(self):
        return ''

    def start_proxy(self):
        """\
        Start the thread runner and wait for the proxy to come up.

        @return: a subprocess instance that knows about the externally started proxy command.
        @rtype: C{instance}

        """
        threading.Thread.start(self)

        # wait for proxy to get started
        _count = 0
        _maxwait = 40
        while self.proxy is None and _count < _maxwait:
            _count += 1
            self.logger('waiting for proxy to come up: 0.4s x %s' % _count, loglevel=log.loglevel_DEBUG)
            gevent.sleep(.4)

        # also wait for fw_tunnel to become active
        _count = 0
        _maxwait = 40
        while not self.fw_tunnel.is_active and _count < _maxwait:
            _count += 1
            self.logger('waiting for port fw tunnel to come up: 0.5s x %s' % _count, loglevel=log.loglevel_DEBUG)
            gevent.sleep(.5)

        return self.proxy

    def ok(self):
        """\
        Check if a proxy instance is up and running.

        @return: Proxy state (C{True} or C{False})
        @rtype C{bool}

        """
        return bool(self.proxy and self.proxy.poll() is None)
