#!/usr/bin/python

import re
from utils                      import o2str
from tags                       import BugTags
from attachments                import Attachments
from person                     import Person
from messages                   import Messages
from nominations                import Nominations
from bug_activity               import Activity

# Bug
#
# A class that provides a convenient interface to a Launchpad bug.
#
class Bug(object):
    # __init__
    #
    # Initialize the Bug instance from a Launchpad bug.
    #
    def __init__(self, service, bug_number, commit_changes=True):
        self.__service      = service
        launchpad           = self.__service.launchpad
        self.lpbug          = launchpad.bugs[bug_number]
        self.id             = self.lpbug.id
        self.commit_changes = commit_changes

        # Cached copies so we don't go back to launchpad as often
        #
        self.__title             = None
        self.__description       = None
        self.__tags              = None
        self.__attachments       = None
        self.__owner             = None
        self.__date_created      = None
        self.__date_last_updated = None
        self.__date_last_message = None
        self.__private           = None
        self.__properties        = None
        self.__security_related  = None

    #--------------------------------------------------------------------------
    # date_created
    #
    # (read-only)
    #
    @property
    def date_created(self):
        if self.__date_created == None:
            self.__date_created = self.lpbug.date_created
        return self.__date_created

    #--------------------------------------------------------------------------
    # date_last_updated
    #
    # (read-only)
    #
    @property
    def date_last_updated(self):
        if self.__date_last_updated == None:
            self.__date_last_updated = self.lpbug.date_last_updated
        return self.__date_last_updated

    #--------------------------------------------------------------------------
    # date_last_message
    #
    # (read-only)
    #
    @property
    def date_last_message(self):
        if self.__date_last_message == None:
            self.__date_last_message = self.lpbug.date_last_message
        return self.__date_last_message

    #--------------------------------------------------------------------------
    # private
    #
    # (read-only)
    #
    @property
    def private(self):
        if self.__private == None:
            self.__private = self.lpbug.private
        return self.__private

    # private
    #
    # (read-only)
    #
    @private.setter
    def private(self, value):
        self.lpbug.private = value
        if self.commit_changes:
            self.lpbug.lp_save()
        self.__private = value
        return

    #--------------------------------------------------------------------------
    # security_related
    #
    # (read-only)
    #
    @property
    def security_related(self):
        if self.__security_related == None:
            self.__security_related = self.lpbug.security_related
        return self.__security_related

    # security_related
    #
    # (read-only)
    #
    @security_related.setter
    def security_related(self, value):
        self.lpbug.security_related = value
        if self.commit_changes:
            self.lpbug.lp_save()
        self.__security_related = value
        return

    #--------------------------------------------------------------------------
    # title
    #
    @property
    def title(self):
        '''A one-line summary of the problem being described by the bug.'''
        if self.__title == None:
            self.__title = o2str(self.lpbug.title)
        return self.__title

    @title.setter
    def title(self, value):
        if not isinstance(value, str):
            raise TypeError("Must be a string")
        self.lpbug.title = value
        if self.commit_changes:
            self.lpbug.lp_save()
        self.__title = value

    #--------------------------------------------------------------------------
    # description
    #
    @property
    def description(self):
        '''As complete as possible description of the bug/issue being reported as a bug.'''
        if self.__description == None:
            self.__description = o2str(self.lpbug.description)
        return self.__description

    @description.setter
    def description(self, value):
        if not isinstance(value, str):
            raise TypeError("Must be a string")
        self.lpbug.description = value
        if self.commit_changes:
            self.lpbug.lp_save()
        self.__description = value

    #--------------------------------------------------------------------------
    # tags
    #
    @property
    def tags(self):
        return BugTags(self)

    #--------------------------------------------------------------------------
    # owner
    #
    @property
    def owner(self):
        if self.__owner == None:
            self.__owner = Person(self, self.lpbug.owner)
        return self.__owner

    #--------------------------------------------------------------------------
    # attachments
    #
    @property
    def attachments(self):
        return Attachments(self)

    def _parse_properties(self, text):
        return props

    #--------------------------------------------------------------------------
    # properties
    #
    @property
    def properties(self):
        '''Returns dict of key: value pairs found in the bug description

        This parses the bug report description into a more
        programmatically digestable dictionary form.
        '''
        if self.__properties is None:
            re_kvp            = re.compile("^(\s*)([\.\-\w]+):\s*(.*)$")
            re_error          = re.compile("^Error:\s*(.*)$")
            self.__properties = {}
            last_key = {'': 'bar'}
            for line in self.description.split("\n"):
                m = re_kvp.match(line)
                if not m:
                    continue

                level = m.group(1)
                item = m.group(2)
                value = m.group(3)
                key = item

                if len(level) > 0:
                    key = "%s.%s" %(last_key[''], item)
                last_key[level] = item

                m = re_error.match(value)
                if not m:
                    self.__properties[key] = value
        
        return self.__properties

    #--------------------------------------------------------------------------
    # messages
    #
    @property
    def messages(self):
        return Messages(self)

    #--------------------------------------------------------------------------
    # tasks
    #
    @property
    def tasks(self):
        # The following import is done here to work around a circular import
        # issue. bug_tasks imports bug.
        #
        from bug_tasks import BugTasks

        return BugTasks(self.__service, self.lpbug.bug_tasks_collection)

    #--------------------------------------------------------------------------
    # new_message
    #     Add a new comment to an existing bug. This is the equivalent of
    #     newMessage.
    #
    def add_comment(self, content, subject=""):
        self.lpbug.newMessage(content=content, subject=subject)

    #--------------------------------------------------------------------------
    # nominations
    #
    @property
    def nominations(self):
        return Nominations(self.__service, self)

    #--------------------------------------------------------------------------
    # activity
    #
    @property
    def activity(self):
        return Activity(self.__service, self)

# vi:set ts=4 sw=4 expandtab:
