/**
 * GTEst test suite for GEIS v2 device handling.
 *
 * Copyright 2012 Canonical Ltd.
 */
#include "geis/geis.h"
#include "gtest_evemu_device.h"
#include "gtest_geis_fixture.h"
#include "libutouch-geis/geis_test_api.h"
#include <memory>
#include <sys/select.h>
#include <sys/time.h>


namespace
{

static const std::string TEST_DEVICE_PROP_FILE(
    TEST_ROOT_DIR "recordings/touchscreen_a/device.prop");
static const std::string TEST_DEVICE_EVENTS_FILE(
    TEST_ROOT_DIR "recordings/touchscreen_a/rotate90.record");

/**
 * Fixture for testing device handling.
 * This is a separate class because gtest uses Java reflection.
 */
class GeisDeviceTests
: public GTestGeisFixture
{
  void
  setup_geis()
  {
    geis_ = geis_new(GEIS_INIT_SYNCHRONOUS_START,
                     GEIS_INIT_NO_ATOMIC_GESTURES,
                     GEIS_CONFIG_DISCARD_DEVICE_MESSAGES,
                     NULL);
  }
};


TEST_F(GeisDeviceTests, filterWithNoDevices)
{
  GeisSubscription sub = geis_subscription_new(geis_,
                                               "no devices",
                                               GEIS_SUBSCRIPTION_NONE);
  EXPECT_TRUE(sub != NULL) << "can not create subscription";
  GeisFilter filter = geis_filter_new(geis_, "rotate");
  EXPECT_TRUE(filter != NULL) << "can not create filter";

  GeisStatus fs = geis_filter_add_term(filter,
            GEIS_FILTER_DEVICE,
            GEIS_DEVICE_ATTRIBUTE_DIRECT_TOUCH, GEIS_FILTER_OP_EQ, GEIS_TRUE,
            NULL);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not add device filter";

  geis_filter_delete(filter);
  geis_subscription_delete(sub);
}


/*
 * Test case 1 for lp:944822 -- device added after subscription activated.
 *
 * Creates a subscriptions with a device filter (there are no devices present in
 * the system).  When initializatiom signalled as complete, device events are
 * reenabled in the geis instance and a new device is added.  When the
 * device-added event is received, a recording is run through the device.
 * Gesture events should be detected.
 */
TEST_F(GeisDeviceTests, addDeviceSubscription)
{
  std::unique_ptr<Testsuite::EvemuDevice> new_device;
  GeisBoolean device_has_been_created = GEIS_FALSE;
  GeisBoolean gesture_events_received = GEIS_FALSE;

  GeisSubscription sub = geis_subscription_new(geis_,
                                               "no devices",
                                               GEIS_SUBSCRIPTION_NONE);
  EXPECT_TRUE(sub != NULL) << "can not create subscription";
  GeisFilter filter = geis_filter_new(geis_, "rotate");
  EXPECT_TRUE(filter != NULL) << "can not create filter";
  GeisStatus fs = geis_filter_add_term(filter,
            GEIS_FILTER_DEVICE,
            GEIS_DEVICE_ATTRIBUTE_DIRECT_TOUCH, GEIS_FILTER_OP_EQ, GEIS_TRUE,
            NULL);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not add device term";
  fs = geis_subscription_add_filter(sub, filter);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not add device term";

  GeisStatus dispatch_status= geis_dispatch_events(geis_);
  while (dispatch_status == GEIS_STATUS_CONTINUE
      || dispatch_status == GEIS_STATUS_SUCCESS)
  {
   GeisEvent  event;
    GeisStatus event_status = geis_next_event(geis_, &event);
    while (event_status == GEIS_STATUS_CONTINUE
        || event_status == GEIS_STATUS_SUCCESS)
    {
      switch (geis_event_type(event))
      {
        case GEIS_EVENT_INIT_COMPLETE:
        {
          EXPECT_EQ(GEIS_STATUS_SUCCESS, geis_subscription_activate(sub))
                      << "can not activate subscription";

          GeisBoolean off = GEIS_FALSE;
          geis_set_configuration(geis_,
                                 GEIS_CONFIG_DISCARD_DEVICE_MESSAGES,
                                 &off);
          new_device.reset(new Testsuite::EvemuDevice(TEST_DEVICE_PROP_FILE));
          device_has_been_created = GEIS_TRUE;
          break;
        }

       case GEIS_EVENT_DEVICE_AVAILABLE:
         new_device->play(TEST_DEVICE_EVENTS_FILE);
         break;

       case GEIS_EVENT_GESTURE_BEGIN:
       case GEIS_EVENT_GESTURE_UPDATE:
       {
         EXPECT_EQ(device_has_been_created, GEIS_TRUE)
           << "gesture events without device";
         gesture_events_received = device_has_been_created;
       }

       default:
          break;
      }

      geis_event_delete(event);
      event_status = geis_next_event(geis_, &event);
    }

    fd_set read_fds;
    FD_ZERO(&read_fds);
    FD_SET(geis_fd(), &read_fds);
    timeval tmo = { 5, 0 };
    int sstat = select(geis_fd() + 1, &read_fds, NULL, NULL, &tmo);
    EXPECT_GT(sstat, -1) << "error in select";
    if (sstat == 0)
      break;
    dispatch_status = geis_dispatch_events(geis_);
  }

  EXPECT_TRUE(gesture_events_received) << "no gesture events received";
  EXPECT_TRUE(device_has_been_created) << "no device created";
  geis_filter_delete(filter);
  geis_subscription_delete(sub);
}


/*
 * Test case 2 for lp:944822 -- device removed after subscription activated.
 *
 * This test really just makes sure nothing segfaults on device removal.
 */
TEST_F(GeisDeviceTests, removeDeviceSubscription)
{
  std::unique_ptr<Testsuite::EvemuDevice> new_device;

  GeisSubscription sub = geis_subscription_new(geis_,
                                               "remove devices",
                                               GEIS_SUBSCRIPTION_NONE);
  EXPECT_TRUE(sub != NULL) << "can not create subscription";
  GeisFilter filter = geis_filter_new(geis_, "rotate");
  EXPECT_TRUE(filter != NULL) << "can not create filter";
  GeisStatus fs = geis_filter_add_term(filter,
            GEIS_FILTER_DEVICE,
            GEIS_DEVICE_ATTRIBUTE_DIRECT_TOUCH, GEIS_FILTER_OP_EQ, GEIS_TRUE,
            NULL);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not add device term";
  fs = geis_subscription_add_filter(sub, filter);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not add device term";

  GeisStatus dispatch_status= geis_dispatch_events(geis_);
  while (dispatch_status == GEIS_STATUS_CONTINUE
      || dispatch_status == GEIS_STATUS_SUCCESS)
  {
   GeisEvent  event;
    GeisStatus event_status = geis_next_event(geis_, &event);
    while (event_status == GEIS_STATUS_CONTINUE
        || event_status == GEIS_STATUS_SUCCESS)
    {
      switch (geis_event_type(event))
      {
        case GEIS_EVENT_INIT_COMPLETE:
        {
          GeisBoolean off = GEIS_FALSE;
          geis_set_configuration(geis_,
                                 GEIS_CONFIG_DISCARD_DEVICE_MESSAGES,
                                 &off);
          new_device.reset(new Testsuite::EvemuDevice(TEST_DEVICE_PROP_FILE));
          break;
        }

       case GEIS_EVENT_DEVICE_AVAILABLE:
       {
         EXPECT_EQ(GEIS_STATUS_SUCCESS, geis_subscription_activate(sub))
                      << "can not activate subscription";
         new_device.reset();
         break;
       }

       case GEIS_EVENT_GESTURE_BEGIN:
       case GEIS_EVENT_GESTURE_UPDATE:
       {
       }

       default:
          break;
      }

      geis_event_delete(event);
      event_status = geis_next_event(geis_, &event);
    }

    fd_set read_fds;
    FD_ZERO(&read_fds);
    FD_SET(geis_fd(), &read_fds);
    timeval tmo = { 5, 0 };
    int sstat = select(geis_fd() + 1, &read_fds, NULL, NULL, &tmo);
    EXPECT_GT(sstat, -1) << "error in select";
    if (sstat == 0)
      break;
    dispatch_status = geis_dispatch_events(geis_);
  }
  geis_filter_delete(filter);
  geis_subscription_delete(sub);
}


} // anonymous namespace
