/**
 * GMyth Library
 *
 * @file gmyth/gmyth_programinfo.h
 * 
 * @brief <p> GMythProgramInfo representing the program info, with the
 * configuration data to the actual remote file in the TV chain.
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia.
 * @author Rosfran Borges <rosfran.borges@indt.org.br>
 * @author Leonardo Sobral Cunha <leonardo.cunha@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef _GMYTH_PROGRAMINFO_H
#define _GMYTH_PROGRAMINFO_H

#include <glib.h>
#include <glib-object.h>

#include "gmyth_stringlist.h"

G_BEGIN_DECLS
#define GMYTH_PROGRAM_INFO_TYPE               (gmyth_program_info_get_type ())
#define GMYTH_PROGRAM_INFO(obj)               (G_TYPE_CHECK_INSTANCE_CAST ((obj), GMYTH_PROGRAM_INFO_TYPE, GMythProgramInfo))
#define GMYTH_PROGRAM_INFO_CLASS(klass)       (G_TYPE_CHECK_CLASS_CAST ((klass), GMYTH_PROGRAM_INFO_TYPE, GMythProgramInfoClass))
#define IS_GMYTH_PROGRAM_INFO(obj)            (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GMYTH_PROGRAM_INFO_TYPE))
#define IS_GMYTH_PROGRAM_INFO_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), GMYTH_PROGRAM_INFO_TYPE))
#define GMYTH_PROGRAM_INFO_GET_CLASS(obj)     (G_TYPE_INSTANCE_GET_CLASS ((obj), GMYTH_PROGRAM_INFO_TYPE, GMythProgramInfoClass))
typedef struct _GMythProgramInfo GMythProgramInfo;
typedef struct _GMythProgramInfoClass GMythProgramInfoClass;

struct _GMythProgramInfoClass {
    GObjectClass    parent_class;

    /*
     * callbacks 
     */
};

/**
 * The GMythProgramInfo structure represents a program information
 * stored in the database. It could be a program from the EPG data,
 * a program scheduled to be recorded, or a program already recorded.
 */
struct _GMythProgramInfo {
    GObject         parent;

    /** The channel unique ID. */
    gint            channel_id;
    /** The program start time. */
    GTimeVal       *startts;
    /** The program end time. */
    GTimeVal       *endts;
    /** The recording schedule start time. */
    GTimeVal       *recstartts;
    /** The recording schedule end time */
    GTimeVal       *recendts;

    /** The program title. */
    GString        *title;
    /** The program subtitle. */
    GString        *subtitle;
    /** The program description. */
    GString        *description;
    /** The program category. */
    GString        *category;

    GString        *chanstr;
    GString        *chansign;
    /** The associated channel name. */
    GString        *channame;
    gint            chancommfree;
    GString        *chanOutputFilters;

    GString        *seriesid;
    /** The program unique id. */
    GString        *program_id;
    GString        *catType;

    GString        *sortTitle;

    /** A flag informing if the program has video or not. */
    gboolean        isVideo;
    gint            lenMins;

    GString        *year;
    gdouble         stars;
    gint            repeat;

    GTimeVal       *originalAirDate;
    GTimeVal       *lastmodified;
    GTimeVal       *lastInUseTime;

    gboolean        hasAirDate;

    gint            spread;
    gint            startCol;

    gint            recpriority2;
    gint            reactivate;

    gint            recordid;
    gint            parentid;

    /** The backend video source id associated to this program.*/
    gint            sourceid;
    /** the backend input id associated to this program.*/
    gint            inputid;
    /** The backend card id associated to this program.*/
    gint            cardid;
    gboolean        shareable;
    gboolean        duplicate;

    GString        *schedulerid;
    gint            findid;

    gint            programflags;
    gint            transcoder;

    GString        *recgroup;
    GString        *playgroup;
    gint            rectype;
    gint            recstatus;
    gint            recpriority;
    gint            dupin;
    gint            dupmethod;

    /** The file size of the recorded program.*/
    gint64          filesize;

    /** The file name of the recorded program.*/
    GString        *pathname;
    GString        *hostname;
};

GType           gmyth_program_info_get_type(void);

GMythProgramInfo *gmyth_program_info_new(void);

GMythStringList *gmyth_program_info_to_string_list(GMythProgramInfo * prog,
                                                   GMythStringList *
                                                   slist);

GMythProgramInfo
    * gmyth_program_info_from_string_list_from_pos(GMythStringList * slist,
                                                   guint pos);

GMythProgramInfo *gmyth_program_info_from_string_list(GMythStringList *
                                                      slist);

GMythProgramInfo
    * gmyth_program_info_from_string_list_next_prog(GMythStringList *
                                                    slist);

const gchar    *gmyth_program_info_to_string(const GMythProgramInfo *
                                             prog);

gboolean        gmyth_program_info_is_equals(const GMythProgramInfo *
                                             prog1,
                                             const GMythProgramInfo *
                                             prog2);

G_END_DECLS
#endif /*_GMYTH_PROGRAMINFO_H*/
