/**
 * GMyth Library
 *
 * @file gmyth/gmyth_monitor_handler.c
 * 
 * @brief <p> GMythMonitorHandler deals with the streaming media events remote/local
 * that are sent to the MythTV frontend.
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia.
 * @author Rosfran Lins Borges <rosfran.borges@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * GStreamer MythTV plug-in properties:
 * - location (backend server hostname/URL) [ex.: myth://192.168.1.73:28722/1000_1092091.nuv]
 * - path (qurl - remote file to be opened)
 * - port number *   
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <unistd.h>
#include <glib.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <errno.h>
#include <stdlib.h>
#include <assert.h>

#include "gmyth_marshal.h"

#include "gmyth_monitor_handler.h"
#include "gmyth_debug.h"

#define GMYTHTV_QUERY_HEADER		"QUERY_FILETRANSFER "

#define GMYTHTV_VERSION							30

#define GMYTHTV_TRANSFER_MAX_WAITS	700

#define GMYTHTV_BUFFER_SIZE					8*1024

#ifdef GMYTHTV_ENABLE_DEBUG
#define GMYTHTV_ENABLE_DEBUG				1
#else
#undef GMYTHTV_ENABLE_DEBUG
#endif

/*
 * this NDEBUG is to maintain compatibility with GMyth library 
 */
#ifndef NDEBUG
#define GMYTHTV_ENABLE_DEBUG				1
#endif

static gboolean gmyth_monitor_handler_listener  (GIOChannel *io_channel,
                                                 GIOCondition condition,
                                                 gpointer data);

static void     gmyth_monitor_handler_default_listener(GMythMonitorHandler
                                                       * monitor,
                                                       gint msg_code,
                                                       gchar * message);

static void     gmyth_monitor_handler_class_init(GMythMonitorHandlerClass *
                                                 klass);
static void     gmyth_monitor_handler_init(GMythMonitorHandler * object);

static void     gmyth_monitor_handler_dispose(GObject * object);
static void     gmyth_monitor_handler_finalize(GObject * object);

static gboolean gmyth_connect_to_backend_monitor(GMythMonitorHandler *
                                                 monitor);

static gboolean gmyth_monitor_handler_setup(GMythMonitorHandler * monitor,
                                            GIOChannel * channel);

void            gmyth_monitor_handler_close(GMythMonitorHandler * monitor);

G_DEFINE_TYPE(GMythMonitorHandler, gmyth_monitor_handler, G_TYPE_OBJECT);

static void
gmyth_monitor_handler_class_init(GMythMonitorHandlerClass * klass)
{
    GObjectClass   *gobject_class;
    GMythMonitorHandlerClass *gmonitor_class;

    gobject_class = (GObjectClass *) klass;
    gmonitor_class = (GMythMonitorHandlerClass *) gobject_class;

    gobject_class->dispose = gmyth_monitor_handler_dispose;
    gobject_class->finalize = gmyth_monitor_handler_finalize;

    gmonitor_class->backend_events_handler_signal_id =
        g_signal_new("backend-events-handler",
                     G_TYPE_FROM_CLASS(gmonitor_class),
                     G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE |
                     G_SIGNAL_NO_HOOKS, 0, NULL, NULL,
                     gmyth_marshal_VOID__INT_STRING, G_TYPE_NONE, 2,
                     G_TYPE_INT, G_TYPE_STRING);

    gmonitor_class->backend_events_handler =
        gmyth_monitor_handler_default_listener;

}

static void
gmyth_monitor_handler_init(GMythMonitorHandler * monitor)
{
    g_return_if_fail(monitor != NULL);

    monitor->event_sock = NULL;
    monitor->hostname = NULL;
    monitor->port = 0;
    monitor->actual_index = 0;
    monitor->allow_msgs_listener = FALSE;
    /*
     * it is used for signalizing the event socket consumer thread 
     */
    monitor->mutex = g_mutex_new();
}

static void
gmyth_monitor_handler_dispose(GObject * object)
{
    GMythMonitorHandler *monitor = GMYTH_MONITOR_HANDLER(object);

    gmyth_monitor_handler_close(monitor);

    monitor->allow_msgs_listener = FALSE;

    if (monitor->io_source != 0) {
        g_source_remove (monitor->io_source);
        monitor->io_source = 0;
    }

    /*
     * mutex to control access to the event socket consumer thread 
     */
    if (monitor->mutex != NULL) {
        // g_mutex_unlock( monitor->mutex );
        g_mutex_free(monitor->mutex);
        monitor->mutex = NULL;
    }

    if (monitor->event_sock != NULL) {
        g_object_unref(monitor->event_sock);
        monitor->event_sock = NULL;
    }

    if (monitor->hostname != NULL) {
        g_free(monitor->hostname);
        monitor->hostname = NULL;
    }


    if (monitor->backend_msgs != NULL) {
        g_hash_table_destroy(monitor->backend_msgs);
        monitor->backend_msgs = NULL;
    }

    /*
     * if ( io_watcher_cond != NULL ) { g_cond_free( io_watcher_cond );
     * io_watcher_cond = NULL; } 
     */


    G_OBJECT_CLASS(gmyth_monitor_handler_parent_class)->dispose(object);
}

static void
gmyth_monitor_handler_finalize(GObject * object)
{
    g_signal_handlers_destroy(object);

    G_OBJECT_CLASS(gmyth_monitor_handler_parent_class)->finalize(object);
}

/** 
 * Creates a new instance of GMyth Monitor Handler.
 * 
 * @return a new instance of the Monitor Handler. 
 */
GMythMonitorHandler *
gmyth_monitor_handler_new(void)
{
    GMythMonitorHandler *monitor =
        GMYTH_MONITOR_HANDLER(g_object_new
                              (GMYTH_MONITOR_HANDLER_TYPE, FALSE));

    return monitor;
}

/** 
 * Acquire the mutex to have access to the IO Watcher listener.
 * 
 * @param monitor The GMythMonitorHandler instance.
 * @param do_wait Tells the IO Watcher to wait on the GCond. (obsolete)
 * 
 * @return <code>true</code>, if the access to IO Watcher was acquired. 
 */
static          gboolean
myth_control_acquire_context(GMythMonitorHandler * monitor,
                             gboolean do_wait)
{

    gboolean        ret = TRUE;

    g_mutex_lock(monitor->mutex);

    return ret;

}

/** 
 * Release the mutex to have access to the IO Watcher listener.
 * 
 * @param monitor The GMythMonitorHandler instance.
 * 
 * @return <code>true</code>, if the access to IO Watcher was released. 
 */
static          gboolean
myth_control_release_context(GMythMonitorHandler * monitor)
{

    gboolean        ret = TRUE;

    g_mutex_unlock(monitor->mutex);

    return ret;
}

void
gmyth_monitor_handler_close(GMythMonitorHandler * monitor)
{
    monitor->allow_msgs_listener = FALSE;

#if 0
    if (monitor->monitor_th != NULL) {
        g_thread_pool_free(monitor->monitor_th, TRUE, FALSE);
        // g_thread_exit( monitor->monitor_th );
        /*
         * if ( monitor->monitor_th != NULL ) g_object_unref(
         * monitor->monitor_th ); 
         */
        monitor->monitor_th = NULL;
    }

    if (monitor->event_sock != NULL) {
        gmyth_socket_close_connection(monitor->event_sock);
    }
#endif

}

/** 
 * Opens connection the the Monitor socket on MythTV backend server,
 * where all status messages are notified to the client.
 * 
 * @param monitor The GMythMonitorHandler instance.
 * @param hostname The remote host name of the MythTV backend server.
 * @param port The remote port number of the MythTV backend server.
 * 
 * @return <code>true</code>, if the connection was successfully opened.
 */
gboolean
gmyth_monitor_handler_open(GMythMonitorHandler * monitor,
                           const gchar * hostname, gint port)
{
    gboolean        ret = TRUE;

    g_return_val_if_fail(hostname != NULL, FALSE);

    if (monitor->hostname != NULL) {
        g_free(monitor->hostname);
        monitor->hostname = NULL;
    }

    monitor->hostname = g_strdup(hostname);
    monitor->port = port;

    gmyth_debug("Monitor event socket --- hostname: %s, port %d\n",
                monitor->hostname, monitor->port);

    if (monitor->event_sock != NULL) {
        g_object_unref(monitor->event_sock);
        monitor->event_sock = NULL;
    }

    /*
     * configure the event socket
     */
    if (!gmyth_connect_to_backend_monitor(monitor)) {
        gmyth_debug("Connection to backend failed (Event Socket)!");
        ret = FALSE;
    } else {
        gmyth_debug ("Remote monitor event socket had been succesfully create");
    }

    return ret;
}

/** 
 * Reads the data got from the connection to the Monitor socket,
 * and looks for some important status messages.
 * 
 * @param monitor The GMythMonitorHandler instance.
 * @param strlist The GMythStringList instance got from the Monitor remote socket.
 * @param back_msg_action A string pointer to the status message detailed description.
 * 
 * @return The backend status message code ID.
 */
static          gint
gmyth_monitor_handler_is_backend_message(GMythMonitorHandler * monitor,
                                         GMythStringList * strlist,
                                         gchar ** back_msg_action)
{
    gint            msg_type = GMYTH_BACKEND_NO_MESSAGE;
    GString        *back_msg = NULL;

    if (gmyth_string_list_length(strlist) > 0) {

        back_msg = gmyth_string_list_get_string(strlist, 0);
        if (back_msg != NULL && back_msg->str != NULL &&
            strstr(back_msg->str, "BACKEND") != NULL) {
            gmyth_debug("MONITOR HANDLER - Received backend message = %s",
                        back_msg->str);
            *back_msg_action =
                gmyth_string_list_get_char_array(strlist, 1);

            if (back_msg_action != NULL) {

                if (g_strstr_len
                    (*back_msg_action, strlen(*back_msg_action),
                     "LIVETV_CHAIN")
                    || g_strstr_len(*back_msg_action,
                                    strlen(*back_msg_action),
                                    "RECORDING_LIST_CHANGE")
                    || g_strstr_len(*back_msg_action,
                                    strlen(*back_msg_action),
                                    "SCHEDULE_CHANGE")
                    || g_strstr_len(*back_msg_action,
                                    strlen(*back_msg_action),
                                    "LIVETV_WATCH")) {
                    gmyth_debug
                        ("MONITOR: message type == GMYTH_BACKEND_PROGRAM_INFO_CHANGED, msg = %s",
                         *back_msg_action);
                    msg_type = GMYTH_BACKEND_PROGRAM_INFO_CHANGED;
                } else if (g_strstr_len
                           (*back_msg_action, strlen(*back_msg_action),
                            "DONE_RECORDING")) {
                    gmyth_debug
                        ("MONITOR: message type == GMYTH_BACKEND_DONE_RECORDING, msg = %s",
                         *back_msg_action);
                    msg_type = GMYTH_BACKEND_DONE_RECORDING;
                } else if (g_strstr_len
                           (*back_msg_action, strlen(*back_msg_action),
                            "QUIT")) {
                    gmyth_debug
                        ("MONITOR: message type == GMYTH_BACKEND_STOP_LIVETV, msg = %s",
                         *back_msg_action);
                    msg_type = GMYTH_BACKEND_STOP_LIVETV;
                }

                /*
                 * g_hash_table_insert ( monitor->backend_msgs,
                 * &(monitor->actual_index), *back_msg_action ); 
                 */

            }
            /*
             * if 
             */
        }
        /*
         * if 
         */
        if (back_msg != NULL) {
            g_string_free(back_msg, TRUE);
            back_msg = NULL;
        }

    }                           /* if - Does Monitor got any message from
                                 * * * backend? */
    else {
        *back_msg_action = g_strdup("");
    }

    return msg_type;

}

static void
gmyth_monitor_handler_default_listener(GMythMonitorHandler * monitor,
                                       gint msg_code, gchar * message)
{
    // assert( message!= NULL ); 
    gmyth_debug("DEFAULT Signal handler ( msg = %s, code = %d )\n",
                message, msg_code);
}

static void
gmyth_monitor_handler_print(GString * str, gpointer ptr)
{
    gmyth_debug("Backend message event: %s --- ", str->str);
}

/** 
 * Opens connection the the Monitor socket on MythTV backend server,
 * where all status messages are notified to the client.
 * 
 * @param data Pointer to the GMythMonitorHandler.
 * 
 * @return Pointer to a gboolean <code>true</code> value, if the data was 
 * 	successfully read.
 */
static gboolean
gmyth_monitor_handler_listener (GIOChannel *io_channel,
                                GIOCondition io_cond,
                                gpointer data)
{
    GMythMonitorHandler *monitor;
    guint           recv = 0;
    gsize           len = 0;
    GMythStringList *strlist = NULL;
    gint bytes_sent = 0;

    monitor = (GMythMonitorHandler *) data;

    gmyth_debug("Entering MONITOR handler listener...");

    myth_control_acquire_context(monitor, TRUE);

    if (((io_cond & G_IO_HUP) != 0) ||
        ((io_cond & G_IO_ERR) != 0)) {
        goto clean_up;
    }


    gmyth_debug("Listening on Monitor socket...!\n");
    strlist = gmyth_string_list_new();

    len = gmyth_socket_read_stringlist(monitor->event_sock, strlist);
    if ((len > 0) && strlist != NULL && gmyth_string_list_length(strlist) > 0) {
        gchar *back_msg_action;
        gint msg_type;

        bytes_sent = gmyth_string_list_get_int(strlist, 0);
        // on  backend  error
        gmyth_debug ("received data buffer from IO event channel... %d strings gone!\n", len);
        recv += len;

        /*
         * debug purpose: prints out all the string list
         * elements
         */
        g_list_foreach(strlist->glist,
                       (GFunc) gmyth_monitor_handler_print,
                       NULL);

         back_msg_action = g_new0(gchar, 1);
         msg_type = gmyth_monitor_handler_is_backend_message(monitor,
            strlist,
            &back_msg_action);

        if (msg_type != GMYTH_BACKEND_NO_MESSAGE) {
            g_signal_emit(monitor,
                          GMYTH_MONITOR_HANDLER_GET_CLASS(monitor)->backend_events_handler_signal_id,
                          0, msg_type, back_msg_action);
        }

        if (back_msg_action != NULL)
            g_free(back_msg_action);

        g_object_unref(strlist);
    }

clean_up:
    myth_control_release_context(monitor);
    return TRUE;
}

/**
 * Opens connection events' socket the the Monitor socket on 
 * MythTV backend server.
 * 
 * @param monitor The GMythMonitorHandler instance.
 * 
 * @return <code>true</code>, if the socket was successfully opened.
 */
static gboolean
gmyth_connect_to_backend_monitor(GMythMonitorHandler * monitor)
{
    gboolean        ret = TRUE;

    monitor->event_sock = gmyth_socket_new();

    /*
     * Connects the socket, send Mythtv ANN Monitor and verify Mythtv
     * protocol version
     */
    if (!gmyth_socket_connect_to_backend_events(monitor->event_sock,
                                                monitor->hostname,
                                                monitor->port, FALSE)) {
        g_object_unref(monitor->event_sock);
        monitor->event_sock = NULL;
        ret = FALSE;
    }

    return ret;
}

/**
 * Opens connection the the Monitor socket on MythTV backend server,
 * where all status messages are notified to the client.
 *
 * @param monitor The GMythMonitorHandler instance.
 * @param channel The GIOChannel instance to the Monitor socket.
 *
 * @return Pointer to the boolean value, and it is <code>true</code> only if the 
 * GMythMonitorHandler could be configured.
 */
static gboolean
gmyth_monitor_handler_setup(GMythMonitorHandler * monitor,
                            GIOChannel * channel)
{
    gboolean ret = TRUE;

    if (channel != NULL) {
        monitor->allow_msgs_listener = TRUE;
        monitor->io_source = g_io_add_watch (channel, G_IO_IN | G_IO_ERR | G_IO_HUP,
                                             gmyth_monitor_handler_listener,
                                             monitor);
    } else {
        ret = FALSE;
    }
    return ret;
}

/**
 * Starts the MonitorHandler thread to the GIOWatcher.
 *
 * @param monitor The GMythMonitorHandler instance.
 *
 * @return <code>true</code>, if the MonitorHandler was started.
 */
gboolean
gmyth_monitor_handler_start(GMythMonitorHandler * monitor)
{
    gboolean        ret = TRUE;

    if (!(ret = g_thread_supported())) {
        gmyth_debug("Thread system wasn't initialized, starting NOW!!!");
        g_thread_init(NULL);
    }

    ret =
        gmyth_monitor_handler_setup(monitor,
                                    monitor->event_sock->sd_io_ch);
    if (ret) {
        gmyth_debug
            ("\n[%s]\tOK! Starting listener on the MONITOR event socket...[thread location = %p]\n",
             __FUNCTION__, g_thread_self());
    } else {
        gmyth_debug
            ("\n[%s]\tERROR! Coudn't start listener on the MONITOR event socket...[thread location = %p]\n",
             __FUNCTION__, g_thread_self());
        ret = FALSE;
    }

    gmyth_debug
        ("[%s] Watch listener function over the IO control channel? %s!!!\n",
         __FUNCTION__, (ret == TRUE ? "YES" : "NO"));

    return ret;
}
