// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief pi0/K0 spectra
  class BESIII_2022_I2513076 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESIII_2022_I2513076);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // projection
      declare(UnstableParticles(Cuts::pid==PID::PI0 || Cuts::pid==PID::K0S), "UFS");
      // find beam energy
      size_t ih = 1;
      for (double eVal : allowedEnergies()) {

        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;

        book(_h[en+"_1"], 1, 1, ih);
        book(_h[en+"_2"], 2, 1, ih);
        book(_c[en], "_aux_sigma_"+en);
        ++ih;
      }
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      _c[_sqs]->fill();
      for (const Particle& p : apply<UnstableParticles>(event, "UFS").particles()) {
        const double pp = p.p3().mod();
        if (p.pid()==PID::PI0) _h[_sqs+"_1"]->fill(pp);
        else                   _h[_sqs+"_2"]->fill(pp);
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_h, crossSectionPerEvent());
      scale(_c, crossSectionPerEvent());
      for (const auto& item : _c) {
        scale(_h[item.first+"_1"s], 1.0/item.second->sumW());
        scale(_h[item.first+"_2"s], 1.0/item.second->sumW());
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    map<string,CounterPtr> _c;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BESIII_2022_I2513076);

}
