// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/Config.h"
#include "Puma/SysCall.h"
#include "Puma/StrCol.h"
#include "Puma/OptsParser.h"
#include <stdlib.h>
#include <string.h>
#include <sstream>


namespace Puma {


// print error messages

#define CONFIG_FILE \
   _err << sev_error << "in config file `" << file << "': "

#define COMMAND_LINE \
   _err << sev_error << "at command line: "

#define ERROR_MSG(__mesg) \
   _err << __mesg << endMessage 

#define ERROR__missing(__name) \
   COMMAND_LINE; \
   ERROR_MSG ("missing `" << __name << "' for `-A'")

#define ERROR__empty(__what,__name) \
   COMMAND_LINE; \
   ERROR_MSG ("no " << __what << " given for `" << __name << "'")

// options:
// -A<PREDICATE>(<ANSWER>)                      // Assert a preprocessor predicate.
// -D<NAME>[=<BODY>]                            // Define a preprocessor macro.
// -U<NAME>                                     // Undefine a preprocessor macro.
// -I<PATH>                                     // Add a new include path.
//OLD: -p|--path <SRC> [<DEST>]                 // Add a new source dest paths pair.
// -p|--path <SRC>                              // Add a new source path.
// -d|--dest <DEST>                             // Add a new dest path.
// -w|--write-protected <PATH_PATTERN>          // Add a new write protected path.
// -s|--suffix <SUFFIX>                         // Set the file save suffix.
// -e|--extension <EXTENSION>                   // Set the extension for source files.
// --save-overwrite|rename-old|new-suffix       // set the file save mode
// --lang-c|ec++|c++|ac++                       // set the source code language
// --skip-bodies-all                            // skip parsing function bodies
// --skip-bodies-tpl                            // skip parsing function bodies of templates
// --skip-bodies-non-prj                        // skip parsing non-project function bodies
// --skip-bodies-non-prim                       // skip parsing function bodies in non-primary files
// --match-expr                                 // parse match expressions
// --vc                                         // support MS Visual C++ extensions
// --gnu                                        // support gnu extensions
// --gnu-2.95
// --gnu-std-hack
// --gnu-nested-fct
// --gnu-condition-scope
// --gnu-param-decl
// --gnu-fct-decl
// --gnu-param-scope
// --gnu-default-args
// --gnu-extended-asm
// --gnu-extended-expr
// --gnu-long-long
// --gnu-name-scope
// --gnu-implicit-int
// --gnu-fct-attribute
// --gnu-if-then-expr

OptsParser::Option pumaOpts[] = {
   {Config::PRE_ASSERT, "A", NULL, "Assert a preprocessor predicate", OptsParser::AT_MANDATORY},
   {Config::PRE_DEFINE, "D", NULL, "Define a preprocessor macro", OptsParser::AT_MANDATORY},
   {Config::PRE_UNDEFINE, "U", NULL, "Undefine a preprocessor macro", OptsParser::AT_MANDATORY},
   {Config::PRE_LOCK_MACRO, 0, "lock-macro", "Define an unchangeable preprocessor macro", OptsParser::AT_MANDATORY},
   {Config::PRE_INCLUDE, "I", NULL, "Add new include path", OptsParser::AT_MANDATORY},
   {Config::PROJ_PATH, "p", "path", "Path to project source", OptsParser::AT_MANDATORY},
   {Config::PROJ_DESTINATION, "d", "dest", "Path to destination for modified sources", OptsParser::AT_MANDATORY},
   {Config::SET_SUFFIX, "s", "suffix", "Set file save suffix", OptsParser::AT_MANDATORY},
   {Config::SET_EXTENSION, "e", "extension", "Set the extension for source files", OptsParser::AT_MANDATORY},
   {Config::SET_WRITE_PROTECTED, "w", "write-protected", "Add a new write protected path", OptsParser::AT_MANDATORY},
   {Config::CFG_FILE, NULL, "config", "Full path to a config file", OptsParser::AT_MANDATORY},
   {Config::SET_OPTION, NULL, "save-overwrite", "Overwrite old files", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "rename-old", "Rename old files", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "new-suffix", "Append new suffix on old files", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "lang-c", "Input language is C", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "lang-ec++", "Input language is EC++", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "lang-c++", "Input language is C++", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "lang-ac++", "Input language is AC++", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "skip-bodies-all", "Skip parsing function bodies", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "skip-bodies-tpl", "Skip parsing function bodies of templates", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "skip-bodies-non-prj", "Skip parsing non-project function bodies", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "skip-bodies-non-prim", "Skip parsing function bodies in non-primary files", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "real-instances", "Do real template instantiation", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "match-expr", "Parse match expression", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-nested-fct", "", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-condition-scope", "", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-param-decl", "", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-fct-decl", "", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-param-scope", "", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-default-args", "", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-extended-asm", "", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-extended-expr", "", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-long-long", "", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-name-scope", "", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-implicit-int", "", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-fct-attribute", "", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-if-then-expr", "", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-std-hack", "", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu", "Support gnu extensions", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "gnu-2.95", "Support gnu extensions for g++ 2.95", OptsParser::AT_NONE},
   {Config::SET_OPTION, NULL, "vc", "Support Visual C++ extensions", OptsParser::AT_NONE},
   {Config::SET_OPTION_ARG, NULL, "import-handler", "Handler for #import directives", OptsParser::AT_MANDATORY},
   {Config::SET_OPTION_ARG, NULL, "include", "Always include file", OptsParser::AT_MANDATORY},
   {Config::SET_OPTION_ARG, NULL, "size-type", "Internal type for size_t", OptsParser::AT_MANDATORY},
   {Config::SET_OPTION_ARG, NULL, "ptrdiff-type", "Internal type for ptrdiff_t", OptsParser::AT_MANDATORY},
   {Config::SET_OPTION_ARG, NULL, "inhibit-macro", "Prevent a preprocessor macro for being defined", OptsParser::AT_MANDATORY},
   {0, 0, 0, 0, OptsParser::AT_NONE}
};


void Config::Read (const char *file) {
  FileHandle fp;
  char buffer[4096];
  string optstr ("");

  // file not given, read from $PUMA_CONFIG or /etc/puma.config
  if (! file)
    file = getenv ("PUMA_CONFIG");
    
  if (file) {
    if ((fp = SysCall::fopen (file, "r")) == NULL) {
      ERROR_MSG ("cannot open config file `" << file << "'");
      return;
    }
  } else {
    file = "/etc/puma.config"; // default location
    if ((fp = SysCall::fopen (file, "r")) == NULL) {
      return;
    }
  }
  
  while (fgets (buffer, 4000, fp)) {
    // skip comments and newlines
    if (buffer[0] != '#' || buffer[0] != '\n') {
      optstr += " ";
      optstr += buffer;
    }
  }

  SysCall::fclose (fp, &_err);

  OptsParser optp (optstr, pumaOpts);

  // process the given options
  while (optp.getOption () != OptsParser::FINISH) {
    Process (optp, true);
  }
}


// read options from command line
void Config::Read (int &argc, char **&argv) {
  OptsParser optp (argc, argv, pumaOpts);
  char** newargv = new char*[argc];
  int    newargc = 1;
        
  // process the given options
  while (optp.getOption () != OptsParser::FINISH) {
    if (Process (optp) == false) {
      // this is not processed by PUMA - copy
      newargv[newargc++] = argv[optp.getCurrentArgNum ()];
      //cout << "add to newargv: " << argv[optp.getCurrentArgNum()] << endl;
    }
  }
        
  // make argv point to new argv and set argc to value of new argc
  argc = newargc;
  for (int i = 1; i < argc; i++)
    argv[i] = newargv[i];
  delete[] newargv;
}


bool Config::Process (OptsParser &parser, bool unescape) {
  bool res=true;
  string arg;
  if (unescape) {
    arg = Unescape (parser.getArgument ());
  } else {
    arg = parser.getArgument ();
  }
        
  switch (parser.getResult ()) {
    case PRE_ASSERT:
      ProcessAssert (arg);
      break;
    case PRE_DEFINE:
      ProcessDefine (arg);
      break;
    case PRE_LOCK_MACRO:
      ProcessLockMacro (arg);
      break;
    case PRE_UNDEFINE:
      ProcessUndefine (arg);
      break;
    case PRE_INCLUDE:
      ProcessPathArgument ("-I", arg);
      break;
    case PROJ_PATH:
      ProcessPathArgument ("-p", arg);
      break;
    case PROJ_DESTINATION:
      ProcessPathArgument ("-d", arg);                        
      break;
    case CFG_FILE:
      ProcessConfigFile ("--config", arg);
      break;
    case SET_SUFFIX:
      ProcessArgument ("-s", arg);
      break;
    case SET_EXTENSION:
      ProcessArgument ("-e", arg);
      break;
    case SET_WRITE_PROTECTED:
      ProcessArgument ("-w", arg);
      break;
    case SET_OPTION:
      //cout << "option: " << parser.getOptionName() << endl;
      Add (parser.getOptionName ().c_str ());
      break;
    case SET_OPTION_ARG:
      //cout << "option: " << parser.getOptionName() << endl;
      Add (parser.getOptionName ().c_str (), arg.c_str ());
      break;
    case OptsParser::UNKNOWN:
    case OptsParser::NOOPTION:
      res = false;
      break;
    default:
    case OptsParser::ERR:
      break;
  }
  return res;
}


bool Config::ProcessConfigFile (const string &opt, const string &arg) {
  bool res = false;

  if (arg.empty () == false) {
    Read (arg.c_str ());
    //cout << opt << " " << arg << endl;
    res = true;
  } else { 
    ERROR__empty ("file name", opt.c_str ()); 
  }
        
  return res;
}


bool Config::ProcessArgument (const string &opt, const string &arg) {
  bool res = false;

  if (arg.empty () == false) {
    Add (opt.c_str (), arg.c_str ());
    //cout << opt << " " << arg << endl;
    res = true;
  } else { 
    ERROR__empty ("argument", opt.c_str ()); 
  }
        
  return res;
}


bool Config::ProcessPathArgument (const string &opt, const string &arg) {
  bool res = false;

  if (arg.empty () == false) {
#if defined (WIN32)
    char *tmp = StrCol::dup (arg.c_str ());
    if (tmp != 0) {
      SysCall::MakeUnixPath (tmp);
      Add (opt.c_str (), tmp);
      delete[] tmp;
    }                
#else
    Add (opt.c_str (), arg.c_str ());
    //cout << opt << " " << arg << endl;
    res = true;
#endif
  } else { 
    ERROR__empty ("argument", opt.c_str ()); 
  }
        
  return res;
}


bool Config::ProcessUndefine (const string &arg) {
  bool res = false;

  if (Valid (arg.c_str ())) {
    Add ("-U", arg.c_str ());
    //cout << "-U " << arg << endl;
    res = true;
  } else { 
    ERROR_MSG ("invalid name for -U"); 
  }
        
  return res;
}


bool Config::ProcessDefine (const string &arg) {
  bool res = false;
  string::size_type pos;
  string name, body;

  if ((pos = arg.find_first_of ('=',0)) != string::npos) {
    body = arg.substr (pos + 1, arg.length () - pos);
    name = arg.substr (0, pos);
  } else {
    body = "1";
    name = arg;
  }
        
  if (Valid (name.c_str ())) {
    Add ("-D", name.c_str (), body.c_str ());
    //cout << "-D " << name << "=" << body << endl;
    res = true;
  } else { 
    ERROR_MSG ("invalid name for -D"); 
  }
        
  return res;
}


bool Config::ProcessLockMacro (const string &arg) {
  bool res = false;
  string::size_type pos;
  string name, body;

  if ((pos = arg.find_first_of ('=',0)) != string::npos) {
    body = arg.substr (pos + 1, arg.length () - pos);
    name = arg.substr (0, pos);
  } else {
    body = "1";
    name = arg;
  }
        
  if (Valid (name.c_str ())) {
    Add ("--lock-macro", name.c_str (), body.c_str ());
    //cout << "--lock-macro " << name << "=" << body << endl;
    res = true;
  } else { 
    ERROR_MSG ("invalid name for --lock-macro"); 
  }
        
  return res;
}


bool Config::ProcessAssert (const string &arg) {
  bool res = false;
  string::size_type pos, end;
  string name, body;
        
  if ((pos = arg.find_first_of ('(',0)) != string::npos) {
    if ((end = arg.find_first_of (')',pos)) != string::npos) {
      name = arg.substr (0, pos);
      body = arg.substr (pos + 1, end - pos - 1);
      if (StrCol::onlySpaces (body.c_str ()) == false) {
        if (Valid (name.c_str ())) {
          Add ("-A", name.c_str (), body.c_str ());
          //cout << "-A " << name << "(" << body << ")" << endl;
          res = true;
        } else { 
          ERROR_MSG  ("invalid name for -A"); 
        }
      } else { 
        ERROR__empty ("answer text", "-A"); 
      }
    } else { 
      ERROR__missing (")"); 
    }
  } else { 
    ERROR__missing ("("); 
  }

  return res;
}


bool Config::Valid (const char *name, const char *file) const {
  if (name) {
    const char *reject = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_$0123456789";
    if (strlen (name) && ! strcspn (name, reject) &&
        ! (*name >= '0' && *name <= '9'))
      return true;
  }
  if (file) 
    CONFIG_FILE;
  else 
    COMMAND_LINE;
  ERROR_MSG ("`" << name << "' is not a valid C identifier");
  return false;
}


void Config::Join (const Config &c) {
  unsigned num = c.Options ();
  for (unsigned i = 0; i < num; i++)
    Add (c.Option (i));
}


const ConfOption *Config::Option (const char *name) const {
  if (name) {
    for (unsigned i = Options (); i > 0; i--) {
      const ConfOption *o = Option (i-1);
      if (! strcmp (o->Name (), name))
        return o;
    }
  }
  return (const ConfOption*)0;
}


string Config::Unescape (const string& arg) {
  // Replacement rules:
  // 1. '\' '"' => '"'
  // 2. '\' '\' => '\' 
  // 3. '"'     => ''
  ostringstream res;
  bool escaped = false;
  for (string::size_type i = 0; i < arg.length(); i++) {
    char c = arg[i];
    if (c == '"') {
      if (escaped) {
        escaped = false;
        res << c;
      } 
    } else if (c == '\\') {
      if (escaped) {
        escaped = false;
        res << c;
      } else {
        escaped = true;
      }
    } else {
      if (escaped) {
        escaped = false;
        res << "\\";
      }
      res << c;
    }
  }
  return res.str();
}


} // namespace Puma
