/*
 * pi3's Linux kernel Runtime Guard
 *
 * Component:
 *  - Intercept 'generic_permission' function
 *
 * Notes:
 *  - Enforce Exploit Detection validation
 *
 * Caveats:
 *  - Originally, this file was placing hooks on 'may_open' function.
 *    Unfortunately, GCC (8+) might enable ISRA optimization when -Ox
 *    switch was used. During kernel compilation it is usually enabled,
 *    and as a side effect we have ISRA optimization as well
 *    (undesired for LKRG). ISRA performs interprocedural scalar
 *    replacement of aggregates, removal of unused parameters and
 *    replacement of parameters passed by reference by parameters passed
 *    by value. Since it's a very invasive modification ISRA changes
 *    symbol name of the functions which was modified.
 *    Alexander (Solar Designer) pointed out that in fact we could hook
 *    inode_permission() instead. This function is exported and
 *    additionally, we'll improve our coverage since it is called on more
 *    cases than 'may_open', including things such as permission checks
 *    on creating/removing of directories, (un)linking of files, and
 *    searching for files in directories with restricted permissions.
 *    LKRG hooks 'generic_permission' since this function is also exported
 *    and is called by inode_permission() after various checks have been
 *    made. It is also called in a few other specialized cases.
 *
 * Timeline:
 *  - Replace 'may_open' with 'generic_permission': 17.IX.2018
 *  - Created: 04.X.2017
 *
 * Author:
 *  - Adam 'pi3' Zabrocki (http://pi3.com.pl)
 *
 */

#ifndef P_LKRG_EXPLOIT_DETECTION_GENERIC_PERMISSION_H
#define P_LKRG_EXPLOIT_DETECTION_GENERIC_PERMISSION_H

/* per-instance private data */
struct p_generic_permission_data {
    ktime_t entry_stamp;
};


int p_generic_permission_ret(struct kretprobe_instance *p_ri, struct pt_regs *p_regs);
int p_generic_permission_entry(struct kretprobe_instance *p_ri, struct pt_regs *p_regs);
int p_install_generic_permission_hook(int p_isra);
void p_uninstall_generic_permission_hook(void);

#endif
