// SPDX-License-Identifier: GPL-2.0-or-later
//
// This software may be used and distributed according to the terms of the
// GNU General Public License version 2 or any later version.
//
// Copyright 2020  Pacien TRAN-GIRARD <pacien.trangirard@pacien.net>

use crate::analytics::sorts::{index_map, NodeSort};
use crate::graph::{Graph, GraphReadError, Parents, Revision, NULL_REVISION};
use std::cmp::max;
use std::collections::{HashMap, HashSet};
use std::result::Result;
use std::result::Result::Ok;

#[derive(Debug)]
pub struct SortIndicators {
    pub xc: usize,
    pub xp: usize,
    pub xa: usize,
    pub xf: usize,
}

pub fn sort_indicators(
    graph: &impl Graph,
    node_sort: &NodeSort,
) -> Result<SortIndicators, GraphReadError> {
    Ok(SortIndicators {
        xc: xc(graph, node_sort)?,
        xp: xp(graph, node_sort)?,
        xa: xa(graph, node_sort)?,
        xf: xf(graph, node_sort)?,
    })
}

///// CUT-BASED INDICATORS

/// X_c(s) := max_i #{ j | j<i \exists k>i s_j->s_k }
/// where a->b := a \in parents(b)
///
/// The X_c indicator corresponds to the maximum number of revisions having one
/// or more child in the other part of the cut during the iteration.
pub fn xc(
    graph: &impl Graph,
    node_sort: &NodeSort,
) -> Result<usize, GraphReadError> {
    let mut pending_parents: HashSet<Revision> = HashSet::new();
    pending_parents.insert(NULL_REVISION);
    let mut max_card = pending_parents.len();

    for rev in node_sort.iter().rev() {
        pending_parents.remove(&rev);
        max_card = max(pending_parents.len(), max_card);
        let Parents([p1, p2]) = graph.parents(*rev)?;
        pending_parents.insert(p1);
        pending_parents.insert(p2);
    }

    Ok(max_card - 1) // -1 to exclude NULL_REVISION
}

/// X_p(s) := max_i #{ k | k>i \exists j<i s_j->s_k }
/// where a->b := a \in parents(b)
///
/// The X_p indicator corresponds to the maximum number of revisions
/// simultaneously having one or more parent in the other part of the cut
/// during the iteration.
pub fn xp(
    graph: &impl Graph,
    node_sort: &NodeSort,
) -> Result<usize, GraphReadError> {
    let sort_indices = index_map(node_sort);
    let mut max_card = 0;
    let mut referenced_child_count = 0;
    let mut child_count_by_rev = HashMap::with_capacity(node_sort.len());

    for i in (1..node_sort.len()).rev() {
        let rev = node_sort[i];
        let next_rev = node_sort[i - 1];

        referenced_child_count -= child_count_by_rev.get(&rev).unwrap_or(&0);
        max_card = max(referenced_child_count, max_card);

        match graph.parents(rev)? {
            Parents([NULL_REVISION, NULL_REVISION]) => {}
            Parents([p, NULL_REVISION]) | Parents([NULL_REVISION, p]) => {
                if p != next_rev {
                    referenced_child_count += 1;
                    *child_count_by_rev.entry(p).or_insert(0) += 1;
                }
            }
            Parents([p1, p2]) => {
                if p1 != next_rev || p2 != next_rev {
                    referenced_child_count += 1;
                    if sort_indices[&p1] < sort_indices[&p2] {
                        *child_count_by_rev.entry(p1).or_insert(0) += 1;
                    } else {
                        *child_count_by_rev.entry(p2).or_insert(0) += 1;
                    }
                }
            }
        }
    }

    Ok(max_card)
}

/// X_a(s) := max_i #{ (j,k) | j<i<k s_j->s_k }
/// where a->b := a \in parents(b)
///
/// The X_a indicator corresponds to the maximum number of arcs linking parents
/// and children crossed by the cut during the iteration.
pub fn xa(
    graph: &impl Graph,
    node_sort: &NodeSort,
) -> Result<usize, GraphReadError> {
    let mut max_card = 0;
    let mut referenced_child_count = 0;
    let mut child_count_by_rev = HashMap::with_capacity(node_sort.len());

    for i in (1..node_sort.len()).rev() {
        let rev = node_sort[i];
        let next_rev = node_sort[i - 1];

        referenced_child_count -= child_count_by_rev.get(&rev).unwrap_or(&0);
        max_card = max(referenced_child_count, max_card);

        let Parents([p1, p2]) = graph.parents(rev)?;

        if p1 != NULL_REVISION && p1 != next_rev {
            *child_count_by_rev.entry(p1).or_insert(0) += 1;
            referenced_child_count += 1;
        }

        if p2 != NULL_REVISION && p2 != next_rev && p2 != p1 {
            *child_count_by_rev.entry(p2).or_insert(0) += 1;
            referenced_child_count += 1;
        }
    }

    Ok(max_card)
}

/// X_f(s) := max_i #{ k | i<k \forall j s_j->s_k => j<i }
/// where a->b := a \in parents(b)
///
/// The X_f indicator corresponds to the maximum number of revisions for which
/// all parents are on the other side of the cut during the iteration.
/// It is similar to the X_p indicator but for nodes that have all parents on
/// the other side.
pub fn xf(
    graph: &impl Graph,
    node_sort: &NodeSort,
) -> Result<usize, GraphReadError> {
    let mut pending_children: HashSet<Revision> = HashSet::new();
    let mut max_card = 0;

    for rev in node_sort.iter().rev() {
        pending_children
            .retain(|child| !graph.parents(*child).unwrap().contains(*rev));
        max_card = max(pending_children.len(), max_card);
        pending_children.insert(*rev);
    }

    Ok(max_card)
}

#[cfg(test)]
mod tests {
    use crate::analytics::sorts::{stable_sort, NodeSort};
    use crate::analytics::sortsindicators::{xa, xc, xf, xp};
    use crate::graph::{Graph, Revision, SizedGraph, StableOrderGraph};
    use crate::testing::graph_io::read_graph;
    use crate::testing::ordering::NodeIDComparator;
    use std::io::{BufRead, Cursor};

    fn make_dummy_graph() -> impl Graph + SizedGraph + StableOrderGraph {
        // rnd2.graph
        let def = "
0000000000000000000000000000000000000100 0000000000000000000000000000000000000000 0000000000000000000000000000000000000000
0000000000000000000000000000000000000101 0000000000000000000000000000000000000100 0000000000000000000000000000000000000000
0000000000000000000000000000000000000102 0000000000000000000000000000000000000100 0000000000000000000000000000000000000101
0000000000000000000000000000000000000103 0000000000000000000000000000000000000102 0000000000000000000000000000000000000100
0000000000000000000000000000000000000104 0000000000000000000000000000000000000103 0000000000000000000000000000000000000000
0000000000000000000000000000000000000105 0000000000000000000000000000000000000101 0000000000000000000000000000000000000100
0000000000000000000000000000000000000106 0000000000000000000000000000000000000100 0000000000000000000000000000000000000000
0000000000000000000000000000000000000107 0000000000000000000000000000000000000101 0000000000000000000000000000000000000100
0000000000000000000000000000000000000108 0000000000000000000000000000000000000105 0000000000000000000000000000000000000103
0000000000000000000000000000000000000109 0000000000000000000000000000000000000107 0000000000000000000000000000000000000000
0000000000000000000000000000000000000110 0000000000000000000000000000000000000108 0000000000000000000000000000000000000109
0000000000000000000000000000000000000111 0000000000000000000000000000000000000108 0000000000000000000000000000000000000110
0000000000000000000000000000000000000112 0000000000000000000000000000000000000104 0000000000000000000000000000000000000000
0000000000000000000000000000000000000113 0000000000000000000000000000000000000102 0000000000000000000000000000000000000000
0000000000000000000000000000000000000114 0000000000000000000000000000000000000109 0000000000000000000000000000000000000106
0000000000000000000000000000000000000115 0000000000000000000000000000000000000102 0000000000000000000000000000000000000000
0000000000000000000000000000000000000116 0000000000000000000000000000000000000105 0000000000000000000000000000000000000000
0000000000000000000000000000000000000117 0000000000000000000000000000000000000105 0000000000000000000000000000000000000000
0000000000000000000000000000000000000118 0000000000000000000000000000000000000101 0000000000000000000000000000000000000101
0000000000000000000000000000000000000119 0000000000000000000000000000000000000101 0000000000000000000000000000000000000000
0000000000000000000000000000000000000120 0000000000000000000000000000000000000119 0000000000000000000000000000000000000105
0000000000000000000000000000000000000121 0000000000000000000000000000000000000109 0000000000000000000000000000000000000108
0000000000000000000000000000000000000122 0000000000000000000000000000000000000103 0000000000000000000000000000000000000000
0000000000000000000000000000000000000123 0000000000000000000000000000000000000106 0000000000000000000000000000000000000000
0000000000000000000000000000000000000124 0000000000000000000000000000000000000112 0000000000000000000000000000000000000115
0000000000000000000000000000000000000125 0000000000000000000000000000000000000118 0000000000000000000000000000000000000124
0000000000000000000000000000000000000126 0000000000000000000000000000000000000120 0000000000000000000000000000000000000000
0000000000000000000000000000000000000127 0000000000000000000000000000000000000100 0000000000000000000000000000000000000103
0000000000000000000000000000000000000128 0000000000000000000000000000000000000123 0000000000000000000000000000000000000120
0000000000000000000000000000000000000129 0000000000000000000000000000000000000103 0000000000000000000000000000000000000105
0000000000000000000000000000000000000130 0000000000000000000000000000000000000101 0000000000000000000000000000000000000000
0000000000000000000000000000000000000131 0000000000000000000000000000000000000100 0000000000000000000000000000000000000000
0000000000000000000000000000000000000132 0000000000000000000000000000000000000106 0000000000000000000000000000000000000100
0000000000000000000000000000000000000133 0000000000000000000000000000000000000107 0000000000000000000000000000000000000000
0000000000000000000000000000000000000134 0000000000000000000000000000000000000114 0000000000000000000000000000000000000000
0000000000000000000000000000000000000135 0000000000000000000000000000000000000107 0000000000000000000000000000000000000130
0000000000000000000000000000000000000136 0000000000000000000000000000000000000104 0000000000000000000000000000000000000000
0000000000000000000000000000000000000137 0000000000000000000000000000000000000121 0000000000000000000000000000000000000000
0000000000000000000000000000000000000138 0000000000000000000000000000000000000135 0000000000000000000000000000000000000000
0000000000000000000000000000000000000139 0000000000000000000000000000000000000131 0000000000000000000000000000000000000114
0000000000000000000000000000000000000140 0000000000000000000000000000000000000135 0000000000000000000000000000000000000135
0000000000000000000000000000000000000141 0000000000000000000000000000000000000140 0000000000000000000000000000000000000101
0000000000000000000000000000000000000142 0000000000000000000000000000000000000141 0000000000000000000000000000000000000102
0000000000000000000000000000000000000143 0000000000000000000000000000000000000142 0000000000000000000000000000000000000103
0000000000000000000000000000000000000144 0000000000000000000000000000000000000143 0000000000000000000000000000000000000104
0000000000000000000000000000000000000145 0000000000000000000000000000000000000144 0000000000000000000000000000000000000105
0000000000000000000000000000000000000146 0000000000000000000000000000000000000145 0000000000000000000000000000000000000106
0000000000000000000000000000000000000147 0000000000000000000000000000000000000146 0000000000000000000000000000000000000107
0000000000000000000000000000000000000148 0000000000000000000000000000000000000147 0000000000000000000000000000000000000108
0000000000000000000000000000000000000149 0000000000000000000000000000000000000148 0000000000000000000000000000000000000109
0000000000000000000000000000000000000150 0000000000000000000000000000000000000149 0000000000000000000000000000000000000110
0000000000000000000000000000000000000151 0000000000000000000000000000000000000150 0000000000000000000000000000000000000111
0000000000000000000000000000000000000152 0000000000000000000000000000000000000151 0000000000000000000000000000000000000112
0000000000000000000000000000000000000153 0000000000000000000000000000000000000152 0000000000000000000000000000000000000113
0000000000000000000000000000000000000154 0000000000000000000000000000000000000153 0000000000000000000000000000000000000114
0000000000000000000000000000000000000155 0000000000000000000000000000000000000154 0000000000000000000000000000000000000115
0000000000000000000000000000000000000156 0000000000000000000000000000000000000155 0000000000000000000000000000000000000116
0000000000000000000000000000000000000157 0000000000000000000000000000000000000156 0000000000000000000000000000000000000117
0000000000000000000000000000000000000158 0000000000000000000000000000000000000157 0000000000000000000000000000000000000118
0000000000000000000000000000000000000159 0000000000000000000000000000000000000158 0000000000000000000000000000000000000119
0000000000000000000000000000000000000160 0000000000000000000000000000000000000159 0000000000000000000000000000000000000120
0000000000000000000000000000000000000161 0000000000000000000000000000000000000160 0000000000000000000000000000000000000121
0000000000000000000000000000000000000162 0000000000000000000000000000000000000161 0000000000000000000000000000000000000122
0000000000000000000000000000000000000163 0000000000000000000000000000000000000162 0000000000000000000000000000000000000123
0000000000000000000000000000000000000164 0000000000000000000000000000000000000163 0000000000000000000000000000000000000124
0000000000000000000000000000000000000165 0000000000000000000000000000000000000164 0000000000000000000000000000000000000125
0000000000000000000000000000000000000166 0000000000000000000000000000000000000165 0000000000000000000000000000000000000126
0000000000000000000000000000000000000167 0000000000000000000000000000000000000166 0000000000000000000000000000000000000127
0000000000000000000000000000000000000168 0000000000000000000000000000000000000167 0000000000000000000000000000000000000128
0000000000000000000000000000000000000169 0000000000000000000000000000000000000168 0000000000000000000000000000000000000129
0000000000000000000000000000000000000170 0000000000000000000000000000000000000169 0000000000000000000000000000000000000130
0000000000000000000000000000000000000171 0000000000000000000000000000000000000170 0000000000000000000000000000000000000131
0000000000000000000000000000000000000172 0000000000000000000000000000000000000171 0000000000000000000000000000000000000132
0000000000000000000000000000000000000173 0000000000000000000000000000000000000172 0000000000000000000000000000000000000133
0000000000000000000000000000000000000174 0000000000000000000000000000000000000173 0000000000000000000000000000000000000134
0000000000000000000000000000000000000175 0000000000000000000000000000000000000174 0000000000000000000000000000000000000135
0000000000000000000000000000000000000176 0000000000000000000000000000000000000175 0000000000000000000000000000000000000136
0000000000000000000000000000000000000177 0000000000000000000000000000000000000176 0000000000000000000000000000000000000137
0000000000000000000000000000000000000178 0000000000000000000000000000000000000177 0000000000000000000000000000000000000138
0000000000000000000000000000000000000179 0000000000000000000000000000000000000178 0000000000000000000000000000000000000139
";
        let lines = Cursor::new(def).lines();
        read_graph::<_, NodeIDComparator>(lines).unwrap()
    }

    fn make_dummy_sort(
        graph: &(impl Graph + SizedGraph + StableOrderGraph),
    ) -> NodeSort {
        stable_sort(graph, (graph.nb_nodes() - 1) as Revision).unwrap()
    }

    #[test]
    fn test_xc() {
        /// X_c(s) := max_i #{ j | j<i \exists k>i s_j->s_k }
        fn naive(graph: &impl Graph, node_sort: &NodeSort) -> usize {
            let mut max_card = 0;
            for i in 0..node_sort.len() {
                let mut q_e_card = 0;
                for rev_j in node_sort[..i].iter() {
                    for rev_k in node_sort[i + 1..].iter() {
                        if graph.parents(*rev_k).unwrap().contains(*rev_j) {
                            q_e_card += 1;
                            break;
                        }
                    }
                }
                if q_e_card > max_card {
                    max_card = q_e_card;
                }
            }
            max_card
        }

        let graph = make_dummy_graph();
        let sort = make_dummy_sort(&graph);
        assert_eq!(xc(&graph, &sort).unwrap(), naive(&graph, &sort));
    }

    #[test]
    fn test_xp() {
        /// X_p(s) := max_i #{ k | k>i \exists j<i s_j->s_k }
        pub fn naive(graph: &impl Graph, node_sort: &NodeSort) -> usize {
            let mut max_card = 0;
            for i in 0..node_sort.len() {
                let mut q_p_card = 0;
                for rev_k in node_sort[i + 1..].iter() {
                    for rev_j in node_sort[0..i].iter() {
                        if graph.parents(*rev_k).unwrap().contains(*rev_j) {
                            q_p_card += 1;
                            break;
                        }
                    }
                }
                println!("{}", q_p_card);
                if q_p_card > max_card {
                    max_card = q_p_card;
                }
            }
            max_card
        }

        let graph = make_dummy_graph();
        let sort = make_dummy_sort(&graph);
        println!("node_sort: {:#?}", sort);
        assert_eq!(xp(&graph, &sort).unwrap(), naive(&graph, &sort));
    }

    #[test]
    fn test_xa() {
        /// X_a(s) := max_i #{ (j,k) | j<i<k s_j->s_k }
        pub fn naive(graph: &impl Graph, node_sort: &NodeSort) -> usize {
            let mut max_card = 0;
            for i in 0..node_sort.len() {
                let mut q_a_card = 0;
                for rev_j in node_sort[..i].iter() {
                    for rev_k in node_sort[i + 1..].iter() {
                        if graph.parents(*rev_k).unwrap().contains(*rev_j) {
                            q_a_card += 1;
                        }
                    }
                }
                if q_a_card > max_card {
                    max_card = q_a_card;
                }
            }
            max_card
        }

        let graph = make_dummy_graph();
        let sort = make_dummy_sort(&graph);
        assert_eq!(xa(&graph, &sort).unwrap(), naive(&graph, &sort));
    }

    #[test]
    fn test_xf() {
        /// X_f(s) := max_i #{ k | i<k \forall j s_j->s_k => j<i }
        pub fn naive(graph: &impl Graph, node_sort: &NodeSort) -> usize {
            let mut max_card = 0;
            for i in 0..node_sort.len() {
                let mut q_h_card = 0;
                'for_all: for rev_k in node_sort[i + 1..].iter() {
                    for (j, rev_j) in node_sort.iter().enumerate() {
                        if graph.parents(*rev_k).unwrap().contains(*rev_j) {
                            if j >= i {
                                continue 'for_all;
                            }
                        }
                    }
                    q_h_card += 1;
                }
                if q_h_card > max_card {
                    max_card = q_h_card;
                }
            }
            max_card
        }

        let graph = make_dummy_graph();
        let sort = make_dummy_sort(&graph);
        assert_eq!(xf(&graph, &sort).unwrap(), naive(&graph, &sort));
    }
}
