// SPDX-License-Identifier: GPL-2.0-or-later
//
// This software may be used and distributed according to the terms of the
// GNU General Public License version 2 or any later version.
//
// Copyright 2020  Pacien TRAN-GIRARD <pacien.trangirard@pacien.net>

use crate::ancestors::OrderedAncestorIterator;
use crate::graph::{
    Graph, GraphReadError, Revision, SizedGraph, StableOrderGraph,
};
use std::collections::HashMap;
use std::result::Result;
use std::result::Result::Ok;

///// BASE DEFS

/// Sorting/permutation of the Revision-s of a graph.
pub type NodeSort = Vec<Revision>;

/// Returns a {Revision -> sort-index} mapping from a given NodeSort.
/// This loosely corresponds to the inverse of a sorting permutation.
pub fn index_map(node_sort: &NodeSort) -> HashMap<Revision, usize> {
    let mut inverse = HashMap::with_capacity(node_sort.len());
    for (i, rev) in node_sort.iter().enumerate() {
        inverse.insert(*rev, i);
    }
    inverse
}

///// SORTERS

/// Stable sort.
pub fn stable_sort(
    graph: &(impl Graph + SizedGraph + StableOrderGraph),
    head: Revision,
) -> Result<NodeSort, GraphReadError> {
    let iter = OrderedAncestorIterator::new(graph, head)?;
    let mut list = iter.collect::<Result<NodeSort, GraphReadError>>()?;
    list.reverse();
    Ok(list)
}

#[cfg(test)]
mod tests {
    use crate::analytics::sorts::{index_map, stable_sort, NodeSort};
    use crate::graph::{
        Graph, GraphReadError, RankedGraph, Revision, SizedGraph,
        StableOrderGraph,
    };
    use crate::testing::graph_io::read_graph;
    use crate::testing::ordering::NodeIDComparator;
    use std::collections::HashSet;
    use std::io::{BufRead, Cursor};

    #[test]
    fn test_index_map() {
        let res = index_map(&vec![1, 2, 3, 4, 0]);
        assert_eq!(res[&0], 4);
        assert_eq!(res[&1], 0);
        assert_eq!(res[&2], 1);
        assert_eq!(res[&3], 2);
        assert_eq!(res[&4], 3);
    }

    fn make_dummy_graph(
    ) -> impl Graph + RankedGraph + SizedGraph + StableOrderGraph {
        // rnd2.graph
        let def = "
0000000000000000000000000000000000000100 0000000000000000000000000000000000000000 0000000000000000000000000000000000000000
0000000000000000000000000000000000000101 0000000000000000000000000000000000000100 0000000000000000000000000000000000000000
0000000000000000000000000000000000000102 0000000000000000000000000000000000000100 0000000000000000000000000000000000000101
0000000000000000000000000000000000000103 0000000000000000000000000000000000000102 0000000000000000000000000000000000000100
0000000000000000000000000000000000000104 0000000000000000000000000000000000000103 0000000000000000000000000000000000000000
0000000000000000000000000000000000000105 0000000000000000000000000000000000000101 0000000000000000000000000000000000000100
0000000000000000000000000000000000000106 0000000000000000000000000000000000000100 0000000000000000000000000000000000000000
0000000000000000000000000000000000000107 0000000000000000000000000000000000000101 0000000000000000000000000000000000000100
0000000000000000000000000000000000000108 0000000000000000000000000000000000000105 0000000000000000000000000000000000000103
0000000000000000000000000000000000000109 0000000000000000000000000000000000000107 0000000000000000000000000000000000000000
0000000000000000000000000000000000000110 0000000000000000000000000000000000000108 0000000000000000000000000000000000000109
0000000000000000000000000000000000000111 0000000000000000000000000000000000000108 0000000000000000000000000000000000000110
0000000000000000000000000000000000000112 0000000000000000000000000000000000000104 0000000000000000000000000000000000000000
0000000000000000000000000000000000000113 0000000000000000000000000000000000000102 0000000000000000000000000000000000000000
0000000000000000000000000000000000000114 0000000000000000000000000000000000000109 0000000000000000000000000000000000000106
0000000000000000000000000000000000000115 0000000000000000000000000000000000000102 0000000000000000000000000000000000000000
0000000000000000000000000000000000000116 0000000000000000000000000000000000000105 0000000000000000000000000000000000000000
0000000000000000000000000000000000000117 0000000000000000000000000000000000000105 0000000000000000000000000000000000000000
0000000000000000000000000000000000000118 0000000000000000000000000000000000000101 0000000000000000000000000000000000000101
0000000000000000000000000000000000000119 0000000000000000000000000000000000000101 0000000000000000000000000000000000000000
0000000000000000000000000000000000000120 0000000000000000000000000000000000000119 0000000000000000000000000000000000000105
0000000000000000000000000000000000000121 0000000000000000000000000000000000000109 0000000000000000000000000000000000000108
0000000000000000000000000000000000000122 0000000000000000000000000000000000000103 0000000000000000000000000000000000000000
0000000000000000000000000000000000000123 0000000000000000000000000000000000000106 0000000000000000000000000000000000000000
0000000000000000000000000000000000000124 0000000000000000000000000000000000000112 0000000000000000000000000000000000000115
0000000000000000000000000000000000000125 0000000000000000000000000000000000000118 0000000000000000000000000000000000000124
0000000000000000000000000000000000000126 0000000000000000000000000000000000000120 0000000000000000000000000000000000000000
0000000000000000000000000000000000000127 0000000000000000000000000000000000000100 0000000000000000000000000000000000000103
0000000000000000000000000000000000000128 0000000000000000000000000000000000000123 0000000000000000000000000000000000000120
0000000000000000000000000000000000000129 0000000000000000000000000000000000000103 0000000000000000000000000000000000000105
0000000000000000000000000000000000000130 0000000000000000000000000000000000000101 0000000000000000000000000000000000000000
0000000000000000000000000000000000000131 0000000000000000000000000000000000000100 0000000000000000000000000000000000000000
0000000000000000000000000000000000000132 0000000000000000000000000000000000000106 0000000000000000000000000000000000000100
0000000000000000000000000000000000000133 0000000000000000000000000000000000000107 0000000000000000000000000000000000000000
0000000000000000000000000000000000000134 0000000000000000000000000000000000000114 0000000000000000000000000000000000000000
0000000000000000000000000000000000000135 0000000000000000000000000000000000000107 0000000000000000000000000000000000000130
0000000000000000000000000000000000000136 0000000000000000000000000000000000000104 0000000000000000000000000000000000000000
0000000000000000000000000000000000000137 0000000000000000000000000000000000000121 0000000000000000000000000000000000000000
0000000000000000000000000000000000000138 0000000000000000000000000000000000000135 0000000000000000000000000000000000000000
0000000000000000000000000000000000000139 0000000000000000000000000000000000000131 0000000000000000000000000000000000000114
0000000000000000000000000000000000000140 0000000000000000000000000000000000000135 0000000000000000000000000000000000000135
0000000000000000000000000000000000000141 0000000000000000000000000000000000000140 0000000000000000000000000000000000000101
0000000000000000000000000000000000000142 0000000000000000000000000000000000000141 0000000000000000000000000000000000000102
0000000000000000000000000000000000000143 0000000000000000000000000000000000000142 0000000000000000000000000000000000000103
0000000000000000000000000000000000000144 0000000000000000000000000000000000000143 0000000000000000000000000000000000000104
0000000000000000000000000000000000000145 0000000000000000000000000000000000000144 0000000000000000000000000000000000000105
0000000000000000000000000000000000000146 0000000000000000000000000000000000000145 0000000000000000000000000000000000000106
0000000000000000000000000000000000000147 0000000000000000000000000000000000000146 0000000000000000000000000000000000000107
0000000000000000000000000000000000000148 0000000000000000000000000000000000000147 0000000000000000000000000000000000000108
0000000000000000000000000000000000000149 0000000000000000000000000000000000000148 0000000000000000000000000000000000000109
0000000000000000000000000000000000000150 0000000000000000000000000000000000000149 0000000000000000000000000000000000000110
0000000000000000000000000000000000000151 0000000000000000000000000000000000000150 0000000000000000000000000000000000000111
0000000000000000000000000000000000000152 0000000000000000000000000000000000000151 0000000000000000000000000000000000000112
0000000000000000000000000000000000000153 0000000000000000000000000000000000000152 0000000000000000000000000000000000000113
0000000000000000000000000000000000000154 0000000000000000000000000000000000000153 0000000000000000000000000000000000000114
0000000000000000000000000000000000000155 0000000000000000000000000000000000000154 0000000000000000000000000000000000000115
0000000000000000000000000000000000000156 0000000000000000000000000000000000000155 0000000000000000000000000000000000000116
0000000000000000000000000000000000000157 0000000000000000000000000000000000000156 0000000000000000000000000000000000000117
0000000000000000000000000000000000000158 0000000000000000000000000000000000000157 0000000000000000000000000000000000000118
0000000000000000000000000000000000000159 0000000000000000000000000000000000000158 0000000000000000000000000000000000000119
0000000000000000000000000000000000000160 0000000000000000000000000000000000000159 0000000000000000000000000000000000000120
0000000000000000000000000000000000000161 0000000000000000000000000000000000000160 0000000000000000000000000000000000000121
0000000000000000000000000000000000000162 0000000000000000000000000000000000000161 0000000000000000000000000000000000000122
0000000000000000000000000000000000000163 0000000000000000000000000000000000000162 0000000000000000000000000000000000000123
0000000000000000000000000000000000000164 0000000000000000000000000000000000000163 0000000000000000000000000000000000000124
0000000000000000000000000000000000000165 0000000000000000000000000000000000000164 0000000000000000000000000000000000000125
0000000000000000000000000000000000000166 0000000000000000000000000000000000000165 0000000000000000000000000000000000000126
0000000000000000000000000000000000000167 0000000000000000000000000000000000000166 0000000000000000000000000000000000000127
0000000000000000000000000000000000000168 0000000000000000000000000000000000000167 0000000000000000000000000000000000000128
0000000000000000000000000000000000000169 0000000000000000000000000000000000000168 0000000000000000000000000000000000000129
0000000000000000000000000000000000000170 0000000000000000000000000000000000000169 0000000000000000000000000000000000000130
0000000000000000000000000000000000000171 0000000000000000000000000000000000000170 0000000000000000000000000000000000000131
0000000000000000000000000000000000000172 0000000000000000000000000000000000000171 0000000000000000000000000000000000000132
0000000000000000000000000000000000000173 0000000000000000000000000000000000000172 0000000000000000000000000000000000000133
0000000000000000000000000000000000000174 0000000000000000000000000000000000000173 0000000000000000000000000000000000000134
0000000000000000000000000000000000000175 0000000000000000000000000000000000000174 0000000000000000000000000000000000000135
0000000000000000000000000000000000000176 0000000000000000000000000000000000000175 0000000000000000000000000000000000000136
0000000000000000000000000000000000000177 0000000000000000000000000000000000000176 0000000000000000000000000000000000000137
0000000000000000000000000000000000000178 0000000000000000000000000000000000000177 0000000000000000000000000000000000000138
0000000000000000000000000000000000000179 0000000000000000000000000000000000000178 0000000000000000000000000000000000000139
";
        let lines = Cursor::new(def).lines();
        read_graph::<_, NodeIDComparator>(lines).unwrap()
    }

    fn is_topological_order(
        graph: &impl Graph,
        node_sort: NodeSort,
    ) -> Result<bool, GraphReadError> {
        let mut discovered = HashSet::with_capacity(node_sort.len());

        for node in node_sort {
            if !discovered.insert(node) {
                return Ok(false); // found duplicate
            }

            for parent in graph.parents(node)? {
                if !discovered.contains(&parent) {
                    return Ok(false);
                }
            }
        }

        Ok(true)
    }

    #[test]
    fn test_stable_sort() {
        let graph = make_dummy_graph();
        let head = (graph.nb_nodes() - 1) as Revision;
        let node_sort = stable_sort(&graph, head).unwrap();
        assert_eq!(node_sort.len(), graph.rank(head).unwrap());
        assert!(is_topological_order(&graph, node_sort).unwrap());
    }
}
