// SPDX-License-Identifier: GPL-2.0-or-later
//
// This software may be used and distributed according to the terms of the
// GNU General Public License version 2 or any later version.
//
// Copyright 2020  Pacien TRAN-GIRARD <pacien.trangirard@pacien.net>

use crate::graph::{
    Graph, GraphReadError, Leap, LeapType, Parents, RankedGraph, Revision,
    SizedGraph, StableOrderGraph, NULL_REVISION,
};
use crate::lazy_ancestors::LazyAncestors;
use std::result::Result;
use std::result::Result::Ok;

#[derive(Debug)]
pub struct LeapCounts {
    pub hard_leap_count: usize,
    pub soft_leap_count: usize,
    pub exclusive_part_size: usize,
}

/// Returns the number of Soft and Hard leaps, as well as the size of the
/// exclusive part from the given list of Leaps.
pub fn count_leaps(leaps: &[Leap]) -> LeapCounts {
    let mut hard_leap_count = 0;
    let mut soft_leap_count = 0;
    let mut exclusive_part_size = 0;

    for leap in leaps {
        exclusive_part_size += leap.since_previous;
        match leap.leap_type {
            LeapType::Hard => hard_leap_count += 1,
            LeapType::Soft => soft_leap_count += 1,
            _ => {}
        }
    }

    LeapCounts {
        soft_leap_count,
        hard_leap_count,
        exclusive_part_size,
    }
}

#[derive(Debug)]
pub struct AggregatedCounts {
    pub nodes_by_rank: Vec<usize>,
    pub root_node_count: usize,
    pub linear_node_count: usize,
    pub merge_node_count: usize,
    pub hard_leap_count: usize,
    pub soft_leap_count: usize,
    pub exclusive_parts_sum: usize, // avg by dividing by merge_node_count
}

/// Aggregates counts on a given graph.
/// Head revisions are needed to visit the graph because we do access the list
/// of nodes directly (it's an internal of the graph implementation).
pub fn aggregate_counts(
    graph: &(impl Graph + RankedGraph + SizedGraph + StableOrderGraph),
    heads: &[Revision],
) -> Result<AggregatedCounts, GraphReadError> {
    let mut nodes_by_rank = vec![0; graph.nb_nodes() + 1];
    let mut root_node_count = 0;
    let mut linear_node_count = 0;
    let mut merge_node_count = 0;
    let mut hard_leap_count = 0;
    let mut soft_leap_count = 0;
    let mut exclusive_parts_sum = 0;

    // FIXME: debug the inclusion of the head nodes (InvalidKey)
    let lazy_ancestors =
        LazyAncestors::new(graph, heads.iter().cloned(), 0, true)?;

    for iter_res in lazy_ancestors.iter() {
        let rev = iter_res?;

        nodes_by_rank[graph.rank(rev)?] += 1;

        match graph.parents(rev)? {
            Parents([NULL_REVISION, NULL_REVISION]) => {
                root_node_count += 1;
            }
            Parents([_, NULL_REVISION]) | Parents([NULL_REVISION, _]) => {
                linear_node_count += 1;
            }
            Parents([_, _]) => {
                let leap_counts = count_leaps(graph.leaps(rev)?);
                hard_leap_count += leap_counts.hard_leap_count;
                soft_leap_count += leap_counts.soft_leap_count;
                exclusive_parts_sum += leap_counts.exclusive_part_size;
                merge_node_count += 1;
            }
        }
    }

    Ok(AggregatedCounts {
        nodes_by_rank,
        root_node_count,
        linear_node_count,
        merge_node_count,
        hard_leap_count,
        soft_leap_count,
        exclusive_parts_sum,
    })
}

#[cfg(test)]
mod tests {
    use crate::analytics::counts::{aggregate_counts, count_leaps};
    use crate::graph::{
        Leap, LeapType, MutableGraph, NodeID, NODE_ID_LEN, NULL_ID,
        NULL_REVISION,
    };
    use crate::testing::graph_in_mem::InMemoryGraph;
    use crate::testing::ordering::NodeIDComparator;
    use std::vec::Vec;

    //noinspection DuplicatedCode
    fn make_dummy_graph() -> InMemoryGraph<NodeIDComparator> {
        let node: Vec<NodeID> =
            (0..=6).map(|i| NodeID([i + 1; NODE_ID_LEN])).collect();

        let mut graph = InMemoryGraph::<NodeIDComparator>::new();
        graph.push(node[0], NULL_ID, NULL_ID).unwrap();
        graph.push(node[1], node[0], NULL_ID).unwrap();
        graph.push(node[2], node[0], NULL_ID).unwrap();
        graph.push(node[3], node[1], NULL_ID).unwrap();
        graph.push(node[4], node[3], node[2]).unwrap();
        graph.push(node[5], node[4], NULL_ID).unwrap();
        graph.push(node[6], node[3], NULL_ID).unwrap();

        graph
    }

    #[test]
    fn test_leap_count() {
        let leaps = [
            Leap {
                source: NULL_REVISION,
                target: NULL_REVISION,
                leap_type: LeapType::Soft,
                since_previous: 1,
                merges_since_previous: 0,
            },
            Leap {
                source: NULL_REVISION,
                target: NULL_REVISION,
                leap_type: LeapType::Soft,
                since_previous: 2,
                merges_since_previous: 0,
            },
            Leap {
                source: NULL_REVISION,
                target: NULL_REVISION,
                leap_type: LeapType::Hard,
                since_previous: 3,
                merges_since_previous: 0,
            },
            Leap {
                source: NULL_REVISION,
                target: NULL_REVISION,
                leap_type: LeapType::Last,
                since_previous: 4,
                merges_since_previous: 0,
            },
        ];
        let res = count_leaps(&leaps);
        assert_eq!(res.soft_leap_count, 2);
        assert_eq!(res.hard_leap_count, 1);
        assert_eq!(res.exclusive_part_size, 10);
    }

    #[test]
    fn test_aggregate_counts() {
        let graph = make_dummy_graph();
        let res = aggregate_counts(&graph, &[5, 6]).unwrap();
        assert_eq!(res.nodes_by_rank[1], 1);
        assert_eq!(res.nodes_by_rank[2], 2);
        assert_eq!(res.nodes_by_rank[3], 1);
        assert_eq!(res.root_node_count, 1);
        assert_eq!(res.linear_node_count, 5);
        assert_eq!(res.merge_node_count, 1);
        assert_eq!(res.hard_leap_count, 0);
        assert_eq!(res.soft_leap_count, 0);
        assert_eq!(res.exclusive_parts_sum, 2);
    }
}
