// SPDX-License-Identifier: GPL-2.0-or-later
//
// This software may be used and distributed according to the terms of the
// GNU General Public License version 2 or any later version.
//
// Copyright 2020  Pacien TRAN-GIRARD <pacien.trangirard@pacien.net>

use crate::graph::{Graph, GraphReadError, Parents, Revision, NULL_REVISION};
use std::cmp::{min, Eq};
use std::collections::HashMap;
use std::hash::Hash;
use std::marker::Copy;
use std::option::Option::Some;
use std::result::Result;
use std::result::Result::Ok;

/// Tells whether a `Revision` is in the given `Parents`.
fn parents_contain(parents: Parents, rev: Revision) -> bool {
    match parents {
        Parents([NULL_REVISION, NULL_REVISION]) => false,
        Parents([p1, NULL_REVISION]) | Parents([NULL_REVISION, p1]) => {
            p1 == rev
        }
        Parents([p1, p2]) => p1 == rev || p2 == rev,
    }
}

/// Returns the number of jumps, i.e. pair of consecutive elements for which
/// there's no arc left -> right in the given graph.
pub fn nb_jumps(
    graph: &impl Graph,
    revs: &[Revision],
) -> Result<usize, GraphReadError> {
    if revs.len() == 0 {
        return Ok(0);
    }

    let mut count = 0;
    let mut prev = revs[0];
    for rev in revs {
        if !parents_contain(graph.parents(prev)?, *rev) {
            count += 1;
        }
        prev = *rev;
    }
    Ok(count)
}

/// Returns the length of the longest prefix common to `a` and `b`.
pub fn common_prefix_len<T: Eq>(a: &[T], b: &[T]) -> usize {
    let max_len = min(a.len(), b.len());
    let mut len = 0;
    for i in 0..max_len {
        if a[i] == b[i] {
            len += 1;
        } else {
            break;
        }
    }
    len
}

/// Returns the length of the longest suffix common to `a` and `b`.
pub fn common_suffix_len<T: Eq>(a: &[T], b: &[T]) -> usize {
    let max_len = min(a.len(), b.len());
    let mut len = 0;
    for i in 1..=max_len {
        if a[a.len() - i] == b[b.len() - i] {
            len += 1;
        } else {
            break;
        }
    }
    len
}

/// Returns the number of breakpoints, i.e. the number of consecutive pairs in
/// `a` not present and consecutive in `b`.
pub fn nb_breakpoints<T: Copy + Hash + Eq>(a: &[T], b: &[T]) -> usize {
    let arcs: HashMap<_, _> =
        b.windows(2).map(|pair| (pair[0], pair[1])).collect();

    a.windows(2)
        .filter(|pair| arcs.get(&pair[0]) != Some(&pair[1]))
        .count()
}

#[cfg(test)]
mod tests {
    use crate::analytics::ancestors::{
        common_prefix_len, common_suffix_len, nb_breakpoints, nb_jumps,
    };
    use crate::ancestors::ordered_ancestors;
    use crate::graph::{
        Graph, LabelledGraph, MutableGraph, NodeID, StableOrderGraph,
        NODE_ID_LEN, NULL_ID,
    };
    use crate::testing::graph_in_mem::InMemoryGraph;
    use crate::testing::ordering::NodeIDComparator;
    use std::vec::Vec;

    //noinspection DuplicatedCode
    fn make_dummy_graph() -> impl Graph + LabelledGraph + StableOrderGraph {
        let node: Vec<NodeID> =
            (0..=6).map(|i| NodeID([i + 1; NODE_ID_LEN])).collect();

        let mut graph = InMemoryGraph::<NodeIDComparator>::new();
        graph.push(node[0], NULL_ID, NULL_ID).unwrap();
        graph.push(node[1], node[0], NULL_ID).unwrap();
        graph.push(node[2], node[0], NULL_ID).unwrap();
        graph.push(node[3], node[1], NULL_ID).unwrap();
        graph.push(node[4], node[3], node[2]).unwrap();
        graph.push(node[5], node[4], NULL_ID).unwrap();
        graph.push(node[6], node[3], NULL_ID).unwrap();

        graph
    }

    #[test]
    fn test_nb_jumps() {
        let graph = make_dummy_graph();
        let ancestors = ordered_ancestors(&graph, 5).unwrap();
        let res = nb_jumps(&graph, &ancestors).unwrap();
        assert_eq!(res, 2);
    }

    #[test]
    fn test_common_prefix_len() {
        let a = [1, 2, 3, 5, 4, 6, 7, 8];
        let b = [1, 2, 3, 4, 5, 6, 7];
        assert_eq!(common_prefix_len(&a, &b), 3);
    }

    #[test]
    fn test_common_suffix_len() {
        let a = [0, 1, 2, 3, 5, 4, 6, 7];
        let b = [1, 2, 3, 4, 5, 6, 7];
        assert_eq!(common_suffix_len(&a, &b), 2);
    }

    #[test]
    fn test_nb_breakpoints() {
        let a = [0, 1, 2, 3, 5, 4, 6, 7];
        let b = [1, 2, 3, 4, 5, 6, 7];
        assert_eq!(nb_breakpoints(&a, &b), 4);
    }
}
