require 'common/errors'
require 'common/runner'

# Perform the removal of users/domains using {RmPlugin}s.
#
class RmRunner
  include Runner

  # Run *plugin* to remove the users/domains in *targets*. The method
  # signature includes the unused *cfg* for consistency with the
  # runners that do need a {Configuration}.
  #
  # @param cfg [Configuration] unused.
  #
  # @param plugin [Class] plugin class that will perform the removal.
  #
  # @param targets [Array<User,Domain>] the users and domains to be
  #   removed.
  #
  def run(cfg, plugin, *targets)
    targets.each do |target|
      remove_target(plugin, target)
    end
  end


  protected;

  # Remove *target* using *plugin*. This operation is separate from
  # the <tt>run()</tt> method so that it can be accessed by the prune
  # runner.
  #
  # @param plugin [RmPlugin] the plugin that will remove the *target*.
  #
  # @param target [User,Domain] the user or domain to remove.
  #
  def remove_target(plugin, target)
    target_description = plugin.describe(target)

    begin
      plugin.remove(target)
      msg =  "Removed #{target.class.to_s().downcase()} "
      msg += add_description(target, target_description)
      msg += '.'

      report(plugin, msg)
    rescue NonexistentDomainError, NonexistentUserError => e
      report(plugin, "#{target.class.to_s()} #{e.to_s} not found.")
    end
  end
end
