require 'common/plugin.rb'

# Plugins for pruning users. By "pruning," we mean the removal of
# leftover non-PostfixAdmin users after the associated user has been
# removed from the Postfixadmin database.
#
module PrunePlugin

  # Absorb the subclass run() magic from the Plugin::Run module.
  extend Plugin::Run

  # The runner class associated with pruning plugins.
  #
  # @return [Class] the {PruneRunner} class.
  #
  def self.runner()
    return PruneRunner
  end


  # The "dummy" runner class associated with pruning plugins.
  #
  # @return [Class] the {PruneDummyRunner} class.
  #
  def self.dummy_runner
    return PruneDummyRunner
  end


  # Determine which domains are "left over" for this plugin. A domain
  # is considered "left over" if it has been removed from Postfixadmin
  # but not some other plugin.
  #
  # The leftovers are determined with respect to the list *db_domains*
  # of domains that Postfixadmin knows about.
  #
  # @param db_domains [Array<Domain>] a list of domains that are present
  #   in the Postfixadmin database.
  #
  # @return [Array<Domain>] a list of domains known to this plugin but
  #   not to Postfixadmin.
  #
  def get_leftover_domains(db_domains)
    # WARNING! Array difference doesn't work for some reason.
    return list_domains().select{ |d| !db_domains.include?(d) }
  end


  # Determine which users are "left over" for this plugin. A user
  # is considered "left over" if it has been removed from Postfixadmin
  # but not some other plugin.
  #
  # The leftovers are determined with respect to the list *db_users*
  # of users that Postfixadmin knows about.
  #
  # @param db_users [Array<User>] a list of users that are present
  #   in the Postfixadmin database.
  #
  # @return [Array<User>] a list of users known to this plugin but
  #   not to Postfixadmin.
  #
  def get_leftover_users(db_users)
    # WARNING! Array difference doesn't work for some reason.
    return list_users().select{ |u| !db_users.include?(u) }
  end
end
