# Methods inherited by the various runner classes ({PruneRunner},
# {MvRunner}, {RmRunner}).
#
module Runner


  # The main thing a runner does is <tt>run()</tt>. Each runner will
  # actually take a different number of arguments, so their
  # <tt>run()</tt> signatures will differ. This stub is only here to
  # let you know that it needs to be implemented.
  #
  # @param args [Array<Object>] whatever arguments the real implementation
  #   would take.
  #
  def run(*args)
    raise NotImplementedError
  end


  # When we're describing a user or domain, we often want to output
  # some additional description of it. But then, sometimes that
  # additional description is pointless, like when it's exactly the
  # same as the username that we're already outputting. This function
  # determines whether or not *desc* is pointless for *target*.
  #
  # @param target [User,Domain] the user or domain described by *desc*.
  #
  # @param desc [String] a string description of *target*.
  #
  # @return [Boolean] true if *desc* is a pointless description of
  #   *target* and false otherwise.
  #
  def pointless?(target, desc)
    return (desc.nil? or desc.empty? or desc == target.to_s())
  end


  # If *desc* is not a pointless description of *target*, return the
  # string representation of *target* followed by *desc* in
  # parentheses. If *desc* is pointless, we return only the string
  # representation of *target*
  #
  # @param target [User,Domain] the user or domain we want to describe
  #   as a string.
  #
  # @param desc [String] a string description of *target*.
  #
  # @return [String] the string representation of *target*, possibly
  #   followed by the non-pointless description *desc*.
  #
  def add_description(target, desc)
    if pointless?(target, desc)
      return target.to_s()
    else
      return target.to_s() + " (#{desc})"
    end
  end


  # Report a message from the given *plugin*. All this does is prefix
  # the message with the plugin name and then print it to stdout.
  #
  # @param plugin [Object] t plugin object that generated the message
  #   we're reporting.
  #
  # @param msg [String] the message to report.
  #
  def report(plugin, msg)
    print "#{plugin.class.to_s()} - "
    puts msg
  end

end
