require 'common/domain'
require 'common/filesystem'
require 'common/plugin'
require 'common/user'

# Code that all Dovecot plugins ({DovecotPrune}, {DovecotRm}, and
# {DovecotMv}) will share.
#
module DovecotPlugin

  # We implement the Plugin "interface."
  include Plugin


  # Initialize this Dovecot {Plugin} with values in *cfg*.
  #
  # @param cfg [Configuration] the configuration for this plugin.
  #
  def initialize(cfg)
    @domain_root = cfg.dovecot_mail_root
  end

  # Describe the given Dovecot domain by its filesystem path. The
  # domain need not exist to obtain its path.
  #
  # @param domain [Domain] the {Domain} object whose description we want.
  #
  # @return [String] a String giving the path under which this domain's
  #   mailboxes would reside on the filesystem.
  #
  def describe_domain(domain)
    return get_domain_path(domain)
  end


  # Describe the given Dovecot user by its filesystem mailbox
  # path. The user need not exist to obtain its mailbox path.
  #
  # @param user [User] the {User} object whose description we want.
  #
  # @return [String] a String giving the path where this user's
  #   mailbox would reside on the filesystem.
  #
  def describe_user(user)
    return get_user_path(user)
  end


  protected;

  # Return the filesystem path for the given {Domain} object.
  #
  # @param domain [Domain] the {Domain} whose path we want.
  #
  # @return [String] the filesystem path where this domain's mail
  #   would be located.
  #
  def get_domain_path(domain)
    return File.join(@domain_root, domain.to_s())
  end


  # Return the filesystem path of this {User}'s mailbox.
  #
  # @param user [User] the {User} whose mailbox path we want.
  #
  # @return [String] the filesystem path where this user's mail
  #   would be located.
  #
  def get_user_path(user)
    domain_path = get_domain_path(user.domain())
    return File.join(domain_path, user.localpart())
  end


  # Produce a list of domains that exist in the Dovecot mailstore.
  #
  # @return [Array<Domain>] an array of {Domain} objects that have
  #   corresponding directories within the Dovecot mailstore.
  #
  def list_domains()
    return Filesystem.get_subdirs(@domain_root).map{ |d| Domain.new(d) }
  end


  # Produce a list of users belonging to the given *domains* in the
  # Dovecot mailstore.
  #
  # @param domains [Array<Domain>] an array of {Domain} objects whose
  #   users we'd like to find.
  #
  # @return [Array<User>] an array of {User} objects that have
  #   corresponding directories within the Dovecot mailstore belonging
  #   to the specified *domains*.
  #
  def list_domains_users(domains)
    users = []

    domains.each do |domain|
      begin
        # Throws a NonexistentDomainError if the domain's path
        # doesn't exist on the filesystem. In this case, we want
        # to report zero users.
        domain_path = get_domain_path(domain)
        usernames = Filesystem.get_subdirs(domain_path)

        usernames.each do |username|
          users << User.new("#{username}@#{domain}")
        end
      rescue NonexistentDomainError
        # Party hard.
      end
    end

    return users
  end


  # Produce a list of all users in the Dovecot mailstore.
  #
  # @return [Array<User>] a list of users who have mailbox directories
  #   within the Dovecot mailstore.
  #
  def list_users()
    domains = list_domains()
    users = list_domains_users(domains)
    return users
  end

end
