require 'common/plugin'
require 'common/user'

# Code that all DAViCal plugins ({DavicalPrune}, {DavicalRm}, and
# {DavicalMv}) will share.
#
module DavicalPlugin

  # We implement the Plugin "interface."
  include Plugin

  # Initialize this DAViCal {Plugin} with values in *cfg*.
  #
  # @param cfg [Configuration] the configuration for this plugin.
  #
  def initialize(cfg)
    @db_hash = {
      :host     => cfg.davical_dbhost,
      :port     => cfg.davical_dbport,
      :options  => cfg.davical_dbopts,
      :dbname   => cfg.davical_dbname,
      :user     => cfg.davical_dbuser,
      :password => cfg.davical_dbpass }
  end


  # Describe the given DAViCal user who is assumed to exist.
  #
  # @param user [User] the {User} object whose description we want.
  #
  # @return [String] a String describing the given *user* in terms
  #   of his DAViCal "Principal ID".
  #
  def describe_user(user)
    principal_id = self.get_principal_id(user)
    return "Principal ID: #{principal_id}"
  end


  #
  # Produce a list of DAViCal users.
  #
  # This method remains public for use in testing.
  #
  # @return [Array<User>] an array of {User} objects, one for each
  #   user found in the DAViCal database.
  #
  def list_users()
    usernames = []

    connection = PG::Connection.new(@db_hash)

    # User #1 is the super-user, and not tied to an email address.
    sql_query = 'SELECT username FROM usr WHERE user_no > 1;'

    begin
      connection.sync_exec(sql_query) do |result|
        usernames = result.field_values('username')
      end
    ensure
      # Make sure the connection gets closed even if the query explodes.
      connection.close()
    end

    return usernames.map{ |u| User.new(u) }
  end


  protected;


  # Find the "Principal ID" of the given user.
  #
  # @param user [User] the user whose Principal ID we want.
  #
  # @return [Fixnum] an integer representing the user's Principal ID
  #   that we obtained from the DAViCal database.
  #
  def get_principal_id(user)
    principal_id = nil

    connection = PG::Connection.new(@db_hash)

    sql_query =  'SELECT principal.principal_id '
    sql_query += 'FROM (principal INNER JOIN usr '
    sql_query += '      ON principal.user_no = usr.user_no) '
    sql_query += 'WHERE usr.username = $1;'

    begin
      connection.sync_exec_params(sql_query, [user.to_s()]) do |result|
        if result.num_tuples > 0
          principal_id = result[0]['principal_id']
        end
      end
    ensure
      # Make sure the connection gets closed even if the query explodes.
      connection.close()
    end

    return principal_id
  end

end
