// Copyright (c) Facebook, Inc. and its affiliates.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package nvd

import (
	"bytes"
	"context"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"os"
	"strings"
	"testing"
)

var (
	cveGoldenMetaFile = strings.Join([]string{
		"lastModifiedDate:2018-03-16T23:05:50-04:00",
		"size:11",
		"zipSize:169",
		"gzSize:33",
		"sha256:B94D27B9934D3E08A52E52D7DA7DABFAC484EFE37A5380EE9088F7ACE2EFCDE9",
	}, "\r\n")

	cveGoldenDataFileGz = []byte{
		0x1f, 0x8b, 0x08, 0x08, 0x42, 0x4d, 0xac, 0x5a, 0x02, 0x03, 0x66, 0x00,
		0xcb, 0x48, 0xcd, 0xc9, 0xc9, 0x57, 0x28, 0xcf, 0x2f, 0xca, 0x49, 0x01,
		0x00, 0x85, 0x11, 0x4a, 0x0d, 0x0b, 0x00, 0x00, 0x00,
	}

	cveGoldenDataFileZip = []byte{
		0x50, 0x4b, 0x03, 0x04, 0x0a, 0x00, 0x02, 0x00, 0x00, 0x00, 0x6f, 0xb8,
		0x70, 0x4c, 0x85, 0x11, 0x4a, 0x0d, 0x0b, 0x00, 0x00, 0x00, 0x0b, 0x00,
		0x00, 0x00, 0x04, 0x00, 0x1c, 0x00, 0x74, 0x65, 0x73, 0x74, 0x55, 0x54,
		0x09, 0x00, 0x03, 0x42, 0x4d, 0xac, 0x5a, 0xeb, 0x80, 0xac, 0x5a, 0x75,
		0x78, 0x0b, 0x00, 0x01, 0x04, 0xad, 0x62, 0x2a, 0x64, 0x04, 0xba, 0x2d,
		0xd3, 0x6f, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0x20, 0x77, 0x6f, 0x72, 0x6c,
		0x64, 0x50, 0x4b, 0x01, 0x02, 0x1e, 0x03, 0x0a, 0x00, 0x02, 0x00, 0x00,
		0x00, 0x6f, 0xb8, 0x70, 0x4c, 0x85, 0x11, 0x4a, 0x0d, 0x0b, 0x00, 0x00,
		0x00, 0x0b, 0x00, 0x00, 0x00, 0x04, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x01, 0x00, 0x00, 0x00, 0xa4, 0x81, 0x00, 0x00, 0x00, 0x00, 0x74,
		0x65, 0x73, 0x74, 0x55, 0x54, 0x05, 0x00, 0x03, 0x42, 0x4d, 0xac, 0x5a,
		0x75, 0x78, 0x0b, 0x00, 0x01, 0x04, 0xad, 0x62, 0x2a, 0x64, 0x04, 0xba,
		0x2d, 0xd3, 0x6f, 0x50, 0x4b, 0x05, 0x06, 0x00, 0x00, 0x00, 0x00, 0x01,
		0x00, 0x01, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x00,
		0x00,
	}
)

func TestCVE(t *testing.T) {
	cases := make([]CVE, 0, len(SupportedCVE))
	for _, cve := range SupportedCVE {
		cases = append(cases, cve)
	}

	td, err := ioutil.TempDir("", "nvdsync-")
	if err != nil {
		t.Fatal(err)
	}
	defer os.RemoveAll(td)

	handler := &cveTestServer{}
	ts, src := httptestNewServer(handler)
	defer ts.Close()

	for _, cve := range cases {
		// run each test twice, one to create the mirror and another to compare
		label := []string{"CreateSync", "UseExistingSync"}
		for i := 0; i < 2; i++ {
			info := fmt.Sprintf("%s/%s", label[i], cve)
			t.Run(info, func(t *testing.T) {
				handler.compression = cve.compression()
				err = cve.Sync(context.Background(), src, td)
				if err != nil {
					t.Fatal(err)
				}
			})
		}
	}
}

type cveTestServer struct {
	compression string
}

func (ts cveTestServer) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	if strings.HasSuffix(r.URL.Path, ".meta") {
		io.Copy(w, bytes.NewBufferString(cveGoldenMetaFile))
		return
	}
	switch ts.compression {
	case "gz":
		io.Copy(w, bytes.NewBuffer(cveGoldenDataFileGz))
	case "zip":
		io.Copy(w, bytes.NewBuffer(cveGoldenDataFileZip))
	}
}
