/*
  Copyright (C) 2009 to 2013 and 2015 Chris Vine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include <stdlib.h>
#include <string.h>

#include <vector>
#include <fstream>
#include <ios>
#include <ostream>

#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h> // the key codes are here

#include <glib.h>

#include "mount_entries.h"
#include "mount_entries_icons.h"
#include "dialogs.h"
#include "utils/tree_path_handle.h"

#include <c++-gtk-utils/shared_handle.h>

#ifdef ENABLE_NLS
#include <libintl.h>
#endif


const int standard_size = 24;

namespace {
namespace ModelColumns {
  enum {device, label, cols_num};
}
} // anonymous namespace

void MountEntriesCB::mount_entries_button_clicked(GtkWidget* widget, void* data) {
  MountEntries* instance = static_cast<MountEntries*>(data);
  
  if (widget == instance->ok_button) {
    if (instance->changed) instance->save_list();
    instance->close();
  }
  else if (widget == instance->cancel_button) {
    instance->changed = false;
    instance->close();
  }
  else if (widget == instance->add_button) {
    instance->add_entry_prompt();
  }
  else if (widget == instance->modify_button) {
    instance->modify_entry_prompt();
  }
  else if (widget == instance->delete_button) {
    instance->delete_entry_prompt();
  }
  else if (widget == instance->up_button) {
    instance->move_up_impl();
  }
  else if (widget == instance->down_button) {
    instance->move_down_impl();
  }
  else {
    write_error("Callback error in MountEntriesCB::mount_entries_button_clicked()\n", false);
    instance->changed = false;
    instance->close();
  }
}

void MountEntriesCB::mount_entries_drag_n_drop(GtkTreeModel*, GtkTreePath*, void* data) {
  static_cast<MountEntries*>(data)->changed = true;
}

MountEntries::MountEntries(GtkWindow* parent): WinBase(gettext("Mount-gtk: mount table"),
						       prog_config.window_icon,
						       true, parent),
					       changed(false) {

  GtkTable* table = GTK_TABLE(gtk_table_new(2, 1, false));
  gtk_container_add(GTK_CONTAINER(get_win()), GTK_WIDGET(table));

  GtkTable* list_table = GTK_TABLE(gtk_table_new(6, 2, false));
  gtk_table_attach(table, GTK_WIDGET(list_table),
		   0, 1, 0, 1,
		   GtkAttachOptions(GTK_FILL | GTK_EXPAND), GtkAttachOptions(GTK_FILL | GTK_EXPAND),
		   standard_size/3, standard_size/3);

  GtkWidget* button_box = gtk_hbutton_box_new();
  gtk_table_attach(table, button_box,
		   0, 1, 1, 2,
		   GtkAttachOptions(GTK_FILL | GTK_EXPAND), GTK_SHRINK,
		   standard_size/3, standard_size/3);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(button_box), GTK_BUTTONBOX_END);
  gtk_box_set_spacing(GTK_BOX(button_box), standard_size/2);

  cancel_button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
  gtk_container_add(GTK_CONTAINER(button_box), cancel_button);
  ok_button = gtk_button_new_from_stock(GTK_STOCK_OK);
  gtk_container_add(GTK_CONTAINER(button_box), ok_button);

  add_button = gtk_button_new();
  gtk_table_attach(list_table, add_button,
		   0, 1, 0, 1,
		   GTK_SHRINK, GTK_SHRINK,
		   standard_size/2, 0);
  modify_button = gtk_button_new();
  gtk_table_attach(list_table, modify_button,
		   0, 1, 1, 2,
		   GTK_SHRINK, GTK_SHRINK,
		   standard_size/2, 0);
  delete_button = gtk_button_new();
  gtk_table_attach(list_table, delete_button,
		   0, 1, 2, 3,
		   GTK_SHRINK, GTK_SHRINK,
		   standard_size/2, 0);
  up_button = gtk_button_new();
  gtk_table_attach(list_table, up_button,
		   0, 1, 3, 4,
		   GTK_SHRINK, GTK_SHRINK,
		   standard_size/2, 0);
  down_button = gtk_button_new();
  gtk_table_attach(list_table, down_button,
		   0, 1, 4, 5,
		   GTK_SHRINK, GTK_SHRINK,
		   standard_size/2, 0);
  GtkWidget* dummy = gtk_label_new(0);
  gtk_table_attach(list_table, dummy,
		   0, 1, 5, 6,
		   GTK_SHRINK, GTK_EXPAND,
		   standard_size/2, 0);
  GtkScrolledWindow* scrolled_window = GTK_SCROLLED_WINDOW(gtk_scrolled_window_new(0, 0));
  gtk_table_attach(list_table, GTK_WIDGET(scrolled_window),
		   1, 2, 0, 6,
		   GtkAttachOptions(GTK_FILL | GTK_EXPAND), GtkAttachOptions(GTK_FILL | GTK_EXPAND),
		   0, 0);

  gtk_scrolled_window_set_shadow_type(scrolled_window, GTK_SHADOW_IN);
  gtk_scrolled_window_set_policy(scrolled_window, GTK_POLICY_ALWAYS,
				 GTK_POLICY_ALWAYS);

  list_store =
    GobjHandle<GtkTreeModel>{GTK_TREE_MODEL(gtk_list_store_new(ModelColumns::cols_num,
							       G_TYPE_STRING, G_TYPE_STRING))};
  // populate the model (containing the entries list)
  read_list();

  // create the tree view
  tree_view = GTK_TREE_VIEW(gtk_tree_view_new_with_model(list_store));
  gtk_container_add(GTK_CONTAINER(scrolled_window), GTK_WIDGET(tree_view));

  // provide renderers for tree view, pack into tree view columns
  // and connect to the tree model columns
  GtkCellRenderer* renderer = gtk_cell_renderer_text_new();
  GtkTreeViewColumn* column =
    gtk_tree_view_column_new_with_attributes(gettext("Device"), renderer,
					     "text", ModelColumns::device,
					     static_cast<void*>(0));
  gtk_tree_view_append_column(tree_view, column);
  
  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(gettext("Label"), renderer,
						    "text", ModelColumns::label,
						    static_cast<void*>(0));
  gtk_tree_view_append_column(tree_view, column);

  GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view);
  // single line selection
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

  // make drag and drop work
  gtk_tree_view_set_reorderable(tree_view, true);

  // have any drag and drop saved to the set-up file
  g_signal_connect(G_OBJECT(list_store.get()), "row_deleted",
		   G_CALLBACK(MountEntriesCB::mount_entries_drag_n_drop), this);

  // bring up the icon size registered in MainWindow::MainWindow()
  GtkIconSize mount_gtk_button_size = gtk_icon_size_from_name("MOUNT_GTK_BUTTON_SIZE");

  GtkWidget* image;
  image = gtk_image_new_from_stock(GTK_STOCK_ADD, mount_gtk_button_size);
  gtk_container_add(GTK_CONTAINER(add_button), image);

  { // provide a scope block for the GobjHandle
    GobjHandle<GdkPixbuf> pixbuf{gdk_pixbuf_new_from_xpm_data(modify_xpm)};
    image = gtk_image_new_from_pixbuf(pixbuf);
    gtk_container_add(GTK_CONTAINER(modify_button), image);
  }

  image = gtk_image_new_from_stock(GTK_STOCK_DELETE, mount_gtk_button_size);
  gtk_container_add(GTK_CONTAINER(delete_button), image);

  image = gtk_image_new_from_stock(GTK_STOCK_GO_UP, mount_gtk_button_size);
  gtk_container_add(GTK_CONTAINER(up_button), image);

  image = gtk_image_new_from_stock(GTK_STOCK_GO_DOWN, mount_gtk_button_size);
  gtk_container_add(GTK_CONTAINER(down_button), image);

  gtk_widget_set_tooltip_text(add_button, gettext("Add new mount item"));
  gtk_widget_set_tooltip_text(modify_button, gettext("Modify mount item"));
  gtk_widget_set_tooltip_text(delete_button, gettext("Delete mount item"));
  gtk_widget_set_tooltip_text(up_button, gettext("Move mount item up"));
  gtk_widget_set_tooltip_text(down_button, gettext("Move mount item down"));

  g_signal_connect(G_OBJECT(ok_button), "clicked",
		   G_CALLBACK(MountEntriesCB::mount_entries_button_clicked), this);
  g_signal_connect(G_OBJECT(cancel_button), "clicked",
		   G_CALLBACK(MountEntriesCB::mount_entries_button_clicked), this);
  g_signal_connect(G_OBJECT(add_button), "clicked",
		   G_CALLBACK(MountEntriesCB::mount_entries_button_clicked), this);
  g_signal_connect(G_OBJECT(modify_button), "clicked",
		   G_CALLBACK(MountEntriesCB::mount_entries_button_clicked), this);
  g_signal_connect(G_OBJECT(delete_button), "clicked",
		   G_CALLBACK(MountEntriesCB::mount_entries_button_clicked), this);
  g_signal_connect(G_OBJECT(up_button), "clicked",
		   G_CALLBACK(MountEntriesCB::mount_entries_button_clicked), this);
  g_signal_connect(G_OBJECT(down_button), "clicked",
		   G_CALLBACK(MountEntriesCB::mount_entries_button_clicked), this);
  
#if GTK_CHECK_VERSION(2,20,0)
  gtk_widget_set_can_default(ok_button, true);
  gtk_widget_set_can_default(cancel_button, true);

  gtk_widget_set_can_default(add_button, false);
  gtk_widget_set_can_default(delete_button, false);
  gtk_widget_set_can_focus(up_button, false);
  gtk_widget_set_can_focus(down_button, false);
#else
  GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);
  GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);

  GTK_WIDGET_UNSET_FLAGS(add_button, GTK_CAN_DEFAULT);
  GTK_WIDGET_UNSET_FLAGS(delete_button, GTK_CAN_DEFAULT);
  GTK_WIDGET_UNSET_FLAGS(up_button, GTK_CAN_FOCUS);
  GTK_WIDGET_UNSET_FLAGS(down_button, GTK_CAN_FOCUS);
#endif

  gtk_container_set_border_width(GTK_CONTAINER(table), standard_size/3);
  gtk_window_set_type_hint(get_win(), GDK_WINDOW_TYPE_HINT_DIALOG);
  gtk_window_set_position(get_win(), GTK_WIN_POS_CENTER_ON_PARENT);
  gtk_widget_grab_focus(cancel_button);
  gtk_window_set_default_size(get_win(), standard_size * 15, standard_size * 8);
  gtk_widget_show_all(GTK_WIDGET(get_win()));
}

int MountEntries::get_exec_val() const {
  return changed;
}

void MountEntries::on_delete_event() {
  changed = false;
  close();
}

void MountEntries::add_entry_prompt() {

  TableEntry dialog{get_win(), 0, 0};
  if (dialog.exec()) add_entry(dialog);
}

void MountEntries::add_entry(const TableEntry& dialog) {

  // this comes pre-stripped
  std::pair<std::string, std::string> result = dialog.get_result();

  if (!result.first.empty()) { // this test should be unnecessary, but ...

    GtkTreeIter iter;
    gtk_list_store_append(GTK_LIST_STORE(list_store.get()), &iter);
    gtk_list_store_set(GTK_LIST_STORE(list_store.get()), &iter,
		       ModelColumns::device, result.first.c_str(),
		       ModelColumns::label, result.second.c_str(),
		       -1);
    changed = true;
  }
}

void MountEntries::modify_entry_prompt() {
  GtkTreeIter iter;
  GtkTreeModel* model;
  GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view);
  gchar* device = 0;
  gchar* label = 0;

  if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gtk_tree_model_get(model, &iter,
		       ModelColumns::device, &device,
		       ModelColumns::label, &label,
		       -1);
    if (device && label) {
      try {
	TableEntry dialog{get_win(), device, label};
	if (dialog.exec()) modify_entry(dialog);
      }
      catch (...) {
	g_free(device);
	g_free(label);
	throw;
      }
    }
    g_free(device);
    g_free(label);
  }
  else beep();
}

void MountEntries::modify_entry(const TableEntry& dialog) {

  // this comes pre-stripped
  std::pair<std::string, std::string> result = dialog.get_result();

  if (!result.first.empty()) { // this test should be unnecessary, but ...

    GtkTreeIter iter;
    GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view);

    if (gtk_tree_selection_get_selected(selection, 0, &iter)) {
      gtk_list_store_set(GTK_LIST_STORE(list_store.get()), &iter,
			 ModelColumns::device, result.first.c_str(),
			 ModelColumns::label, result.second.c_str(),
			 -1);
      changed = true;
    }
  }
}

void MountEntries::delete_entry_prompt() {
  GtkTreeIter iter;
  GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view);

  if (gtk_tree_selection_get_selected(selection, 0, &iter)) {
    PromptDialog dialog{gettext("Delete selected mount entry?"),
			gettext("Gtk-mount: delete entry"),
	                get_win()};
    if (dialog.exec()) delete_entry();
  }
  else beep();
}

void MountEntries::delete_entry() {

  GtkTreeIter iter;
  GtkTreeModel* model;
  GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view);

  if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
    // delete the address by removing it from the list store
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
    changed = true;
  }
}

void MountEntries::read_list() {

  gchar** device_list = 0;
  gchar** label_list = 0;

  if ((device_list = g_key_file_get_string_list(prog_config.key_file,
						"Mounts",
						"Devices",
						0, 0))
      && (label_list = g_key_file_get_string_list(prog_config.key_file,
						  "Mounts",
						  "Labels",
						  0, 0))
      && *device_list) {

    gtk_list_store_clear(GTK_LIST_STORE(list_store.get()));

    gchar** device_tmp;
    gchar** label_tmp;
    for (device_tmp = device_list, label_tmp = label_list;
	 device_tmp && *device_tmp && label_tmp && *label_tmp;
	 ++device_tmp, ++label_tmp) {


      GtkTreeIter iter;
      gtk_list_store_append(GTK_LIST_STORE(list_store.get()), &iter);

      gtk_list_store_set(GTK_LIST_STORE(list_store.get()), &iter,
			 ModelColumns::device,
			 g_strstrip(*device_tmp),
			 -1);
      gtk_list_store_set(GTK_LIST_STORE(list_store.get()), &iter,
			 ModelColumns::label,
			 g_strstrip(*label_tmp),
			 -1);
    }
  }
  // nothing in this function can throw - this is fine
  // g_strfreev() does nothing if the argument is NULL
  g_strfreev(device_list);
  g_strfreev(label_list);
}

void MountEntries::save_list() {

  std::ofstream fileout(prog_config.rc_filename.c_str(), std::ios::out);

  if (fileout) {

    // use std::vector, as the standard requires it to have
    // contiguous memory so it can be passed as an array argument
    std::vector<gchar*> device_list;
    std::vector<gchar*> label_list;

    gchar* device;
    gchar* label;
    GtkTreeIter iter;
    bool not_at_end = gtk_tree_model_get_iter_first(list_store, &iter);
    try {
      while(not_at_end) {
	gtk_tree_model_get(list_store, &iter,
			   ModelColumns::device, &device,
			   ModelColumns::label, &label,
			   -1);
	device_list.push_back(device);
	label_list.push_back(label);
	not_at_end = gtk_tree_model_iter_next(list_store, &iter);
      }
      if (!device_list.empty()) {
	g_key_file_set_string_list(prog_config.key_file,
				   "Mounts",
				   "Devices",
				   &device_list[0],
				   device_list.size());
	g_key_file_set_string_list(prog_config.key_file,
				   "Mounts",
				   "Labels",
				   &label_list[0],
				   label_list.size());
      }
      else {
	g_key_file_remove_key(prog_config.key_file,
			      "Mounts",
			      "Devices",
			      0);
	g_key_file_remove_key(prog_config.key_file,
			      "Mounts",
			      "Labels",
			      0);
      }
      GcharScopedHandle data{g_key_file_to_data(prog_config.key_file, 0, 0)};
      fileout << (char*)data.get();
    }
    catch (...) {
      // this is just to make sure we clean up
      // the strings in the two vector objects
      // in the next two 'for' statements
      // we shouldn't actually ever be here so it is
      // a bit of a formalism
    }

    for (auto& elt: device_list) g_free(elt);
    for (auto& elt: label_list) g_free(elt);
  }
  else {
    std::string message{"Can't open file "};
    message += prog_config.rc_filename;
    message += " for saving mount entries\n";
    write_error(message.c_str(), true);
  }    
}

void MountEntries::move_up_impl() {

  GtkTreeIter selected_iter;
  GtkTreeModel* model;
  GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view);

  if (gtk_tree_selection_get_selected(selection, &model, &selected_iter)) {
    TreePathScopedHandle path{gtk_tree_model_get_path(model, &selected_iter)};
    if (gtk_tree_path_prev(path)) {

      GtkTreeIter prev_iter;
      
      if (!gtk_tree_model_get_iter(model, &prev_iter, path)) {
	write_error("Iterator error in MountEntries::move_up_impl()\n", false);
	beep();
      }
      else {
	gchar* selected_device = 0;
	gchar* selected_label = 0;
	gchar* prev_device = 0;
	gchar* prev_label = 0;

	// just do a swap of values
	gtk_tree_model_get(model, &selected_iter,
			   ModelColumns::device, &selected_device,
			   ModelColumns::label, &selected_label,
			   -1);

	gtk_tree_model_get(model, &prev_iter,
			   ModelColumns::device, &prev_device,
			   ModelColumns::label, &prev_label,
			   -1);

	gtk_list_store_set(GTK_LIST_STORE(model), &selected_iter,
			   ModelColumns::device, prev_device,
			   ModelColumns::label, prev_label,
			   -1);
      
	gtk_list_store_set(GTK_LIST_STORE(model), &prev_iter,
			   ModelColumns::device, selected_device,
			   ModelColumns::label, selected_label,
			   -1);

	// we have not placed the gchar* strings given by gtk_tree_model_get()
	// in a GcharScopedHandle object so we need to free them by hand
	g_free(selected_device);
	g_free(selected_label);
	g_free(prev_device);
	g_free(prev_label);

	gtk_tree_selection_select_iter(selection, &prev_iter);
	changed = true;
      }
    }
    else beep();
  }
  else beep();
}

void MountEntries::move_down_impl() {

  GtkTreeIter selected_iter;
  GtkTreeModel* model;
  GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view);

  if (gtk_tree_selection_get_selected(selection, &model, &selected_iter)) {
    GtkTreeIter next_iter = selected_iter;;
    if (gtk_tree_model_iter_next(model, &next_iter)) {
      
      gchar* selected_device = 0;
      gchar* selected_label = 0;
      gchar* next_device = 0;
      gchar* next_label = 0;

      // just do a swap of values
      gtk_tree_model_get(model, &selected_iter,
			 ModelColumns::device, &selected_device,
			 ModelColumns::label, &selected_label,
			 -1);

      gtk_tree_model_get(model, &next_iter,
			 ModelColumns::device, &next_device,
			 ModelColumns::label, &next_label,
			 -1);

      gtk_list_store_set(GTK_LIST_STORE(model), &selected_iter,
			 ModelColumns::device, next_device,
			 ModelColumns::label, next_label,
			 -1);
      
      gtk_list_store_set(GTK_LIST_STORE(model), &next_iter,
			 ModelColumns::device, selected_device,
			 ModelColumns::label, selected_label,
			 -1);

      // we have not placed the gchar* strings given by gtk_tree_model_get()
      // in a GcharScopedHandle object so we need to free them by hand
      g_free(selected_device);
      g_free(selected_label);
      g_free(next_device);
      g_free(next_label);

      gtk_tree_selection_select_iter(selection, &next_iter);
      changed = true;
    }
    else beep();
  }
  else beep();
}

void TableEntryCB::new_entry_selected(GtkWidget* widget, void* data) {
  TableEntry* instance = static_cast<TableEntry*>(data);

  if (widget == instance->ok_button) {
    if (instance->selected_impl()) {
      instance->accepted = true;
      instance->close();
    }
  }
  else if (widget == instance->cancel_button) instance->close();
  else {
    write_error("Callback error in TableEntryCB::new_entry_selected()\n", false);
    instance->close();
  }
}


gboolean TableEntryCB::new_entry_key_press_event(GtkWidget*, GdkEventKey* event, void* data) {
  TableEntry* instance = static_cast<TableEntry*>(data);
  int keycode = event->keyval;
#if GTK_CHECK_VERSION(2,20,0)
  bool cancel_focus = gtk_widget_has_focus(instance->cancel_button);
#else
  bool cancel_focus = GTK_WIDGET_HAS_FOCUS(instance->cancel_button);
#endif

#if GTK_CHECK_VERSION(2,99,0)
  if (keycode == GDK_KEY_Return && !cancel_focus) {
    if (instance->selected_impl()) {
      instance->accepted = true;
      instance->close();
    }
    return true;  // stop processing here
  }
#else
  if (keycode == GDK_Return && !cancel_focus) {
    if (instance->selected_impl()) {
      instance->accepted = true;
      instance->close();
    }
    return true;  // stop processing here
  }
#endif
  return false;   // continue processing key events
}

TableEntry::TableEntry(GtkWindow* parent,  const gchar* device, const gchar* label):
      WinBase((device && label) ? gettext("Mount-gtk: modify entry")
	                        : gettext("Mount-gtk: new entry"),
	      prog_config.window_icon,
	      true, parent),
      accepted(false) {

  GtkTable* table = GTK_TABLE(gtk_table_new(3, 2, false));
  gtk_container_add(GTK_CONTAINER(get_win()), GTK_WIDGET(table));


  GtkWidget* device_label = gtk_label_new(gettext("Device:"));
  gtk_table_attach(table, device_label,
		   0, 1, 0, 1,
		   GTK_FILL, GTK_SHRINK,
		   standard_size/2, standard_size/4);
  device_entry = gtk_entry_new();
  gtk_table_attach(table, device_entry,
		   1, 2, 0, 1,
		   GtkAttachOptions(GTK_FILL | GTK_EXPAND), GTK_SHRINK,
		   standard_size/2, standard_size/4);
  GtkWidget* label_label = gtk_label_new(gettext("Label:"));
  gtk_table_attach(table, label_label,
		   0, 1, 1, 2,
		   GTK_FILL, GTK_SHRINK,
		   standard_size/2, standard_size/4);
  label_entry = gtk_entry_new();
  gtk_table_attach(table, label_entry,
		   1, 2, 1, 2,
		   GtkAttachOptions(GTK_FILL | GTK_EXPAND), GTK_SHRINK,
		   standard_size/2, standard_size/4);
  GtkWidget* button_box = gtk_hbutton_box_new();
  gtk_table_attach(table, button_box,
		   0, 2, 2, 3,
		   GtkAttachOptions(GTK_FILL | GTK_EXPAND), GTK_SHRINK,
		   standard_size/2, standard_size/4);
  
  gtk_label_set_justify(GTK_LABEL(device_label), GTK_JUSTIFY_RIGHT);
  gtk_label_set_justify(GTK_LABEL(label_label), GTK_JUSTIFY_RIGHT);
  gtk_misc_set_alignment(GTK_MISC(device_label), 1, 0.5);
  gtk_misc_set_alignment(GTK_MISC(label_label), 1, 0.5);
  gtk_widget_set_size_request(device_entry, standard_size * 8, -1);
  gtk_widget_set_size_request(label_entry, standard_size * 8, -1);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(button_box), GTK_BUTTONBOX_END);
  gtk_box_set_spacing(GTK_BOX(button_box), standard_size/2);

  if (device && label) {
    gtk_entry_set_text(GTK_ENTRY(device_entry), device);
    gtk_entry_set_text(GTK_ENTRY(label_entry), label);
  }

  cancel_button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
  gtk_container_add(GTK_CONTAINER(button_box), cancel_button);
  ok_button = gtk_button_new_from_stock(GTK_STOCK_OK);
  gtk_container_add(GTK_CONTAINER(button_box), ok_button);

  g_signal_connect(G_OBJECT(ok_button), "clicked",
		   G_CALLBACK(TableEntryCB::new_entry_selected), this);
  g_signal_connect(G_OBJECT(cancel_button), "clicked",
		   G_CALLBACK(TableEntryCB::new_entry_selected), this);
  g_signal_connect(G_OBJECT(get_win()), "key_press_event",
		   G_CALLBACK(TableEntryCB::new_entry_key_press_event), this);

#if GTK_CHECK_VERSION(2,20,0)
  gtk_widget_set_can_default(ok_button, true);
  gtk_widget_set_can_default(cancel_button, true);
#else
  GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);
  GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);
#endif

  gtk_window_set_type_hint(get_win(), GDK_WINDOW_TYPE_HINT_DIALOG);

  gtk_container_set_border_width(GTK_CONTAINER(get_win()), standard_size/2);

  gtk_widget_grab_focus(device_entry);

  gtk_window_set_position(get_win(), GTK_WIN_POS_CENTER_ON_PARENT);
  gtk_window_set_resizable(get_win(), false);

  gtk_widget_show_all(GTK_WIDGET(get_win()));
}

int TableEntry::get_exec_val() const {
  return accepted;
}

bool TableEntry::selected_impl() {

  bool return_val = false;

  const gchar* device_text = gtk_entry_get_text(GTK_ENTRY(device_entry));
  const gchar* label_text = gtk_entry_get_text(GTK_ENTRY(label_entry));

  if (!*device_text || !*label_text) beep();
  
  else {
    result.first = (const char*)device_text;
    result.second = (const char*)label_text;
    strip(result.first);
    strip(result.second);
    return_val = true;
  }
  return return_val;
}
