#ifndef VISUAL_CONVEX_H
#define VISUAL_CONVEX_H

// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "displaylist.h"
#include "num_util.h"
#include "color.h"
#include "platform.h"

namespace visual {

class convex : public DisplayObject
{
 private:
	template <int N>
	struct vlist
	{
		vector v[N];
	};
	
	// A face is a triangle consisting of three vectors.
	struct face : vlist<3>
	{
		vector n; // Normal vector of the face.
		double d; // perpendicular distance from face to origin
		inline face() : d(0.0f) {}
		inline face( vector a, vector b, vector c)
		{
			v[0] = a; v[1]=b; v[2]=c;
			n = (v[1]-v[0]).cross(v[2]-v[0]).norm();
			d = n.dot(v[0]);
		}

		inline bool
		visibleFrom( const vector& p)
		{ return n.dot(p) > d; }
	};

	struct edge : vlist<2>
	{
		inline edge( vector a, vector b)
		{ v[0]=a; v[1]=b; }

		inline bool
		operator==( const edge& b) const
		{
			// There are two cases where a pair of edges are equal, the first is
			// occurs when the endpoints are both the same, while the other occurs 
			// when the edge have the same endpoints in opposite directions.  
			// Since the first case never happens when we construct the hull, we
			// only test for the second case here.
			return (v[0] == b.v[1] && v[1] == b.v[0]);
		}
	};

	struct jitter_table
	{
		enum { mask = 1023 };
		enum { count = mask+1 };
		double v[count];

		jitter_table()
		{
			for(int i=0; i<count; i++)
				v[i] = (static_cast<double>(rand()) / RAND_MAX - 0.5) * 2 * 1e-6;
		}
	}; 
	static jitter_table jitter;  // Use default construction for initialization.
	
	array pos;
	// The actual amount of storage allocated in the array.
	int preallocated_size;
	// The number of elements actually used in the array.
	int count;
	
	long checksum;
	// The hull is an array of constructed faces. to be drawn
	std::vector<face> hull;
	bool degenerate;

 public:
	convex();
	convex( const convex& other);
	virtual void glRender( rView& view);
	virtual double rayIntersect( const vector& cam, const vector& ray);
	virtual void refreshCache();
	void append( vector _pos); // Append a single position element.
	void append_t( const boost::python::object& _pos);
	// Append a block of position elements.
	// void append( const array& );


	rgb get_color() { return color; }
	boost::python::object get_pos();
	void set_color( rgb c);
	void set_pos( array);
	// Provide an overload for code that expects implicit conversion from 
	//   list->array.
	void set_pos_l( const boost::python::list& );

 private:
	void addpoint( int, vector);
	void recalc();
	// Change the length of the pos array that is actually used, potentially
	// enlarging the array.
	void set_length( int);
	long computeChecksum() const;
};

void convex_init_type();

} // !namespace visual

#endif // !VISUAL_CONVEX_H
