/* KInterbasDB Python Package - Implementation of Connection
 *
 * Version 3.2
 *
 * The following contributors hold Copyright (C) over their respective
 * portions of code (see license.txt for details):
 *
 * [Original Author (maintained through version 2.0-0.3.1):]
 *   1998-2001 [alex]  Alexander Kuznetsov   <alexan@users.sourceforge.net>
 * [Maintainers (after version 2.0-0.3.1):]
 *   2001-2002 [maz]   Marek Isalski         <kinterbasdb@maz.nu>
 *   2002-2006 [dsr]   David Rushby          <woodsplitter@rocketmail.com>
 * [Contributors:]
 *   2001      [eac]   Evgeny A. Cherkashin  <eugeneai@icc.ru>
 *   2001-2002 [janez] Janez Jere            <janez.jere@void.si>
 */


static int Connection_close(
    CConnection *con, boolean allowed_to_raise, boolean actually_detach
  );


/* Creates and returns new connection object. */
static CConnection *Connection_create(void) {
  CConnection *con = PyObject_New(CConnection, &ConnectionType);
  if (con == NULL) { goto fail; }

  con->state = CON_STATE_CLOSED;

  con->python_wrapper_obj = NULL; /* 2005.06.19 */

  con->dialect = SQL_DIALECT_DEFAULT;
  con->db_handle = NULL_DB_HANDLE;

  con->trans_handle = NULL_TRANS_HANDLE;
  con->group = NULL;

  #if (defined(ENABLE_FIELD_PRECISION_DETERMINATION) && !defined(INTERBASE_7_OR_LATER))
  con->desc_cache = NULL;
  #endif
  con->open_cursors = NULL;
  con->n_prepared_statements_executed_since_trans_start = 0;

  con->blob_charset_cache = NULL; /* 2005.06.21 */

  con->type_trans_in = NULL;
  con->type_trans_out = NULL;

  con->output_type_trans_return_type_dict = NULL;

  con->open_blobreaders = NULL;

  #ifdef ENABLE_CONNECTION_TIMEOUT
    con->timeout = NULL;
  #endif

  con->dsn = NULL;
  con->dsn_len = -1;
  con->dpb = NULL;
  con->dpb_len = -1;

  return con;

  fail:
    assert (PyErr_Occurred());
    return NULL;
} /* Connection_create */


static void Connection_delete(CConnection *con) {
  /* Detaches from the database and cleans up the resources used by an existing
   * connection (but does not free the CConnection structure itself;
   * that's the responsibility of pyob_Connection___del__). */
  #ifdef ENABLE_CONNECTION_TIMEOUT
    boolean might_need_to_close = TRUE;

    if (Connection_timeout_enabled(con)) {
      ConnectionOpState achieved_state;
      assert (CURRENT_THREAD_OWNS_CON_TP(con));
      achieved_state = ConnectionTimeoutParams_trans_while_already_locked(
          con->timeout, CONOP_IDLE, CONOP_PERMANENTLY_CLOSED
        );
      if (   achieved_state == CONOP_TIMED_OUT_TRANSPARENTLY
          || achieved_state == CONOP_TIMED_OUT_NONTRANSPARENTLY
         )
      {
        might_need_to_close = FALSE;
      }
    }
  #endif /* ENABLE_CONNECTION_TIMEOUT */

  /* If the CConnection is being garbage collected, that must mean there are no
   * Cursors or BlobReaders open on it, because if there were, they'd still be
   * holding references to the CConnection and it wouldn't have been targeted
   * for garbage collection. */
  assert (con->open_blobreaders == NULL);

  #ifdef ENABLE_CONNECTION_TIMEOUT
  if (might_need_to_close) {
  #endif /* ENABLE_CONNECTION_TIMEOUT */
    if (con->db_handle != NULL_DB_HANDLE || con->trans_handle != NULL_TRANS_HANDLE) {
      if (Connection_close(con, FALSE, TRUE) == 0) {
        assert(con->db_handle == NULL_DB_HANDLE);
        assert(CON_GET_TRANS_HANDLE(con) == NULL_TRANS_HANDLE); /* 2003.10.15a:OK */
      } else {
        /* Connection_close failed, but since we're garbage collecting and
         * can't reasonably raise an exception, ignore the failure to close,
         * and discard the db_handle anyway. */
        con->db_handle = NULL_DB_HANDLE;
        con->state = CON_STATE_CLOSED;
      }
    }
  #ifdef ENABLE_CONNECTION_TIMEOUT
  } else {
    assert (con->trans_handle == NULL_TRANS_HANDLE);
    assert (con->db_handle == NULL_DB_HANDLE);
  }
  #endif /* ENABLE_CONNECTION_TIMEOUT */
  assert(con->state == CON_STATE_CLOSED);

  #if (defined(ENABLE_FIELD_PRECISION_DETERMINATION) && !defined(INTERBASE_7_OR_LATER))
  assert(con->desc_cache == NULL);
  #endif
  assert(con->blob_charset_cache == NULL); /* 2005.06.21 */

  assert (con->open_cursors == NULL);

  Py_XDECREF(con->group);

  Py_XDECREF(con->type_trans_in);
  Py_XDECREF(con->type_trans_out);

  Py_XDECREF(con->output_type_trans_return_type_dict);

  /* There is no need to DECREF python_wrapper_obj, because we never owned a
   * reference to it. */
  con->python_wrapper_obj = NULL;

  if (con->dsn != NULL) {
    kimem_main_free(con->dsn);
  }
  if (con->dpb != NULL) {
    kimem_main_free(con->dpb);
  }
} /* Connection_delete */


static void pyob_Connection___del__(PyObject *con_) {
  CConnection *con = (CConnection *) con_;
  #ifdef ENABLE_CONNECTION_TIMEOUT
    const boolean timeout_enabled = Connection_timeout_enabled(con);

    if (timeout_enabled) {
      ACQUIRE_CON_TP_WITH_GIL_HELD(con);
    }
  #endif /* ENABLE_CONNECTION_TIMEOUT */

  Connection_delete(con);

  #ifdef ENABLE_CONNECTION_TIMEOUT
    if (timeout_enabled) {
      TP_UNLOCK(con->timeout);

      ConnectionTimeoutParams_destroy(&con->timeout);
      assert (con->timeout == NULL);
    }
  #endif

  PyObject_Del(con);
} /* pyob_Connection___del__ */


/* pyob_Connection_close is necessary to comply with the Python DB API spec.
 * Previously, the Python Connection class simply deleted its reference to the
 * C CConnection; there was no explicit call to a _kinterbasdb function to
 * close the underlying Firebird database handle.
 * The problem with that behavior is revealed by a closer reading of the DB
 * API spec's description of Connection.close:
 * """
 * Close the connection now (rather than whenever __del__ is called). The
 * connection will be unusable from this point forward; an Error (or subclass)
 * exception will be raised if any operation is attempted with the connection.
 * The same applies to all cursor objects trying to use the connection.
 * """
 * The Python Connection class previously just deleted its reference to the C
 * CConnection, but any cursors open on the CConnection retained
 * references to it, preventing it from being garbage collected.  Therefore,
 * the cursors never realized they were working with an "officially closed"
 * connection.
 *
 * pyob_Connection_close was added so that CursorObjects will immediately know
 * when they check their ->connection->state that the connection has been
 * explicitly closed.  The cursors will therefore refuse to work as soon as the
 * close method has been called on the Connection, "rather than whenever
 * __del__ is called" (to quote the DB API spec). */
static PyObject *pyob_Connection_close(PyObject *self, PyObject *args) {
  CConnection *con;

  if (!PyArg_ParseTuple(args, "O!", &ConnectionType, &con)) { return NULL; }

  if (Connection_close(con, TRUE, TRUE) != 0) { goto fail; }

  RETURN_PY_NONE;
  fail:
    assert (PyErr_Occurred());
    return NULL;
} /* pyob_Connection_close */


static int Connection_close_(
    CConnection *con, boolean allowed_to_raise, boolean actually_detach,
    boolean should_untrack_con_from_CTM,
    boolean should_clear_cursor_tracker
  )
{
  /* If the parameter allowed_to_raise is FALSE, this function will
   * refrain from raising a Python exception, and will indicate its status
   * solely via the return code. */
  int result = 0;

  /* The actually_detach parameter is present to support:
   *  1) Situations where the caller would like this function to close
   *     every member of the connection except its db_handle.  (The db_handle
   *     field is still nullified, however.)
   *  2) A workaround for a Firebird 1.0.x bug related to event handling. */

  /* A Connection that's part of a ConnectionGroup must not be close()d
   * directly; instead, the containing group must be close()d.  An exception
   * should have been be raised at the Python level to prevent the assertion
   * below from failing. */
  assert (con->group == NULL);

  #ifdef ENABLE_CONNECTION_TIMEOUT
  if (should_untrack_con_from_CTM) {
    int removal_status;

    /* If timeout is not enabled on this connection, the caller should not have
     * set should_untrack_con_from_CTM to true: */
    assert (Connection_timeout_enabled(con));
    /* The ConnectionTimeoutThread performs the untracking manually (for the
     * sake of efficiency): */
    assert (NOT_RUNNING_IN_CONNECTION_TIMEOUT_THREAD);

    /* The current thread should hold con->timeout->lock, and should already
     * have transitioned it to a ->state other than CONOP_IDLE, so that when we
     * release con->timeout->lock before calling CTM_remove (as we must do for
     * deadlock avoidance), the ConnectionTimeoutThread will not try to time
     * con out even as this thread is trying to do so manually. */
    assert (CURRENT_THREAD_OWNS_CON_TP(con));
    assert (con->timeout->state != CONOP_IDLE);

    LEAVE_GIL_WITHOUT_AFFECTING_DB
      TP_UNLOCK(con->timeout);
        removal_status = CTM_remove(con);
      TP_LOCK(con->timeout);
    ENTER_GIL_WITHOUT_AFFECTING_DB

    if (removal_status != 0) {
      if (allowed_to_raise) {
        PyErr_SetString(OperationalError, "Connection was unable to untrack"
            " itself."
          );
        return -1;
      } else {
        SUPPRESS_EXCEPTION;
        result = -1;
      }
    }
  }
  #endif /* ENABLE_CONNECTION_TIMEOUT */

  /* rollback_ANY_transaction will clear self->open_blobreaders; we don't need
   * to. */

  if (con->open_cursors != NULL) {
    if (should_clear_cursor_tracker) {
      if (CursorTracker_release(&con->open_cursors) == 0) {
        assert (con->open_cursors == NULL);
      } else {
        if (allowed_to_raise) {
          assert (PyErr_Occurred());
          return -1;
        } else {
          SUPPRESS_EXCEPTION;
          result = -1;
        }
      }
    } else {
      /* Although we aren't clearing con's Cursor tracker, we must close every
       * PreparedStatement open on all of the Cursors, because those
       * PreparedStatements will be unusable after the connection closes. */
      CursorTracker *node_cur = con->open_cursors;
      CursorTracker *node_prev = NULL;

      while (node_cur != NULL) {
        int status;
        Cursor *cur = node_cur->contained;
        assert (cur != NULL);

        status = Cursor_close_prepared_statements(cur, allowed_to_raise);
        if (status == 0) {
          /* If the prepared statements were successfully closed, then clear
           * various other cursor member fields by calling Cursor_clear.  That
           * function will change the cursor's state flag to
           * CURSOR_STATE_CLOSED, but since we're "cleansing" the cursor
           * without closing it, we restore the state flag to its original
           * value. */
          const CursorState orig_state = cur->state;
          status = Cursor_clear(cur, allowed_to_raise);
          if (status == 0) {
            assert (cur->state == CURSOR_STATE_CLOSED);
            cur->state = orig_state;
          }
        }

        if (status != 0) {
          if (allowed_to_raise) {
            return -1;
          } else {
            result = -1;
            SUPPRESS_EXCEPTION;
            /* Note that we ignore this exception and keep going, since, if
             * !allowed_to_raise, it's important that we try our best to close
             * all possible PreparedStatements. */
          }
        }
        node_prev = node_cur;
        node_cur = node_cur->next;
      }
    }
  }

  /* free_field_precision_cache will typically involve calls to
   * isc_dsql_free_statement, so it *must* be performed before the database
   * handle is invalidated. */
  #if (defined(ENABLE_FIELD_PRECISION_DETERMINATION) && !defined(INTERBASE_7_OR_LATER))
  free_field_precision_cache(con->desc_cache,
      (boolean) (con->state == CON_STATE_OPEN && con->db_handle != NULL_DB_HANDLE),
      con->status_vector
    );
  con->desc_cache = NULL;
  #endif

  if (con->blob_charset_cache != NULL) {
    Py_DECREF(con->blob_charset_cache);
    con->blob_charset_cache = NULL;
  }

  /* If there is an associated transaction and it is still active (has not been
   * commit()ed or rollback()ed), it is now rollback()ed.  The DB API Spec
   * says:  "Closing a connection without committing the changes first will
   * cause an implicit rollback to be performed." */
  if (con->trans_handle != NULL_TRANS_HANDLE) { /* No need for CON_GET_TRANS_HANDLE here. */
    /* It isn't strictly necessary to use rollback_ANY_transaction here instead
     * of a naked rollback_transaction, since we already know that the
     * connection is not a group member.  However, calling
     * rollback_ANY_transaction helps (slightly) to keep the rollback logic
     * centralized. */
    if (   rollback_ANY_transaction(con, allowed_to_raise)
        == OP_RESULT_OK
       )
    {
      /* Success; the transaction is dead. */
      con->trans_handle = NULL_TRANS_HANDLE;
    } else {
      /* The rollback attempt failed.  If we're *not* allowed to raise Python
       * exception, continue as though the rollback attempt had succeeded.
       * Otherwise, return -1 (rollback_transaction will already have raised a
       * Python exception) and leave the trans_handle unchanged. */
      assert (PyErr_Occurred());
      if (allowed_to_raise) {
        assert (PyErr_Occurred());
        return -1;
      } else {
        con->trans_handle = NULL_TRANS_HANDLE;
        SUPPRESS_EXCEPTION;
        result = -1;
      }
    }
  }

  if (con->db_handle != NULL_DB_HANDLE) {
    if (!actually_detach) {
      con->db_handle = NULL_DB_HANDLE;
    } else {
      #ifdef ENABLE_CONNECTION_TIMEOUT
        assert (CURRENT_THREAD_OWNS_CON_TP(con));
      #endif /* ENABLE_CONNECTION_TIMEOUT */

      LEAVE_GIL_WITHOUT_AFFECTING_DB
      ENTER_GDAL_WITHOUT_LEAVING_PYTHON
      ENTER_GCDL_WITHOUT_LEAVING_PYTHON

      isc_detach_database(con->status_vector, &con->db_handle);

      LEAVE_GCDL_WITHOUT_ENTERING_PYTHON
      LEAVE_GDAL_WITHOUT_ENTERING_PYTHON
      ENTER_GIL_WITHOUT_AFFECTING_DB

      if (!DB_API_ERROR(con->status_vector)) {
        assert (con->db_handle == NULL_DB_HANDLE);
      } else {
        if (allowed_to_raise) {
          raise_sql_exception(OperationalError, "Connection_close_: ",
              con->status_vector
            );
          return -1;
        } else {
          SUPPRESS_EXCEPTION;
          result = -1;
        }
      }
    }
  }

  con->state = CON_STATE_CLOSED;

  return result;
} /* Connection_close_ */

static int Connection_close(
    CConnection *con, boolean allowed_to_raise, boolean actually_detach
  )
{
  int res = 0;

  #ifdef ENABLE_CONNECTION_TIMEOUT
    /* This function can be called as an indirect result of a Cursor's
     * destruction.  pyob_Cursor___del__ acquires con's TP lock, so we need to
     * avoid trying to do that again (the attempt would deadlock). */
    const boolean already_owned_tp =
          Connection_timeout_enabled(con)
        ? CURRENT_THREAD_OWNS_TP(con->timeout)
        : TRUE
      ;

    if (!already_owned_tp) {
      ACQUIRE_CON_TP_WITH_GIL_HELD(con);
    }

  if (!Connection_timeout_enabled(con)) {
  #endif
    if (con->state != CON_STATE_CLOSED) {
      res = Connection_close_(con, allowed_to_raise, actually_detach, FALSE,
          TRUE
        );
    } else {
      goto fail_already_closed;
    }
  #ifdef ENABLE_CONNECTION_TIMEOUT
  } else {
    ConnectionTimeoutParams *tp = con->timeout;
    const ConnectionOpState orig_state = tp->state;

    switch (orig_state) {
      case CONOP_IDLE: /* The normal case. */
       {
        /* DEADLOCK AVOIDANCE:
         * This thread, which is not the ConnectionTimeoutThread, currently
         * holds the GIL and tp->lock, but it does not hold the CTM lock.
         *
         * If we continued to hold tp->lock while we called Connection_close_,
         * which in turn calls CTM_remove (which acquires the CTM lock), then a
         * deadlock could result if the ConnectionTimeoutThread is currently
         * performing a sweep, and is holding the CTM lock while trying to
         * acquire tp->lock to see whether con needs to be timed out.
         *
         * Therefore, we transition tp's state from CONOP_IDLE to CONOP_ACTIVE
         * for the duration of the Connection_close_ call.  Connection_close_
         * will release tp->lock before calling CTM_remove, but if the
         * ConnectionTimeoutThread is waiting to check tp->state, the fact that
         * the state is CONOP_ACTIVE will prevent the ConnectionTimeoutThread
         * from attempting to time con out even as we try to manually close
         * con. */
        ConnectionOpState achieved_state =
          ConnectionTimeoutParams_trans_while_already_locked(
              tp, CONOP_IDLE, CONOP_ACTIVE
            );
        /* This should never happen: */
        if (achieved_state != CONOP_ACTIVE) {
          raise_exception(InternalError, "Connection_close was unable to"
              " change the connection's state before calling"
              " Connection_close_."
            );
          goto fail;
        }

        res = Connection_close_(con, allowed_to_raise, actually_detach, TRUE,
            TRUE
          );

        if (res != 0) {
          goto fail;
        } else {
          achieved_state = ConnectionTimeoutParams_trans_while_already_locked(
              tp, CONOP_ACTIVE, CONOP_PERMANENTLY_CLOSED
            );
          if (achieved_state != CONOP_PERMANENTLY_CLOSED) {
            raise_exception(InternalError, "Connection_close was unable to"
                " change the connection's state after calling"
                " Connection_close_."
              );
            goto fail;
          }
        }

        break;
       }

      case CONOP_ACTIVE:
        raise_exception(InternalError, "[Connection_close] tp->state was"
            " CONOP_ACTIVE."
          );
        goto fail;

      case CONOP_PERMANENTLY_CLOSED:
        goto fail_already_closed;

      case CONOP_TIMED_OUT_TRANSPARENTLY:
      case CONOP_TIMED_OUT_NONTRANSPARENTLY:
        /* The ConnectionTimeoutThread already timed con out; we'll just make
         * the closure permanent. */
        if (ConnectionTimeoutParams_trans_while_already_locked(
              tp, orig_state, CONOP_PERMANENTLY_CLOSED
            ) == CONOP_PERMANENTLY_CLOSED
           )
        {
          assert (res == 0);
        } else {
          raise_exception(InternalError, "Connection_close was unable to"
              " change the connection's state from timed out to permanently"
              " closed."
            );
          goto fail;
        }
        break;
    }
  }
  #endif /* ENABLE_CONNECTION_TIMEOUT */

  goto exit;
  fail_already_closed:
    raise_exception(ProgrammingError, "The connection is already closed.");
    /* Fall through to fail: */
  fail:
    assert (PyErr_Occurred());
    res = -1;
    /* Fall through to exit: */
  exit:
    #ifdef ENABLE_CONNECTION_TIMEOUT
      if (!already_owned_tp) {
        RELEASE_CON_TP(con);
      }
    #endif
    return res;
} /* Connection_close */

#ifdef ENABLE_CONNECTION_TIMEOUT
static int Connection_close_from_CTT(volatile CConnection *con) {
  int status;
  BlobReader **blob_readers = NULL;
  Py_ssize_t n_blobreaders = -1;

  assert (RUNNING_IN_CONNECTION_TIMEOUT_THREAD);
  assert (CURRENT_THREAD_OWNS_CON_TP(con));

  if (con->open_blobreaders != NULL) {
    blob_readers = Connection_copy_BlobReader_pointers(con, &n_blobreaders);
    if (blob_readers == NULL) { goto fail; }
    assert (n_blobreaders != -1);
  }

  status = Connection_close_(DV_CCON(con), TRUE, TRUE, FALSE, FALSE);
  if (status != 0) { goto fail; }

  if (n_blobreaders > 0) {
    Connection_former_BlobReaders_flag_timeout_and_free(
        blob_readers, n_blobreaders
      );
    blob_readers = NULL;
  }

  goto clean;
  fail:
    status = -1;
    /* Fall through to clean: */
  clean:
    if (blob_readers != NULL) {
      assert (n_blobreaders != -1);
      Connection_former_BlobReader_pointers_free(blob_readers);
    }
    return status;
} /* Connection_close_from_CTT */

static BlobReader **Connection_copy_BlobReader_pointers(
    volatile CConnection *con, Py_ssize_t *count
  )
{
  /* The GIL must be held when this function is called. */
  /* Although the BlobReader objects will not be garbage collected as a
   * result of their connection's timeout, the connection's BlobReaderTracker
   * will be cleared.  We want future attempts to use any of these
   * BlobReaders to raise ConnectionTimedOut rather than ProgrammingError, so
   * we temporarily store pointers to all of the open BlobReaders that will
   * be remove from the tracker, then (in
   * Connection_former_BlobReaders_flag_timeout_and_free) go back and change
   * their state flag from BLOBREADER_STATE_CLOSED to
   * BLOBREADER_STATE_CONNECTION_TIMED_OUT to record the reason for their
   * closure. */
  BlobReader **blob_readers = NULL;
  Py_ssize_t n_blobreaders = 0;
  BlobReaderTracker *node_cur;
  BlobReaderTracker *node_prev;
  Py_ssize_t i;

  assert (con->open_blobreaders != NULL);

  node_cur = con->open_blobreaders;
  node_prev = NULL;
  while (node_cur != NULL) {
    ++n_blobreaders;

    node_prev = node_cur;
    node_cur = node_cur->next;
  }

  blob_readers = kimem_main_malloc(sizeof(BlobReader *) * n_blobreaders);
  if (blob_readers == NULL) { goto fail; }

  i = 0;
  node_cur = con->open_blobreaders;
  node_prev = NULL;
  while (node_cur != NULL) {
    blob_readers[i++] = node_cur->contained;

    node_prev = node_cur;
    node_cur = node_cur->next;
  }

  *count = n_blobreaders;
  return blob_readers;

  fail:
    assert (PyErr_Occurred());
    if (blob_readers != NULL) {
      Connection_former_BlobReader_pointers_free(blob_readers);
    }
    *count = -1;
    return NULL;
} /* Connection_copy_BlobReader_pointers */

static void Connection_former_BlobReaders_flag_timeout_and_free(
    BlobReader **blob_readers, Py_ssize_t count
  )
{
  Py_ssize_t i;
  for (i = 0; i < count; ++i) {
    assert (blob_readers[i]->state == BLOBREADER_STATE_CLOSED);
    blob_readers[i]->state = BLOBREADER_STATE_CONNECTION_TIMED_OUT;
  }

  Connection_former_BlobReader_pointers_free(blob_readers);
} /* Connection_former_BlobReaders_flag_timeout_and_free */
#endif /* ENABLE_CONNECTION_TIMEOUT */

static int Connection_require_open(CConnection *self,
    char *failure_message
  )
{
  /* If self is not an open connection, raises the supplied error message (or a
   * default if no error message was supplied).
   * Returns 0 if the connection was open; -1 if was closed. */
  if (self != NULL && self->state == CON_STATE_OPEN) {
    return 0;
  }

  if (failure_message == NULL) {
    failure_message = "Invalid connection state.  The connection must be"
      " open to perform this operation.";
  }

  raise_exception(ProgrammingError, failure_message);
  return -1;
} /* Connection_require_open */

static int Connection_attach_from_members(CConnection *con
    #ifdef ENABLE_CONNECTION_TIMEOUT
      , ConnectionTimeoutParams *tp
    #endif
  )
{
  assert (con != NULL);
  assert (con->state == CON_STATE_CLOSED);
  assert (con->db_handle == NULL_DB_HANDLE);
  assert (con->dialect > 0);
  assert (con->dsn != NULL);
  assert (con->dsn_len >= 0);
  assert (con->dpb == NULL ? TRUE : con->dpb_len >= 0);

  #ifdef ENABLE_CONNECTION_TIMEOUT
  if (tp != NULL) {
    assert (CURRENT_THREAD_OWNS_TP(tp));
  }
  #endif

  LEAVE_GIL_WITHOUT_AFFECTING_DB
  ENTER_GDAL_WITHOUT_LEAVING_PYTHON
  ENTER_GCDL_WITHOUT_LEAVING_PYTHON

  isc_attach_database(con->status_vector,
      con->dsn_len, con->dsn, &con->db_handle, con->dpb_len, con->dpb
    );

  LEAVE_GCDL_WITHOUT_ENTERING_PYTHON
  LEAVE_GDAL_WITHOUT_ENTERING_PYTHON
  ENTER_GIL_WITHOUT_AFFECTING_DB

  if (DB_API_ERROR(con->status_vector)) {
    raise_sql_exception(OperationalError, "isc_attach_database: ",
        con->status_vector
      );
    goto fail;
  }

  assert (con->db_handle != NULL_DB_HANDLE);
  con->state = CON_STATE_OPEN;

  #ifdef ENABLE_CONNECTION_TIMEOUT
    /* At this point, we know that the connection attempt was successful.  If a
     * timeout was requested for this connection, we now add the connection to
     * the timeout tracker.  It is the responsibility of CTM_add to associate
     * our previously initialized timeout parameter structure with con, iff con
     * is successfully added to the tracker. */
    assert (con->timeout == NULL);
    if (tp != NULL) {
      int status;

      tp->connected_at = time_millis();

      assert (tp->state != CONOP_IDLE);

      LEAVE_GIL_WITHOUT_AFFECTING_DB
      status = CTM_add(con, tp);
      ENTER_GIL_WITHOUT_AFFECTING_DB
      if (status != 0) {
        raise_exception(OperationalError, "[Connection_attach_from_members]"
            " Unsuccessful call to CTM_add."
          );
        goto fail;
      }
    }
    assert (con->timeout == tp);
  #endif /* ENABLE_CONNECTION_TIMEOUT */

  return 0;

  fail:
    assert (PyErr_Occurred());
    return -1;
} /* Connection_attach_from_members */


static int Connection_attach(CConnection *con,
    char *dsn, short dsn_len,
    char *dpb, short dpb_len
    #ifdef ENABLE_CONNECTION_TIMEOUT
      , ConnectionTimeoutParams *tp
    #endif
  )
{
  assert (con != NULL);
  assert (con->state == CON_STATE_CLOSED);
  assert (con->db_handle == NULL_DB_HANDLE);
  assert (con->dialect > 0);

  assert (dsn != NULL);
  assert (dsn_len >= 0);
  assert (dpb == NULL ? TRUE : dpb_len >= 0);

  assert (con->dsn == NULL);
  con->dsn = kimem_main_malloc(dsn_len);
  if (con->dsn == NULL) { goto fail; }
  memcpy(con->dsn, dsn, dsn_len);
  con->dsn_len = dsn_len;

  assert (con->dpb == NULL);
  con->dpb = kimem_main_malloc(dpb_len);
  if (con->dpb == NULL) { goto fail; }
  memcpy(con->dpb, dpb, dpb_len);
  con->dpb_len = dpb_len;

  {
    int status;

    #ifndef ENABLE_CONNECTION_TIMEOUT
      status = Connection_attach_from_members(con);
    #else
      if (tp != NULL) {
        ACQUIRE_TP_WITH_GIL_HELD(tp);
      }
      status = Connection_attach_from_members(con, tp);
      if (tp != NULL) {
        TP_UNLOCK(tp);
      }
    #endif
    if (status != 0) { goto fail; }
  }

  return 0;

  fail:
    assert (PyErr_Occurred());
    /* If we allocated any memory for con->dsn or con->dpb, con's destructor
     * will release it. */
    return -1;
} /* Connection_attach */


static PyObject *pyob_Connection_connect(PyObject *self, PyObject *args) {
  CConnection *con = NULL;
  PyObject *python_wrapper_obj = NULL;

  /* We will receive pointers to pre-rendered DSN and DPB buffers from the
   * Python level. */
  char *dsn = NULL;
  Py_ssize_t dsn_len = 0;
  char *dpb = NULL;
  Py_ssize_t dpb_len = 0;

  long dialect = 0;
  PyObject *py_timeout = NULL;
  #ifdef ENABLE_CONNECTION_TIMEOUT
    ConnectionTimeoutParams *timeout_params = NULL;
  #endif

  /* Parse and validate the arguments. */
  if (!PyArg_ParseTuple(args, "Os#s#lO",
         &python_wrapper_obj, &dsn, &dsn_len, &dpb, &dpb_len, &dialect,
         &py_timeout
       )
     )
  { goto fail; }

  if (dsn_len > MAX_DSN_SIZE) {
    PyObject *err_msg = PyString_FromFormat(
        "DSN too long (" Py_ssize_t_STRING_FORMAT " bytes out of %d allowed).",
        dsn_len, MAX_DSN_SIZE
      );
    if (err_msg != NULL) {
      raise_exception(ProgrammingError, PyString_AS_STRING(err_msg));
      Py_DECREF(err_msg);
    }
    goto fail;
  }

  if (dpb_len > MAX_DPB_SIZE) {
    PyObject *err_msg = PyString_FromFormat(
        "Database parameter buffer too large (" Py_ssize_t_STRING_FORMAT
        " bytes out of %d allowed).",
        dpb_len, MAX_DPB_SIZE
      );
    if (err_msg != NULL) {
      raise_exception(ProgrammingError, PyString_AS_STRING(err_msg));
      Py_DECREF(err_msg);
    }
    goto fail;
  }

  /* A negative value for the dialect is not acceptable because the IB/FB API
   * requires an unsigned short. */
  if (dialect < 0 || dialect > USHRT_MAX) {
    PyObject *err_msg = PyString_FromFormat(
        "Connection dialect must be between 0 and %d (%ld is out of range).",
        USHRT_MAX, dialect
      );
    if (err_msg != NULL) {
      raise_exception(ProgrammingError, PyString_AS_STRING(err_msg));
      Py_DECREF(err_msg);
    }
    goto fail;
  }

  /* Validate the connection timeout settings, if any. */
  #ifndef ENABLE_CONNECTION_TIMEOUT
    if (py_timeout != Py_None) {
      raise_exception(NotSupportedError, "The connection timeout feature is"
          " disabled in this build."
        );
      goto fail;
    }
  #else
    if (py_timeout == Py_None) {
      assert (timeout_params == NULL);
    } else {
      timeout_params = c_timeout_from_py(py_timeout);
      if (timeout_params == NULL) {
        assert (PyErr_Occurred());
        goto fail;
      }
    }
  #endif /* ENABLE_CONNECTION_TIMEOUT */

  /* Create a CConnection struct (this is essentially a memory allocation
   * operation, not the actual point of connection to the database server): */
  con = Connection_create();
  if (con == NULL) { goto fail; }

  /* con->dialect is set to a default value in the Connection_create function,
   * so we only need to change it if we received a dialect argument to this
   * function. */
  if (dialect > 0) {
    con->dialect = (unsigned short) dialect;
  }
  assert (con->dialect > 0);

  /* Yes, it is correct not to INCREF this PyObject * before storing it--see
   * comment about the CConnection.python_wrapper_obj field in
   * _kinterbasdb.h */
  con->python_wrapper_obj = python_wrapper_obj;
  assert (con->python_wrapper_obj != NULL);

  /* IT'S IMPORTANT THAT THE CONNECTION BE ADDED TO THE TIMEOUT TRACKER ONLY AS
   * THE LAST STEP HERE, WHEN THE CONNECTION'S MEMBERS HAVE ALREADY BEEN
   * INITIALIZED! */
  /* The casts from Py_ssize_t to short are no problem because the Py_ssize_t
   * values have already been verified as small enough to fit into a short. */
  if (Connection_attach(con, dsn, (short) dsn_len, dpb, (short) dpb_len
          #ifdef ENABLE_CONNECTION_TIMEOUT
            , timeout_params
          #endif
        ) != 0
      )
  {
    goto fail;
  }
  #ifdef ENABLE_CONNECTION_TIMEOUT
  else {
    timeout_params = NULL;
  }
  #endif

  return (PyObject *) con;

  fail:
    assert (PyErr_Occurred());
    #ifdef ENABLE_CONNECTION_TIMEOUT
      if (timeout_params != NULL) {
        if (ConnectionTimeoutParams_destroy(&timeout_params) == 0) {
          assert (timeout_params == NULL);
        }
      }
    #endif
    Py_XDECREF((PyObject *) con);
    return NULL;
} /* pyob_Connection_connect */

#ifdef INTERBASE_6_OR_LATER

static PyObject *pyob_Connection_closed_get(PyObject *self, PyObject *args) {
  CConnection *con;
  boolean closed;

  if (!PyArg_ParseTuple(args, "O!", &ConnectionType, &con)) { return NULL; }

  #ifdef ENABLE_CONNECTION_TIMEOUT
  {
    const boolean needed_to_acquire_tp = !CURRENT_THREAD_OWNS_CON_TP(con);
    if (needed_to_acquire_tp) {
      ACQUIRE_CON_TP_WITH_GIL_HELD(con);
    }
  #endif /* ENABLE_CONNECTION_TIMEOUT */
    closed = (boolean) (con->state == CON_STATE_CLOSED);
  #ifdef ENABLE_CONNECTION_TIMEOUT
    if (needed_to_acquire_tp) {
      RELEASE_CON_TP(con);
    }
  }
  #endif /* ENABLE_CONNECTION_TIMEOUT */

  return PyBool_FromLong(closed);
} /* Connection_dialect_get */

static PyObject *pyob_Connection_dialect_get(PyObject *self, PyObject *args) {
  CConnection *con;

  if (!PyArg_ParseTuple(args, "O!", &ConnectionType, &con)) { return NULL; }

  CONN_REQUIRE_OPEN(con);

  return PyInt_FromLong(con->dialect);
} /* Connection_dialect_get */

static PyObject *pyob_Connection_dialect_set(PyObject *self, PyObject *args) {
  CConnection *con;
  short dialect;

  if (!PyArg_ParseTuple(args, "O!h", &ConnectionType, &con, &dialect)) {
    return NULL;
  }

  CONN_REQUIRE_OPEN(con);

  /* A negative value for the dialect is not acceptable because the IB/FB API
   * requires an UNSIGNED SHORT. */
  if (dialect < 0) {
    raise_exception(ProgrammingError, "connection dialect must be >= 0");
    return NULL;
  }

  con->dialect = (unsigned short) dialect;

  RETURN_PY_NONE;
} /* Connection_dialect_set */

#endif /* INTERBASE_6_OR_LATER */

static PyObject *pyob_Connection_group_get(PyObject *self, PyObject *args) {
  CConnection *con;
  PyObject *group;

  if (!PyArg_ParseTuple(args, "O!", &ConnectionType, &con)) { return NULL; }

  group = con->group;
  if (group == NULL) {
    RETURN_PY_NONE;
  } else {
    Py_INCREF(group);
    return group;
  }
} /* pyob_Connection_group_get */

static PyObject *pyob_Connection_group_set(PyObject *self, PyObject *args) {
  CConnection *con;
  PyObject *group;

  if (!PyArg_ParseTuple(args, "O!O", &ConnectionType, &con, &group)) {
    return NULL;
  }

  if (group == Py_None) {
    con->group = NULL;
  } else {
    /* Due to the specifics of the ConnectionGroup class, there should be a
     * con._set_group(None) call between any con._set_group(group) calls. */
    if (con->group != NULL) {
      raise_exception(InternalError, "Attempt to set connection group when"
          " previous setting had not been cleared."
        );
      return NULL;
    }
    /* The ConnectionGroup code always calls con._set_group(None) when con is
     * removed from its group, including when that removal is invoked
     * implicitly by the ConnectionGroup.__del__.  Therefore, the connection
     * can avoid creating a cycle by never owning a reference to its group, yet
     * knowing that ->group will never refer to a dead object. */
    con->group = group;
  }

  RETURN_PY_NONE;
} /* pyob_Connection_group_set */

static PyObject *pyob_Connection_x_info(boolean for_isc_database_info,
    PyObject *self, PyObject *args
  )
{
  /* If for_isc_database_info is false, then we're issuing a request to
   * isc_transaction_info. */
  CConnection *con;
  const char *err_preamble =
    for_isc_database_info ? "isc_database_info: " : "isc_transaction_info: ";

  char req_buf[] = {isc_info_end, isc_info_end};
  char res_type; /* 'i'-short integer, 's'-string */

  char *res_buf = NULL;
  size_t res_buf_size = 0;
  size_t res_len = 0;
  size_t i = 0;

  PyObject *py_res = NULL;

  if (!PyArg_ParseTuple( args, "O!bc",
         &ConnectionType, &con, &req_buf[0], &res_type
       )
     )
  { return NULL; }

  CON_ACTIVATE(con, return NULL);

  res_buf_size = 64;
  for (;;) {
    assert (res_buf_size <= SHRT_MAX);
    {
      char *res_buf_bigger = kimem_main_realloc(res_buf, res_buf_size);
      if (res_buf_bigger == NULL) {
        /* res_buf might still be non-NULL, but the cleanup block will take
         * care of that. */
        goto fail;
      }
      res_buf = res_buf_bigger;
    }

    memset(res_buf, '\0', res_buf_size);

    ENTER_GDAL
    if (for_isc_database_info) {
      isc_database_info(con->status_vector, &con->db_handle,
          sizeof(req_buf), req_buf,
          /* The cast below is safe because we've validated res_buf_size: */
          (short) res_buf_size, res_buf
        );
    } else {
      isc_transaction_info(con->status_vector, &con->trans_handle,
          sizeof(req_buf), req_buf,
          /* The cast below is safe because we've validated res_buf_size: */
          (short) res_buf_size, res_buf
        );
    }
    LEAVE_GDAL
    if (DB_API_ERROR(con->status_vector)) {
      raise_sql_exception(OperationalError, err_preamble, con->status_vector);
      goto fail;
    }

    /* Some info codes, such as isc_info_user_names, don't represent the length
     * of their results in a standard manner, so we scan backward from the end
     * of the result buffer to find where the results end. */
    for (i = res_buf_size - 1; i >= 0; --i) {
      if (res_buf[i] != '\0') {
        break;
      }
    }

    if (res_buf[i] != isc_info_truncated) {
      /* Success; leave the for(ever) loop. */
      break;
    } else {
      /* The previous result buffer wasn't large enough; enlarge it and try
       * again.  The size mustn't exceed SHRT_MAX because the size parameters
       * to isc_database_info are of type short. */
      if (res_buf_size == SHRT_MAX) {
        raise_exception(OperationalError, "Underlying info function demanded"
            " result buffer larger than SHRT_MAX."
          );
        goto fail;
      }

      res_buf_size *= 2;
      if (res_buf_size > SHRT_MAX) {
        res_buf_size = SHRT_MAX;
      }
    }
  }

  if (res_buf[i] != isc_info_end) {
    raise_exception(InternalError, "Exited request loop successfully, but"
        " res_buf[i] != isc_info_end."
      );
    goto fail;
  }
  if (res_buf[0] != req_buf[0]) {
    raise_exception(InternalError, "Result code does not match request code.");
    goto fail;
  }

  res_len = i;

  switch (res_type) {
    case 'i':
    case 'I':
      {
        LONG_LONG int_value;
        /* The cluster length will differ depending on whether the integer
         * result is returned as a byte, a short, or an int. */
        short cluster_len;

        ENTER_GDAL
        cluster_len = (short) isc_vax_integer(res_buf + 1, sizeof(short));
        int_value = isc_portable_integer(
            #ifndef INTERBASE_7_OR_LATER
              (unsigned char *)
            #endif
            (res_buf + 1 + sizeof(short)),
            cluster_len
          );
        LEAVE_GDAL

        py_res = PythonIntOrLongFrom64BitValue(int_value);
      }
      break;

    case 's':
    case 'S':
      /* Some info codes, such as isc_info_user_names, don't represent the
       * length of their results in a manner compatible with the previous
       * parsing logic, so we need to return the entire buffer to the Python
       * level. */
      py_res = PyString_FromStringAndSize(res_buf,
          SIZE_T_TO_PYTHON_SIZE(res_len)
        );
      break;

    default:
      raise_exception(InterfaceError, "Unknown result type requested (must be"
          " 'i' or 's')."
        );
      goto fail;
  }

  assert (py_res != NULL);
  assert (!PyErr_Occurred());
  goto clean;
  fail:
    assert (PyErr_Occurred());
    if (py_res != NULL) {
      Py_DECREF(py_res);
      py_res = NULL;
    }
    /* Fall through to clean: */
  clean:
    if (res_buf != NULL) {
      kimem_main_free(res_buf);
    }

    CON_PASSIVATE(con);
    CON_MUST_NOT_BE_ACTIVE(con);

    return py_res;
} /* pyob_Connection_x_info */

static PyObject *pyob_Connection_database_info(PyObject *self, PyObject *args)
{
  return pyob_Connection_x_info(TRUE, self, args);
}

static PyObject *pyob_Connection_transaction_info(
    PyObject *self, PyObject *args
  )
{
  return pyob_Connection_x_info(FALSE, self, args);
}

static PyObject *pyob_Connection_python_wrapper_obj_set(
    PyObject *self, PyObject *args
  )
{
  CConnection *con;
  PyObject *python_wrapper_obj;

  if (!PyArg_ParseTuple(args, "O!O",
         &ConnectionType, &con, &python_wrapper_obj
       )
     )
  { return NULL; }

  /* This function is called only by internal code, and should only be called
   * once. */
  if (con->python_wrapper_obj != NULL) {
    raise_exception(InternalError, "Attempt to set Python wrapper object"
        " reference when it had already been set."
      );
    return NULL;
  }

  /* Yes, it is correct not to INCREF this PyObject * before storing it--see
   * comment about the CConnection.python_wrapper_obj field in
   * _kinterbasdb.h */
  con->python_wrapper_obj = python_wrapper_obj;

  RETURN_PY_NONE;
} /* pyob_Connection_python_wrapper_obj_set */

static void Connection_clear_transaction_stats(CConnection *con) {
  con->n_prepared_statements_executed_since_trans_start = 0;
} /* Connection_clear_transaction_stats */

static PyObject *pyob_Connection_clear_transaction_stats(
    PyObject *self, PyObject *args
  )
{
  CConnection *con;
  if (!PyArg_ParseTuple(args, "O!", &ConnectionType, &con)) { return NULL; }

  Connection_clear_transaction_stats(con);

  RETURN_PY_NONE;
} /* pyob_Connection_clear_transaction_stats */

static PyObject *pyob_Connection_timeout_enabled(
    PyObject *self, PyObject *args
  )
{
  CConnection *con;
  if (!PyArg_ParseTuple(args, "O!", &ConnectionType, &con)) { return NULL; }

  #ifdef ENABLE_CONNECTION_TIMEOUT
    return PyBool_FromLong(Connection_timeout_enabled(con));
  #else
    Py_INCREF(Py_False);
    return Py_False;
  #endif
} /* pyob_Connection_timeout_enabled */
