/* LocalTransferHandler.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.dnd;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import javax.swing.JComponent;
import javax.swing.TransferHandler;

/**
 * Transfer handler for drag and drop of objects within
 * the same virtual machine. Differs from the standard {@link TransferHandler}
 * in the following ways:
 * <ul>
 * <li>Delegates drag functionality to a separate object of type {@link DragHandler}.
 * <li>Delegates drop functionality to a list of registered drop handlers.
 * <li>Translates 'flavors' to 'classes'.
 * </ul>
 * Typical usage is as follows: create an object of this class,
 * install a drag handler,
 * add one or more drop handlers, enable the parent component
 * for dragging and register the object as a transfer handler with
 * the parent:
 * <pre>
 *    LocalTransferHandler handler = new LocalTransferHandler ();
 *    handler.setDragHandler (someDragHandler);
 *    handler.addDropHandler (handlerForSomeClass);
 *    handler.addDropHandler (handlerForSomeOtherClass);
 *    ...
 *    parent.setDragEnabled (true);
 *    parent.setTransferHandler (handler);
 * </pre>
 *
 * Clients
 * may choose to register handlers as part of the constructor of an extension of this class.<p>
 * <h2>Sources and targets</h2>
 * Methods {@link #isTransferAllowed}, {@link #exportDone} provide both a {@code source}
 * and {@code target} parameter which can be used by client implementations. Because
 * Swing support for this kind of information is poor (at least in versions of Java prior 
 * to 6.0), source and target references
 * are stored as static variables of this class. As a consequence, they are only
 * reliable when the transfer handler for both source and target are of type
 * {@link LocalTransferHandler}.
 * <b>Note</b> Like {@link TransferHandler}, objects of this class can be shared
 * by different components.
 */
public class LocalTransferHandler extends TransferHandler {
    
    /**
     * Create a handler of this type.
     */
    public LocalTransferHandler() {
        this.handlers = new ArrayList<Object> ();
    }
    
    /**
     * List of handlers and corresponding types.
     */
    private final List<Object> handlers;
    
    /**
     * Find a handler for a given class.
     */
    private DropHandler handlerForClass(JComponent comp, Class<?> type) {
        for (int i=handlers.size()-1; i >= 0; i--) {
            DropHandler handler = (DropHandler)handlers.get(i);
            if (handler.getDropClass(comp).isAssignableFrom(type))
                return handler;
        }
        return null;
    }
    
    /**
     * Find a handler for a given flavor. First try the flavor representation
     * class itself, and then, if this does not succeed and the class is an array class,
     * try the element class.
     */
    private DropHandler handlerForFlavor(JComponent comp, DataFlavor flavor) {
        Class clazz = flavor.getRepresentationClass();
        DropHandler handler = handlerForClass(comp, clazz);
        if (handler == null) {
            clazz = clazz.getComponentType();
            if (clazz != null)
                handler = handlerForClass(comp, clazz);
        }
        return handler;
    }
    
    
    /**
     * Register a drop handler. If an object is dropped that
     * can be handled by more than one handler, the last handler takes precedence.
     * If an array of objects is dropped, the class first searches for a handler
     * for the array type, and if not found, for a handler of the element type.
     * In the latter case, a drop will result in multiple
     * calls to {@code handler.acceptDrop}.
     */
    public void addDropHandler(DropHandler handler) {
        handlers.add(handler);
    }
    
    //
    private DragHandler dragHandler;
    
    /**
     * Install a drag handler.
     * If no drag handler is installed, dragging is effectively disabled.
     */
    public void setDragHandler(DragHandler dragHandler) {
        this.dragHandler = dragHandler;
    }
    
    /**
     * Return the drag handler used by this transfer handler, or {@code null}
     * when none is registered.
     */
    public DragHandler getDragHandler() {
        return dragHandler;
    }
    
    /**
     * Indicates whether a transfer from a given source to given destination
     * is allowed. This check is performed before the list of drop handlers is
     * consulted for a supported data type.<p>
     * Delegates to the drag handler, or returns true when no drag handler is
     * registered.
     * @param source The component that is the source of the data.
     * @param target The component that is the target of the data, or {@code null}
     * if the target transfer handler is not of this type.
     */
    protected boolean isTransferAllowed(JComponent source, JComponent target) {
        if (dragHandler == null)
            return true;
        else
            return dragHandler.isTransferAllowed(source, target);
    }
    
    
    /**
     * Overrides the standard functionality of {@link TransferHandler} by delegating
     * to the individual drop handlers registered with this object.<p>
     * <b>Note:</b> At this point it is not always possible to recognize a multiple
     * drag onto a drop target that only accepts single elements. This special case
     * is handled by a subsequent call to {@link #importData}.
     */
    public boolean canImport(JComponent comp, DataFlavor[] transferFlavors) {
        if (source != null && !isTransferAllowed(source, comp))
            return false;
        for (DataFlavor flavor : transferFlavors) {
            if (handlerForFlavor(comp, flavor) != null)
                return true;
        }
        return false;
    }
    
    //
    private static JComponent source;
    
    //
    private static JComponent target;
    
    /**
     * Overrides the standard functionality of {@link TransferHandler} by delegating
     * to the individual drop handlers registered with this object.
     */
    public boolean importData(JComponent comp, Transferable t) {
        for (DataFlavor flavor : t.getTransferDataFlavors()) {
            
            Class clazz = flavor.getRepresentationClass();
            DropHandler handler = handlerForClass(comp, clazz);
            try {
                if (handler == null) {
                    clazz = clazz.getComponentType();
                    if (clazz != null) {
                        handler = handlerForClass(comp, clazz);
                        if (handler != null) {
                            Object[] array = (Object[])t.getTransferData(flavor);
                            if (array.length == 0)
                                return false;
                            else if (array.length == 1 || handler.allowsMultipleDrops(comp)) {
                                for (int i=0; i < array.length; i++) {
                                    Object object = array[i];
                                    if (! handler.acceptDrop(comp, object, i))
                                        return false;
                                }
                                this.target = comp;
                                return true;
                            }
                        }
                    }
                } else {
                    boolean result = handler.acceptDrop(comp, t.getTransferData(flavor), 0);
                    if (result)
                        this.target = comp;
                    return result;
                }
            } catch (UnsupportedFlavorException e) {
                assert false : "Unexpected exception: " + e;
            } catch (IOException e) {
                assert false : "Unexpected I/O-exception: " + e;
            }
        }
        
        return false;
    }
    
    /**
     * Returns the type of transfer actions supported by the given source component.
     * Should return one {@link #COPY}, {@link #MOVE},
     * {@link #COPY_OR_MOVE}, {@link #LINK} or {@link #NONE}. This implementation
     * delegates to the drag handler if it exists or otherwise returns
     * {@code NONE}.
     */
    public int getSourceActions(JComponent source) {
        if (dragHandler == null)
            return NONE;
        else
            return dragHandler.getSourceActions(source);
    }
    
    // must be the first of both methods with the same name,
    // because of javadoc references
    /**
     * Invoked after data have been dragged-and-dropped from a component managed
     * by this handler. Typically, when the action
     * is {@code MOVE} and source and target are not the same (or are not views
     * of the same model),
     * the objects need to be removed from the source. When the
     * action is {@code COPY} or {@code LINK}, nothing needs to be done.<p>
     * This implementation delegates to the drag handler if one is registered
     * and otherwise does nothing.
     * @param objects Array of objects which have been exported
     * @param type Element type of this array
     * @param action the actual action that was performed, will never be NONE
     * @param source The component that is the source of the data.
     * @param target The component that was the target of the data, or {@code null}
     * if the source transfer handler is not of this type.
     */
    protected void exportDone(JComponent source, JComponent target, Object[] objects, Class<?> type, int action) {
        if (dragHandler != null)
            dragHandler.exportDone(source,target,objects,type,action);
    }
    
    /**
     * Delegates to {@link #exportDone}.
     */
    protected final void exportDone(JComponent source, Transferable data, int action) {
        if (action == NONE)
            return;   // # Grinvin bug #103, ctrl-X on a read-only source generates action==NONE
        LocalTransferable lt = (LocalTransferable)data;
        exportDone(source, target, lt.objects, lt.elementClass, action);
        this.source = null;
        this.target = null;
    }
    
    /**
     * Return the object(s) to be exported by a drag-and-drop
     * or cut-and-paste operation. If multiple
     * objects are exported at the same time, this should return an array.<p>
     * This implementation delegates to the drag handler if one is registered.
     * @param source component from which data should be exported
     */
    protected Object getExportedObjects(JComponent source) {
        if (dragHandler == null)
            return null;
        else
            return dragHandler.getExportedObjects(source);
    }
    
    /**
     * Return the class of objects being exported. If this
     * returns null, the class of the object
     * returned by {@link #getExportedObjects} is used, or the element type if an array is returned.<p>
     * This implementation delegates to drag handler if one is registered and returns
     * null otherwise.
     * This implementation delegates to the drag handler if one is registered.
     * @param source component from which data should be exported
     */
    protected Class getExportedClass(JComponent source) {
        if (dragHandler == null)
            return null;
        else
            return dragHandler.getExportedClass(source);
    }
    
    
    /**
     * Creates a transferable encapsulating the objects returned by
     * {@link #getExportedObjects} with a data flavor derived from the value
     * of {@link #getExportedClass}.
     */
    protected final Transferable createTransferable(JComponent comp) {
        this.target = null;
        this.source = comp;
        return new LocalTransferable(getExportedObjects(comp), getExportedClass(comp));
    }
    
    /**
     * Transferable for this component.
     */
    private static class LocalTransferable implements Transferable {
        
        //
        public final Object[] objects;
        
        //
        private final DataFlavor myFlavor;
        
        //
        private final DataFlavor[] myFlavors;
        
        //
        public final Class<?> elementClass;
        
        //
        public LocalTransferable(Object object, Class clazz) {
            if (object.getClass().isArray()) {
                this.objects = (Object[])object;
            } else {
                this.objects = new Object[] { object};
            }
            
            if (clazz == null)
                clazz = objects[0].getClass();
            this.elementClass = clazz;
            
            DataFlavor flavor = null;
            try {
                String flavorType = DataFlavor.javaJVMLocalObjectMimeType +
                        ";class=\"[L" + clazz.getName() + ";\"";
                flavor = new DataFlavor(flavorType);
            } catch (ClassNotFoundException ex) {
                throw new IllegalArgumentException("Cannot make class into flavor", ex);
            }
            
            this.myFlavor = flavor;
            this.myFlavors = new DataFlavor [] { flavor };
        }
        
        //
        public boolean isDataFlavorSupported(DataFlavor df) {
            return myFlavor.equals(df);
        }
        
        //
        public Object getTransferData(DataFlavor df) throws UnsupportedFlavorException, IOException {
            if (myFlavor.equals(df)) {
                return objects;
            } else {
                throw new UnsupportedFlavorException(myFlavor);
            }
        }
        
        public DataFlavor[] getTransferDataFlavors() {
            return myFlavors;
        }
        
    }
}
