/*
 * @(#)AnalysisModuleSet.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.datastore;

import java.util.Hashtable;
import java.util.Vector;

import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;

/**
 * Contains data associated with each analysis module.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:26 $
 * @since     December 15, 2002
 */
public class AnalysisModuleSet
{
    private Hashtable moduleNameToIndex = new Hashtable();
    private Vector moduleList = new Vector();

    
    /**
     * Default constructor
     */
    public AnalysisModuleSet()
    {
        // do nothing
    }
    
    
    /**
     * Create an instance using the given array of modules as the default
     * set.
     *
     * @param modules collection of modules to initialize the set with.
     * @exception IllegalArgumentException if <tt>modules</tt> is
     *      <tt>null</tt>.
     */
    public AnalysisModuleSet( IAnalysisModule[] modules )
    {
        if (modules == null)
        {
            throw new IllegalArgumentException( "No null args." );
        }
        
        addAnalysisModules( modules );
    }
    
    
    /**
     * Copy <tt>ams</tt>'s modules into this new set.
     *
     * @param ams the set to copy from
     * @exception IllegalArgumentException if <tt>ams</tt> is <tt>null</tt>.
     */
    public AnalysisModuleSet( AnalysisModuleSet ams )
    {
        if (ams == null)
        {
            throw new IllegalArgumentException( "No null args." );
        }
        
        joinAnalysisModuleSet( ams );
    }
    
    
    /**
     * Add an array of modules to this set.
     *
     * @param modules the array of modules to add to the set
     * @exception IllegalArgumentException if <tt>modules</tt> is
     *      <tt>null</tt>, or if any element in the array is <tt>null</tt>.
     */
    public void addAnalysisModules( IAnalysisModule[] modules )
    {
        if (modules == null)
        {
            throw new IllegalArgumentException( "No null args." );
        }
        
        for (int i = 0; i < modules.length; ++i)
        {
            addAnalysisModule( modules[i] );
        }
    }
    
    
    /**
     * Add a module to the set.
     *
     * @param module the module to add.
     * @exception IllegalArgumentException if <tt>module</tt> is <tt>null</tt>.
     * @exception IllegalStateException if <tt>module</tt> has a measure name
     *      that is the same as a previously added module.
     */
    public void addAnalysisModule( IAnalysisModule module )
    {
        if (module == null)
        {
            throw new IllegalArgumentException( "No null args." );
        }
        
        String name = module.getMeasureName();
        
        synchronized (this.moduleNameToIndex)
        {
            if (this.moduleNameToIndex.containsKey( name ))
            {
                throw new IllegalStateException( "Already have module with "+
                    "measure '"+name+"'." );
            }
            
            int nextIndex = this.moduleList.size();
            if (nextIndex >= Short.MAX_VALUE)
            {
                throw new IllegalStateException(
                    "Too many modules.  There is a maximum of "+
                    Short.MAX_VALUE+" modules allowed." );
            }
            this.moduleNameToIndex.put( name, new Short( (short)nextIndex ) );
            this.moduleList.addElement( module );
        }
    }
    
    
    /**
     * Join a module set to this set.  It will not add to this set any
     * repeated measure names that are in the given set.
     *
     * @param set the module set to add.
     * @exception IllegalArgumentException if <tt>set</tt> is <tt>null</tt>.
     */
    public void joinAnalysisModuleSet( AnalysisModuleSet set )
    {
        if (set == null)
        {
            throw new IllegalArgumentException( "No null args." );
        }
        
        java.util.Enumeration enum = set.moduleList.elements();
        while (enum.hasMoreElements())
        {
            IAnalysisModule module = (IAnalysisModule)enum.nextElement();
            String name = module.getMeasureName();
            if (!this.moduleNameToIndex.containsKey( name ))
            {
                addAnalysisModule( module );
            }
        }
    }
    
    
    /**
     * Return a complete list of modules in order.
     *
     * @return the ordered array of stored modules.
     */
    public IAnalysisModule[] getAnalysisModules()
    {
        synchronized( this.moduleNameToIndex )
        {
            int len = this.moduleList.size();
            IAnalysisModule[] am = new IAnalysisModule[ len ];
            this.moduleList.copyInto( am );
            return am;
        }
    }
    
    
    /**
     * Retrieves the index associated with the module with the given
     * measure name.
     *
     * @param measureName the measure name of the module.
     * @return the index of the module, or -1 if there is no such module
     *      with the given measure name.
     * @exception IllegalArgumentException if <tt>measureName</tt> is
     *      <tt>null</tt>.
     * @see #getAnalysisModuleIndex( IAnalysisModule )
     */
    public short getMeasureIndex( String measureName )
    {
        if (measureName == null)
        {
            throw new IllegalArgumentException( "No null args." );
        }
        Short i = (Short)this.moduleNameToIndex.get( measureName );
        if (i == null)
        {
            return -1;
        }
        return i.shortValue();
    }
    
    
    /**
     * Returns the module <tt>am</tt>'s index.
     *
     * @param am the module.
     * @return the index of <tt>am</tt>, or -1 if it is not stored in this
     *      set.
     * @exception IllegalArgumentException if <tt>am</tt> is <tt>null</tt>.
     * @see #getMeasureIndex( Stirng )
     */
    public short getAnalysisModuleIndex( IAnalysisModule am )
    {
        if (am == null)
        {
            throw new IllegalArgumentException( "No null args." );
        }
        return getMeasureIndex( am.getMeasureName() );
    }
    
    
    /**
     * Retrieve the number of modules stored in this set.
     *
     * @return the number of modules in the set.
     */
    public int getAnalysisModuleCount()
    {
        return this.moduleList.size();
    }
    
    
    /**
     * Returns the module stored at the given index <tt>index</tt>.
     *
     * @return the module at index <tt>index</tt>.
     * @exception IllegalArgumentException if <tt>index</tt> is less than 0
     *      or greater than or equal to the returned value of
     *      <tt>getAnalysisModuleCount()</tt>.
     */
    public IAnalysisModule getAnalysisModuleAt( short index )
    {
        int iindex = (int)index;
        if (iindex < 0 || iindex >= this.moduleList.size())
        {
            throw new IllegalArgumentException( "index out of range [0.."+
                (this.moduleList.size()-1)+"]" );
        }
        return (IAnalysisModule)this.moduleList.elementAt( iindex );
    }
}

