/*--------------------------------------------------------------------------+
$Id: MedianAggregator.java 29788 2010-08-19 08:46:02Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.math;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

/**
 * Median aggregator. Note that for an even number of elements this aggregator
 * returns the mean value of both middle elements.
 * 
 * @author deissenb
 * @author $Author: juergens $
 * @version $Rev: 29788 $
 * @levd.rating GREEN Hash: 263069760E186DA03CC4463A773263DD
 */
public class MedianAggregator implements IAggregator {

	/**
	 * Aggregates by finding the median.
	 * 
	 * @return {@link Double#NaN} for empty input collection
	 */
	@Override
	public double aggregate(Collection<? extends Number> values) {
		if (values.isEmpty()) {
			return Double.NaN;
		}

		ArrayList<Double> doubleValues = new ArrayList<Double>();

		for (Number value : values) {
			doubleValues.add(value.doubleValue());
		}

		Collections.sort(doubleValues);

		// odd number of elements
		if (doubleValues.size() % 2 == 1) {
			int medianIndex = (int) Math.ceil(doubleValues.size() / 2);
			return doubleValues.get(medianIndex);
		}

		// even number of elements
		int lowerMedianIndex = doubleValues.size() / 2 - 1;
		double lowerMedian = doubleValues.get(lowerMedianIndex);
		double upperMedian = doubleValues.get(lowerMedianIndex + 1);

		return (lowerMedian + upperMedian) / 2;
	}

	/** Returns {@link Double#NaN}. */
	@Override
	public double getNeutralElement() {
		return Double.NaN;
	}
}