/*
 * The Cryptonit security software suite is developped by IDEALX
 * Cryptonit Team (http://IDEALX.org/ and http://cryptonit.org).
 *
 * Copyright 2003-2006 IDEALX
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301, USA. 
 *
 * In addition, as two special exceptions:
 *
 * 1) IDEALX S.A.S gives permission to:
 *  * link the code of portions of his program with the OpenSSL library under
 *    certain conditions described in each source file
 *  * distribute linked combinations including the two, with respect to the
 *    OpenSSL license and with the GPL
 *
 * You must obey the GNU General Public License in all respects for all of the
 * code used other than OpenSSL. If you modify file(s) with this exception,
 * you may extend this exception to your version of the file(s), but you are
 * not obligated to do so. If you do not wish to do so, delete this exception
 * statement from your version, in all files (this very one along with all
 * source files).

 * 2) IDEALX S.A.S acknowledges that portions of his sourcecode uses (by the
 * way of headers inclusion) some work published by 'RSA Security Inc.'. Those
 * portions are "derived from the RSA Security Inc. PKCS #11Cryptographic
 * Token Interface (Cryptoki)" as described in each individual source file.
 */

#ifndef _PKCS12_HH_
#define _PKCS12_HH_

#include <vector>

#include "env.hh"
#include "Certificate.hh"
#include "pkcs8.hh"
#include "OpenSSL++.hh"


namespace Cryptonit{

    /** this class represents a PKCS#12
     */
    class pkcs12 {

    private:
	PKCS12* mP12;
	Certificate *cert;
	pkcs8 *pkey;
      
	std::vector<Certificate> *CA;
    
	/* the clean machine */
	void free();

	/** 
	 * fill key , X509 et CA fields after a PKCS12_parse
	 */
	void fillCertAndCA(EVP_PKEY *key , X509 *cer , STACK_OF(X509) *ca);
      
    public : 
	/** default constructor */
	pkcs12();
    
      
	/** copy constructor */
	pkcs12( const pkcs12& p );

   
	/** default destructor */
	~pkcs12();
      
	/** load a pkcs12 from a file
	 * @param filename a filename where to read the pkcs#12
	 * @param password password used to decrypt the pkcs#12
	 *
	 * @return an error code. 
	 */
	int load(const char* filename, const char *password);
      
      
	/** load a pkcs#12 from a bio
	 * @param bin a bio where to read the pkcs#12;
	 * @param password password used to decrypt the pkcs#12
	 *
	 * @return an error code. 
	 */
	int load(BIO *bin, const char* password);
      
      
	int load(const char* filename);
      

	/** get the X509 Certificate contained in the pkcs12
	 * @return a Certificate
	 */
	Certificate const &getCertificate() ;

      
	/** get the PRIVATE key contained in the pkcs12
	 * @return an pkcs8 
	 */
	pkcs8 &getKey() const;

      
	bool parse( const char* password ) ;
      
	bool parse( const std::string password );

	/** get a CA's certificates vector
	 * @return a vector of Certificate
	 */
	std::vector<Certificate> &getCA() const;
      
	/** Create a PKCS#12 object from a private key and a certificate.
	 *  
	 * @param privateKey : private key 
	 * @param certificate : certificate
	 * @param password : password used to encrypt the private key.
	 * @param name : not mandatory
	 * @param nid_key : use the defaut value
	 * @param nid_cert : use the defaut value
	 * @param iter : use the defaut value
	 * @param mac_iter : use the defaut value
	 * @param keytype : use the defaut value
	 */
	int create( pkcs8 privateKey, Certificate certificate, 
		    char* password, char* name = "",
		    int nid_key = 0, int nid_cert = 0, int iter = 0, int mac_iter = 0, 
		    int keytype = 0 );

	int save( const char* filename );
	int save( FILE* fp );
	int save( BIO* bp );

	/** true if this pkcs12 has been parsed
	 */
	bool parsed();

	std::string getHash();

	pkcs12& operator=( const pkcs12 &src );

    };
}


#endif
