/* GrinvinWindowWrapper.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.workspace;

import java.awt.Window;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;

import org.grinvin.io.IOFormatException;
import org.grinvin.io.LoaderSaverHelper;
import org.grinvin.main.GrinvinWindow;
import org.grinvin.io.SectionLoader;
import org.grinvin.io.SectionSaver;
import org.grinvin.main.GrinvinWindow.GrinvinWindowComponentType;

import org.jdom.DocType;
import org.jdom.Document;
import org.jdom.Element;

/**
 * Wrapper for the main GrInvIn window.
 */
public class GrinvinWindowWrapper extends DefaultWindowWrapper<GrinvinWindow> {
    
    //
    private static final String PUBLIC_ID = "-//GrInvIn IO//Grinvin 1.1//EN";
    
    public String getElementName() {
        return "grinvinWindow";
    }
    
    //
    private Map<String,Window> tempWindowMap;

    
    public void setTempWindowMap(Map<String, Window> tempWindowMap) {
        this.tempWindowMap = tempWindowMap;
    }
    
    private void registerAllOtherWindows (GrinvinWindow grinvinWindow) {
        for (Map.Entry<String, Window> entry : tempWindowMap.entrySet()) {
            grinvinWindow.registerWindow(entry.getKey(), entry.getValue());
        }
    }
    
    /**
     * Create all top level windows
     * and add them to the menus of the main window.
     */
    @Override
    public GrinvinWindow create() {
        GrinvinWindow window = new GrinvinWindow();
        registerAllOtherWindows(window);
        
        // create menus etc. for the main window
        // This cannot be done earlier because some of the menus refer to other windows
        window.setupContents();
        
        //
        window.setVisible(true);
        return window;
    }
    
    /**
     * Load this window and all dependent windows.
     */
    @Override
    public GrinvinWindow load(Element element, SectionLoader sloader) throws IOException {
        try {
            URI uri = new URI(element.getAttributeValue("uri"));
            String name = uri.getSchemeSpecificPart();
            GrinvinWindow window = load(sloader, name);
            window.setURI(uri);
            return window;
        } catch (URISyntaxException ex) {
            throw new IOFormatException("invalid uri");
        }
    }
    
    private GrinvinWindow load(SectionLoader sloader, String name) throws IOException {
        InputStream rootinput = sloader.openSection(name);
        
        Document document = LoaderSaverHelper.loadDocument(rootinput);
        if (!PUBLIC_ID.equals(document.getDocType().getPublicID()))
            throw new IOFormatException("Unkown document type: " + document.getDocType().getPublicID() + " expects " + PUBLIC_ID);
        Element element = document.getRootElement();

        // create the window
        GrinvinWindow window = new GrinvinWindow();
        registerAllOtherWindows(window);
        
        window.setupContents();
        
        String version = element.getAttributeValue("version");
        
        // make window visible
        loadWindowProperties(element, window);
        
        // select tab
        try{
            GrinvinWindowComponentType selectedTab = GrinvinWindowComponentType.valueOf(element.getAttributeValue("tab"));
            window.setSelectedTab(selectedTab);
        } catch (IllegalArgumentException ex){
            //do nothing: just select first tab
        }
        return window;
    }
    
    /**
     * Save this window and all dependent windows.
     */
    @Override
    public void save(GrinvinWindow window, Element parent, SectionSaver ssaver) throws IOException {
        String name = LoaderSaverHelper.prepareSessionURI(window, "grinvinwindows", "gwindow");
        save(window, ssaver, name);
        
        Element grinvinwindow = new Element(getElementName());
        grinvinwindow.setAttribute("uri", window.getURI().toString());
        parent.addContent(grinvinwindow);
    }
    
    private void save(GrinvinWindow window, SectionSaver ssaver, String name) throws IOException {
        Element element = baseElement();
        saveWindowProperties(element, window);
        // save tab
        element.setAttribute("tab", window.getSelectedTab().toString());        
        LoaderSaverHelper.outputXML(element,
                new DocType("grinvinWindow", PUBLIC_ID, "http://downloads.grinvin.org/dtds/io/grinvin-1.1.dtd"),
                ssaver.createSection(name));
    }
    
}
