/* WorksheetModel.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.worksheet;

import java.net.URI;
import javax.swing.event.EventListenerList;

import org.grinvin.conjecture.engine.APEngine;
import org.grinvin.conjecture.engine.EngineRunner;
import org.grinvin.conjecture.filter.CompoundGraphFilter;
import org.grinvin.conjecture.filter.GraphFilter;
import org.grinvin.conjecture.filter.GraphFilterListener;
import org.grinvin.conjecture.history.ConjectureEntry;
import org.grinvin.conjecture.history.ConjectureHistoryEntry;
import org.grinvin.conjecture.history.StateEntry;
import org.grinvin.expr.Expression;
import org.grinvin.list.ConjectureHistoryList;
import org.grinvin.list.ConjectureHistoryListModel;
import org.grinvin.list.ConjectureList;
import org.grinvin.list.ConjectureListModel;
import org.grinvin.list.DefaultConjectureHistoryListModel;
import org.grinvin.list.DefaultConjectureListModel;
import org.grinvin.list.DefaultFilterListModel;
import org.grinvin.list.FilterList;
import org.grinvin.list.FilterListModel;
import org.grinvin.list.HasName;
import org.grinvin.list.HasURI;
import org.grinvin.list.generators.DefaultGraphGeneratorInstanceListModel;
import org.grinvin.list.generators.GraphGeneratorInstanceList;
import org.grinvin.list.generators.GraphGeneratorInstanceListModel;
import org.grinvin.list.graphs.DefaultGraphListModel;
import org.grinvin.list.graphs.GraphList;
import org.grinvin.list.graphs.GraphListModel;
import org.grinvin.list.invariants.DefaultInvariantListModel;
import org.grinvin.list.invariants.InvariantList;
import org.grinvin.list.invariants.InvariantListModel;

/**
 * Model for the conjecturing window.
 */
public class WorksheetModel implements HasName, HasURI {
    
    //
    private final GraphListModel graphListModel = new DefaultGraphListModel();
    
    //
    private final InvariantListModel invariantListModel = new DefaultInvariantListModel();
    
    //
    private final GraphGeneratorInstanceListModel graphGeneratorInstancesListModel = new DefaultGraphGeneratorInstanceListModel();
    
    //
    private final FilterListModel filterListModel = new DefaultFilterListModel();
    
    //
    private final ConjectureListModel conjectureListModel = new DefaultConjectureListModel();
    
    //
    private final EngineRunner runner;
    
    //
    private final ConjectureHistoryListModel conjectureHistoryListModel = new DefaultConjectureHistoryListModel();
    
    //
    private String name;
    
    //
    private URI uri;
    
    /**
     * Creates a new instance of WorksheetModel
     */
    public WorksheetModel() {
        runner = new EngineRunner(new APEngine(), this); //TODO: make engine pluggable
        filter = new CompoundGraphFilter(filterListModel);
        filter.addGraphFilterListener(new GraphFilterListener() {
            public void filterChanged(GraphFilter filter) {
                fireFilterChanged();
            }
        });
    }
    
    //
    private final EventListenerList listenerList = new EventListenerList();
    
    //
    public void addWorksheetModelListener(WorksheetModelListener l){
        listenerList.add(WorksheetModelListener.class, l);
        invariantListModel.addInvariantListModelListener(l);
        graphListModel.addGraphListModelListener(l);
    }
    
    //
    public void removeWorksheetModelListener(WorksheetModelListener l){
        listenerList.remove(WorksheetModelListener.class, l);
        invariantListModel.removeInvariantListModelListener(l);
        graphListModel.removeGraphListModelListener(l);
    }
    
    //
    protected void fireEngineChanged(){
        Object[] listeners = listenerList.getListenerList();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i].equals(WorksheetModelListener.class))
                ((WorksheetModelListener)listeners[i+1]).engineChanged();
    }
    
    //
    protected void fireEngineConfigurationChanged(){
        Object[] listeners = listenerList.getListenerList();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i].equals(WorksheetModelListener.class))
                ((WorksheetModelListener)listeners[i+1]).engineConfigurationChanged();
    }
    
    //
    protected void fireFilterChanged(){
        Object[] listeners = listenerList.getListenerList();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i].equals(WorksheetModelListener.class))
                ((WorksheetModelListener)listeners[i+1]).filterChanged();
    }

    //
    protected void fireNameChanged(){
        Object[] listeners = listenerList.getListenerList();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i].equals(WorksheetModelListener.class))
                ((WorksheetModelListener)listeners[i+1]).worksheetModelNameChanged();
    }

    //
    public GraphList getGraphList() {
        return graphListModel;
    }
    
    //
    GraphListModel getGraphListModel() {
        return graphListModel;
    }
    
    //
    public GraphGeneratorInstanceList getGeneratorInstanceList() {
        return graphGeneratorInstancesListModel;
    }

    //
    public GraphGeneratorInstanceListModel getGeneratorInstanceListModel() {
        return graphGeneratorInstancesListModel;
    }
    
    //
    public InvariantList getInvariantList() {
        return invariantListModel;
    }
    
    //
    InvariantListModel getInvariantListModel() {
        return invariantListModel;
    }
    
    //
    public FilterList getFilterList() {
        return filterListModel;
    }
    
    //
    public FilterListModel getFilterListModel() {
        return filterListModel;
    }
    
    //
    public ConjectureList getConjectureList() {
        return conjectureListModel;
    }
    
    //
    public ConjectureListModel getConjectureListModel(){
        return conjectureListModel;
    }
    
    //
    public EngineRunner getEngineRunner() {
        return runner;
    }
    
    //
    public void addResult(Expression expression) {
        if (expression != null) {
            // TODO: log the null case
            conjectureListModel.add(expression);
            conjectureHistoryListModel.add(new ConjectureEntry(expression));
        }
    }
    
    //
    public void writeStateToHistory(){
        conjectureHistoryListModel.add(new StateEntry(graphListModel, invariantListModel, graphGeneratorInstancesListModel, filterListModel));
    }
    
    //
    public void addHistoryEntry(ConjectureHistoryEntry logEntry){
        conjectureHistoryListModel.add(logEntry);
    }
    
    //
    public ConjectureHistoryList getConjectureHistoryList(){
        return conjectureHistoryListModel;
    }

    //
    public ConjectureHistoryListModel getConjectureHistoryListModel(){
        return conjectureHistoryListModel;
    }

    //
    private CompoundGraphFilter filter;
    
    //
    public GraphFilter getFilter(){
        return filter;
    }
    
    //
    public String getName() {
        return name;
    }
    
    //
    public void setName(String name) {
        this.name = name;
        fireNameChanged();
    }

    //
    public boolean isNameEditable() {
        return true;
    }
    
    //
    public URI getURI() {
        return uri;
    }
    
    //
    public void setURI(URI uri) {
        this.uri = uri;
    }

}
