/* AbstractGraphFilter.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.conjecture.filter;

import java.text.MessageFormat;
import java.util.List;
import java.util.ResourceBundle;
import java.util.concurrent.CopyOnWriteArrayList;

import org.grinvin.graphs.GraphBundle;
import org.grinvin.graphs.GraphBundleView;

/**
 * Basic implementation of GraphFilter that offers the possibility for negation.
 */
public abstract class AbstractGraphFilter implements GraphFilter {
    
    private boolean negate;
    
    private static final ResourceBundle RBUNDLE = ResourceBundle.getBundle("org.grinvin.conjecture.filter.resources");
    
    /** Creates a new instance of AbstractGraphFilter */
    public AbstractGraphFilter() {
        this(false);
    }
    
    public AbstractGraphFilter(boolean negate) {
        this.negate = negate;
    }    

    public boolean accepts(GraphBundleView gb) {
        if(negate)
            return !accepts_impl(gb);
        else
            return accepts_impl(gb);
    }
    
    abstract public boolean accepts_impl(GraphBundleView gb);

    public String getDescription() {
        if(negate)
            return MessageFormat.format(RBUNDLE.getString("filter.name.negative"), getDescription_impl());
        else
            return getDescription_impl();
    }
    
    abstract public String getDescription_impl();

    public void negate(boolean aFlag) {
        if(negate != aFlag){
            negate = aFlag;
            fireFilterChanged();
        }
    }

    public boolean isNegated() {
        return negate;
    }
    
    //
    private List<GraphFilterListener> listeners = new CopyOnWriteArrayList<GraphFilterListener>();

    //
    public void addGraphFilterListener(GraphFilterListener listener) {
        listeners.add(listener);
    }
    
    //
    public void removeGraphFilterListener(GraphFilterListener listener) {
        listeners.remove(listener);
    }
    
    public void fireFilterChanged() {
        for(GraphFilterListener listener : listeners)
            listener.filterChanged(this);
    }
}
