/* VersionInfo.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.about;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Contains simple versioning info for a single module of GrInvIn.
 */
public class VersionInfo {
    
    //
    private final String name;
    
    //
    private final String version;
    
    //
    private final String revision;
    
    //
    private final String build;
    
    /**
     * Name of the module. E.g., <code>core</code>,
     * <code>factories</code>, ...
     */
    public String getName() {
        return name;
    }
    
    /**
     * Version number. E.g. <code>1.0.3</code>
     */
    public String getVersion() {
        return version;
    }
    
    
    /**
     * Revision number (from subversion). E.g. 1745M
     */
    public String getRevision() {
        return revision;
    }
    
    
    /**
     * Date string for the build.
     */
    public String getBuild() {
        return build;
    }
    
    /**
     * Return one of the four string properties.
     */
    String getColumn(int i) {
        switch (i) { // NOPMD
            case 0: return getName();
            case 1: return getVersion();
            case 2: return getRevision();
            case 3: return getBuild();
            default:
                throw new IndexOutOfBoundsException ("Called with wrong index: " + i);
        }
    }
    
    private VersionInfo(String name, String version, String revision, String build) {
        this.name = name;
        this.version = version;
        this.revision = revision;
        this.build = build;
    }
    
    /**
     * Load version info from the corresponding properties file. Returns null
     * when the file is not found and hence presumably the module is not in
     * the class path.
     * @param name Name of the module for which the information must be loaded.
     */
    public static VersionInfo loadVersionInfo(String name) {
        try {
            String pathName = "/versions-grinvin-" + name + "/version.properties";
            InputStream inputStream = VersionInfo.class.getResourceAsStream(pathName);
            if (inputStream == null)
                return null;
            Properties properties = new Properties();
            properties.load(inputStream);
            inputStream.close();
            return new VersionInfo(name, properties.getProperty("version"),
                    properties.getProperty("revision"), properties.getProperty("builddate"));
        } catch (IOException ex) {
            Logger.getLogger("org.grinvin.about").log(Level.WARNING,
                    "Could not load version information for module '" + name + "'", ex);
            return null;
        }
    }
    
}
